
/*! \file
    Provides file logging capability at the hwInterface level.  Logs, register reads, writes, etc. */

#ifndef HW_INTERFACE_LOGGING_TOKEN
#define HW_INTERFACE_LOGGING_TOKEN


#include "hwInterface.h"
#include "hwInterfaceExtended.h"
#include <stdio.h>

#ifndef DLLEXPORT
#ifdef WIN32
#define DLLEXPORT __declspec(dllexport)
#else
#define DLLEXPORT
#endif
#endif

#define MAX_LOG_LINE_SIZE 512

#ifdef __cplusplus
  extern "C" {
#endif

#ifdef REGLIBLOGGING

#include "regLib.h"

/*! Register a mapping from (adapterNumber, PHY_ID) pair to PartNumber.  Return
 * 1 on success, 0 on failure. */
DLLEXPORT uint32_t HW_Logging_SetPartNumberMapping
(    
    /*! The number of the adapter in the HW_FilteredAdapterList. */
  uint32_t adapterNumber,
    /*! The address of the PHY being addressed. */
  uint32_t PHY_ID,
    /*! The part number to use when looking up RegLib entries. */
  PartNumber pn
);

/*! Get the part number that is mapped to this (adapterNumber, PHY_ID) pair. 
 * If none, the INVALID will be returned. */
PartNumber HW_Logging_GetPartNumber
(    
    /*! The number of the adapter in the HW_FilteredAdapterList. */
  uint32_t adapterNumber,
    /*! The address of the PHY being addressed. */
  uint32_t PHY_ID
);

#endif
/* #endif REGLIBLOGGING */

enum
{
    HWLOG_CRITICAL = 0,
    HWLOG_ERROR,
    HWLOG_WARNING,
    HWLOG_INFO,
    HWLOG_DEBUG,
    MAX_LOG_LEVEL_COUNT
};

enum
{
	HWLOG_INTERFACE = 0,
	HWLOG_DEVICE_CONTROL,
	HWLOG_GENLOG,
	HWLOG_DRIVER_INSTALLER,
	MAX_FLAG_COUNT
};
void HW_Log( uint32_t level, uint32_t flags, const char *format, ... );

/*! Initialize the logging stuff.  Will reset the part number mappings. */
void HW_Logging_Initialize(void);


/*! Enable logging of all operations (MDIO read, MDIO write etc.) to standard out. */
DLLEXPORT void HW_EnableLoggingStdout (void);


/*! Enable logging of all operations (MDIO read, MDIO write etc.) to standard error. */
DLLEXPORT void HW_EnableLoggingStderr (void);


/*! Enable logging of all operations (MDIO read, MDIO write etc.) to Python logger. */\
#ifndef NOPYTHON
DLLEXPORT void HW_EnablePyLogging( PyObject *logger, PyObject *levels);
#endif

/*! Enable logging of all operations (MDIO read, MDIO write etc.) to f. */
DLLEXPORT void HW_EnableLogging( FILE* f );


/*! Disable logging. If close != 0, logging file descriptor will be closed.*/
DLLEXPORT void HW_DisableLogging (int close);


/*! If logging is enable, print msg (and a newline) to the log. */
DLLEXPORT void HW_LogMsg (const char* msg);


/*! If logging is enabled, write a log message describing an MDIO write operation. */
void HW_Logging_MDIOWrite
(
    /*! The number of the adapter in the HW_FilteredAdapterList. */
  uint32_t adapterNumber,
    /*! The address of the PHY being addressed. */
  uint32_t PHY_ID,
    /*! The address of the MMD within the Aquantia PHY being addressed.*/
  uint32_t MMD,
    /*! The 16-bit address being written to.*/
  uint32_t address,
    /*! The 16-bits of data being sent out with the write command. This may be either address or data.*/
  uint32_t data
#ifdef EXTRA_READ_BEFORE_WRITE
  ,
    /*! The value of the register prior to the write. */
  uint32_t previousData
#endif
);


/*! If logging is enabled, write a log message describing an MDIO read operation. */
void HW_Logging_MDIORead
(
    /*! The number of the adapter in the HW_FilteredAdapterList. */
  uint32_t adapterNumber,
    /*! The address of the PHY being addressed. */
  uint32_t PHY_ID,
    /*! The address of the MMD within the Aquantia PHY being addressed.*/
  uint32_t MMD,
    /*! The 16-bit address being read from to.*/
  uint32_t address,
    /*! The 16-bits of data read with the read operation.*/
  uint32_t data
);


/*! If logging is enabled, write a log message describing an asynchronous MDIO read request operation. */
void HW_Logging_BlockMDIORead
(
	/*! The number of the adapter in the HW_FilteredAdapterList. */
  uint32_t adapterNumber,
    /*! The address of the PHY being addressed. */
  uint32_t PHY_ID,
    /*! The address of the MMD within the Aquantia PHY being addressed.*/
  uint32_t MMD,
    /*! The 16-bit address being read from.*/
  uint32_t address
);


/*! If logging is enabled, write a log message describing an unflushed MDIO write operation. */
void HW_Logging_BlockMDIOWrite
(
/*! The number of the adapter in the HW_FilteredAdapterList. */
  uint32_t adapterNumber,
    /*! The address of the PHY being addressed. */
  uint32_t PHY_ID,
    /*! The address of the MMD within the Aquantia PHY being addressed.*/
  uint32_t MMD,
    /*! The 16-bit address being written to.*/
  uint32_t address,
    /*! The 16-bit data to write.*/
  uint32_t data
);

#if 0
/*! If logging is enabled, write a log message describing an asynchronous MDIO read fetch operation. */
void HW_Logging_BlockMDIOFetch
(
    /*! The number of the adapter in the HW_FilteredAdapterList. */
  uint32_t adapterNumber,
    /*! The address of the PHY being addressed. */
  uint32_t PHY_ID,
    /*! The address of the MMD within the Aquantia PHY being addressed.*/
  uint32_t MMD,
    /*! The 16-bit address being written to.*/
  uint32_t address,
    /*! The 16-bits of data read with the fetch operation.*/
  uint32_t data
#ifdef EXTRA_READ_BEFORE_WRITE
  ,
    /*! The value of the register prior to the write. */
  uint32_t previousData
#endif
);
#endif


/*! If logging is enabled, write a log message describing an asynchronous MDIO read fetch operation. */
void HW_Logging_BlockMDIOFetch
(
    /*! The number of the adapter in the HW_FilteredAdapterList. */
  uint32_t adapterNumber,
    /*! The address of the PHY being addressed. */
  uint32_t PHY_ID,
    /*! If we're logging the execution of a write operation, set to 1.  Otherwise, 0. */
  uint32_t isWrite,
    /*! The 16-bit address being written to.*/
  uint32_t MMD,
    /*! The 16-bit address being written to.*/
  uint32_t address,
    /*! The 16-bits of data read with the fetch operation.*/
  uint32_t data
#ifdef EXTRA_READ_BEFORE_WRITE
  ,
    /*! The value of the register prior to the write. */
  uint32_t previousData
#endif
);

/*! If logging is enabled, write a log message describing an write operation. */
void HW_Logging_Write32
(
	/*! The number of the adapter in the HW_FilteredAdapterList. */
	uint32_t adapterNumber,
	/*! The address of the PHY being addressed. */
	uint32_t DEV_ID,
	/*! The 16-bit address being written to.*/
	uint32_t address,
	/*! The 32-bits of data being sent out with the write command. This may be either address or data.*/
	uint32_t data
#ifdef EXTRA_READ_BEFORE_WRITE
,
	/*! The value of the register prior to the write. */
	uint32_t previousData
#endif
);


/*! If logging is enabled, write a log message describing an read operation. */
void HW_Logging_Read32
(
	/*! The number of the adapter in the HW_FilteredAdapterList. */
	uint32_t adapterNumber,
	/*! The address of the PHY being addressed. */
	uint32_t DEV_ID,
	/*! The 16-bit address being read from to.*/
	uint32_t address,
	/*! The 32-bits of data read with the read operation.*/
	uint32_t data
);


/*! If logging is enabled, write a log message describing an asynchronous read request operation. */
void HW_Logging_BlockRead32
(
	/*! The number of the adapter in the HW_FilteredAdapterList. */
	uint32_t adapterNumber,
	/*! The address of the PHY being addressed. */
	uint32_t DEV_ID,
	/*! The 16-bit address being read from.*/
	uint32_t address
);


/*! If logging is enabled, write a log message describing an unflushed write operation. */
void HW_Logging_BlockWrite32
(
	/*! The number of the adapter in the HW_FilteredAdapterList. */
	uint32_t adapterNumber,
	/*! The address of the PHY being addressed. */
	uint32_t DEV_ID,
	/*! The 16-bit address being written to.*/
	uint32_t address,
	/*! The 32-bit data to write.*/
	uint32_t data
);

/*! If logging is enabled, write a log message describing an asynchronous read fetch operation. */
void HW_Logging_BlockFetch32
(
	/*! The number of the adapter in the HW_FilteredAdapterList. */
	uint32_t adapterNumber,
	/*! The address of the PHY being addressed. */
	uint32_t DEV_ID,
	/*! If we're logging the execution of a write operation, set to 1.  Otherwise, 0. */
	uint32_t isWrite,
	/*! The 16-bit address being written to.*/
	uint32_t address,
	/*! The 32-bits of data read with the fetch operation.*/
	uint32_t data
#ifdef EXTRA_READ_BEFORE_WRITE
	,
	/*! The value of the register prior to the write. */
	uint32_t previousData
#endif
);

/*! If logging is enabled, write a log message describing an write operation. */
void HW_Logging_Write64
(
	/*! The number of the adapter in the HW_FilteredAdapterList. */
	uint32_t adapterNumber,
	/*! The address of the PHY being addressed. */
	uint32_t DEV_ID,
	/*! The 16-bit address being written to.*/
	uint32_t address,
	/*! The 64-bits of data being sent out with the write command. This may be either address or data.*/
	uint64_t data
#ifdef EXTRA_READ_BEFORE_WRITE
,
	/*! The value of the register prior to the write. */
	uint64_t previousData
#endif
);


/*! If logging is enabled, write a log message describing an read operation. */
void HW_Logging_Read64
(
	/*! The number of the adapter in the HW_FilteredAdapterList. */
	uint32_t adapterNumber,
	/*! The address of the PHY being addressed. */
	uint32_t DEV_ID,
	/*! The 16-bit address being read from to.*/
	uint32_t address,
	/*! The 64-bits of data read with the read operation.*/
	uint64_t data
);


/*! If logging is enabled, write a log message describing an asynchronous read request operation. */
void HW_Logging_BlockRead64
(
	/*! The number of the adapter in the HW_FilteredAdapterList. */
	uint32_t adapterNumber,
	/*! The address of the PHY being addressed. */
	uint32_t DEV_ID,
	/*! The 16-bit address being read from.*/
	uint32_t address
);


/*! If logging is enabled, write a log message describing an unflushed write operation. */
void HW_Logging_BlockWrite64
(
	/*! The number of the adapter in the HW_FilteredAdapterList. */
	uint32_t adapterNumber,
	/*! The address of the PHY being addressed. */
	uint32_t DEV_ID,
	/*! The 16-bit address being written to.*/
	uint32_t address,
	/*! The 64-bit data to write.*/
	uint64_t data
);

/*! If logging is enabled, write a log message describing an asynchronous read fetch operation. */
void HW_Logging_BlockFetch64
(
	/*! The number of the adapter in the HW_FilteredAdapterList. */
	uint32_t adapterNumber,
	/*! The address of the PHY being addressed. */
	uint32_t DEV_ID,
	/*! If we're logging the execution of a write operation, set to 1.  Otherwise, 0. */
	uint32_t isWrite,
	/*! The 16-bit address being written to.*/
	uint32_t address,
	/*! The 64-bits of data read with the fetch operation.*/
	uint64_t data
#ifdef EXTRA_READ_BEFORE_WRITE
	,
	/*! The value of the register prior to the write. */
	uint64_t previousData
#endif
);
#ifdef __cplusplus
  }
#endif

#endif

