// SPDX-License-Identifier: GPL-2.0-only
/*
 * aQuantia Corporation Network Driver
 * Copyright (C) 2019-2020 aQuantia Corporation. All rights reserved
 */

/* File hw_atl2_llh.c: Definitions of bitfield and register access functions for
 * Atlantic registers.
 */

#include "hw_atl2_llh.h"
#include "hw_atl2_llh_internal.h"
#include "../aq_hw_utils.h"


void hw_atl2_phi_ext_tag_set(struct aq_hw_s *aq_hw, u32 val)
{
	aq_hw_write_reg_bit(aq_hw, HW_ATL2_PHI_EXTENDED_TAG_EN_ADR,
			    HW_ATL2_PHI_EXTENDED_TAG_EN_MSK,
			    HW_ATL2_PHI_EXTENDED_TAG_EN_SHIFT, val);
}

u32 hw_atl2_phi_ext_tag_get(struct aq_hw_s *aq_hw)
{
	return aq_hw_read_reg_bit(aq_hw, HW_ATL2_PHI_EXTENDED_TAG_EN_ADR,
				  HW_ATL2_PHI_EXTENDED_TAG_EN_MSK,
				  HW_ATL2_PHI_EXTENDED_TAG_EN_SHIFT);
}

void hw_atl2_rpf_rss_hash_type_set(struct aq_hw_s *aq_hw, u32 rssHashType)
{
	aq_hw_write_reg_bit(aq_hw, HW_ATL2_RPF_PIF_RPF_RSS_HASH_TYPEI_ADR,
			    HW_ATL2_RPF_PIF_RPF_RSS_HASH_TYPEI_MSK,
			    HW_ATL2_RPF_PIF_RPF_RSS_HASH_TYPEI_SHIFT,
			    rssHashType);
}

u32 hw_atl2_rpf_rss_hash_type_get(struct aq_hw_s *aq_hw)
{
	return aq_hw_read_reg_bit(aq_hw,
				  HW_ATL2_RPF_PIF_RPF_RSS_HASH_TYPEI_ADR,
				  HW_ATL2_RPF_PIF_RPF_RSS_HASH_TYPEI_MSK,
				  HW_ATL2_RPF_PIF_RPF_RSS_HASH_TYPEI_SHIFT);
}


/* rpf */

void hw_atl2_rpf_new_enable_set(struct aq_hw_s *aq_hw, u32 enable)
{
	aq_hw_write_reg_bit(aq_hw, HW_ATL2_RPF_NEW_EN_ADR,
			    HW_ATL2_RPF_NEW_EN_MSK,
			    HW_ATL2_RPF_NEW_EN_SHIFT,
			    enable);
}

void hw_atl2_rpfl2_uc_flr_tag_set(struct aq_hw_s *aq_hw, u32 tag, u32 filter)
{
	aq_hw_write_reg_bit(aq_hw, HW_ATL2_RPFL2UC_TAG_ADR(filter),
			    HW_ATL2_RPFL2UC_TAG_MSK,
			    HW_ATL2_RPFL2UC_TAG_SHIFT,
			    tag);
}

void hw_atl2_rpfl2_bc_flr_tag_set(struct aq_hw_s *aq_hw, u32 tag)
{
	aq_hw_write_reg_bit(aq_hw, HW_ATL2_RPF_L2_BC_TAG_ADR,
			    HW_ATL2_RPF_L2_BC_TAG_MSK,
			    HW_ATL2_RPF_L2_BC_TAG_SHIFT,
			    tag);
}

u32 hw_atl2_new_rpf_rss_redir_get(struct aq_hw_s *aq_hw, u32 tc, u32 index)
{
	return aq_hw_read_reg_bit(aq_hw, HW_ATL2_RPF_RSS_REDIR_ADR(tc, index),
				  HW_ATL2_RPF_RSS_REDIR_MSK(tc),
				  HW_ATL2_RPF_RSS_REDIR_SHIFT(tc));
}

void hw_atl2_new_rpf_rss_redir_set(struct aq_hw_s *aq_hw, u32 tc, u32 index,
				     u32 queue)
{
	aq_hw_write_reg_bit(aq_hw, HW_ATL2_RPF_RSS_REDIR_ADR(tc, index),
			    HW_ATL2_RPF_RSS_REDIR_MSK(tc),
			    HW_ATL2_RPF_RSS_REDIR_SHIFT(tc),
			    queue);
}

void hw_atl2_rpf_vlan_flr_tag_set(struct aq_hw_s *aq_hw, u32 tag, u32 filter)
{
	aq_hw_write_reg_bit(aq_hw, HW_ATL2_RPF_VL_TAG_ADR(filter),
			    HW_ATL2_RPF_VL_TAG_MSK,
			    HW_ATL2_RPF_VL_TAG_SHIFT,
			    tag);
};


void hw_atl2_rpf_etht_flr_tag_set(struct aq_hw_s *aq_hw, u32 tag, u32 filter)
{
	aq_hw_write_reg_bit(aq_hw, HW_ATL2_RPF_ET_TAG_ADR(filter),
			    HW_ATL2_RPF_ET_TAG_MSK,
			    HW_ATL2_RPF_ET_TAG_SHIFT, tag);
}

void hw_atl2_rpf_l3_l4_enf_set(struct aq_hw_s *aq_hw, u32 val, u32 filter)
{
	aq_hw_write_reg_bit(aq_hw, HW_ATL2_RPF_L3_L4_ENF_ADR(filter),
			HW_ATL2_RPF_L3_L4_ENF_MSK,
			HW_ATL2_RPF_L3_L4_ENF_SHIFT, val);
}

void hw_atl2_rpf_l3_v6_enf_set(struct aq_hw_s *aq_hw, u32 val, u32 filter)
{
	aq_hw_write_reg_bit(aq_hw, HW_ATL2_RPF_L3_V6_ENF_ADR(filter),
			HW_ATL2_RPF_L3_V6_ENF_MSK,
			HW_ATL2_RPF_L3_V6_ENF_SHIFT, val);
}

void hw_atl2_rpf_l3_saf_en_set(struct aq_hw_s *aq_hw, u32 val, u32 filter)
{
	aq_hw_write_reg_bit(aq_hw, HW_ATL2_RPF_L3_SAF_EN_ADR(filter),
			HW_ATL2_RPF_L3_SAF_EN_MSK,
			HW_ATL2_RPF_L3_SAF_EN_SHIFT, val);
}

void hw_atl2_rpf_l3_daf_en_set(struct aq_hw_s *aq_hw, u32 val, u32 filter)
{
	aq_hw_write_reg_bit(aq_hw, HW_ATL2_RPF_L3_DAF_EN_ADR(filter),
			HW_ATL2_RPF_L3_DAF_EN_MSK,
			HW_ATL2_RPF_L3_DAF_EN_SHIFT, val);
}

void hw_atl2_rpf_l4_spf_en_set(struct aq_hw_s *aq_hw, u32 val, u32 filter)
{
	aq_hw_write_reg_bit(aq_hw, HW_ATL2_RPF_L4_SPF_EN_ADR(filter),
			HW_ATL2_RPF_L4_SPF_EN_MSK,
			HW_ATL2_RPF_L4_SPF_EN_SHIFT, val);
}

void hw_atl2_rpf_l4_dpf_en_set(struct aq_hw_s *aq_hw, u32 val, u32 filter)
{
	aq_hw_write_reg_bit(aq_hw, HW_ATL2_RPF_L4_DPF_EN_ADR(filter),
			HW_ATL2_RPF_L4_DPF_EN_MSK,
			HW_ATL2_RPF_L4_DPF_EN_SHIFT, val);
}

void hw_atl2_rpf_l4_protf_en_set(struct aq_hw_s *aq_hw, u32 val, u32 filter)
{
	aq_hw_write_reg_bit(aq_hw, HW_ATL2_RPF_L4_PROTF_EN_ADR(filter),
			HW_ATL2_RPF_L4_PROTF_EN_MSK,
			HW_ATL2_RPF_L4_PROTF_EN_SHIFT, val);
}

void hw_atl2_rpf_l3_arpf_en_set(struct aq_hw_s *aq_hw, u32 val, u32 filter)
{
	aq_hw_write_reg_bit(aq_hw, HW_ATL2_RPF_L3_ARPF_EN_ADR(filter),
			HW_ATL2_RPF_L3_ARPF_EN_MSK,
			HW_ATL2_RPF_L3_ARPF_EN_SHIFT, val);
}

void hw_atl2_rpf_l3_l4_rxqf_en_set(struct aq_hw_s *aq_hw, u32 val, u32 filter)
{
	aq_hw_write_reg_bit(aq_hw, HW_ATL2_RPF_L3_L4_RXQF_EN_ADR(filter),
			HW_ATL2_RPF_L3_L4_RXQF_EN_MSK,
			HW_ATL2_RPF_L3_L4_RXQF_EN_SHIFT, val);
}

void hw_atl2_rpf_l3_l4_mng_rxqf_set(struct aq_hw_s *aq_hw, u32 val, u32 filter)
{
	aq_hw_write_reg_bit(aq_hw, HW_ATL2_RPF_L3_L4_MNG_RXQF_ADR(filter),
			HW_ATL2_RPF_L3_L4_MNG_RXQF_MSK,
			HW_ATL2_RPF_L3_L4_MNG_RXQF_SHIFT, val);
}

void hw_atl2_rpf_l3_l4_actf_set(struct aq_hw_s *aq_hw, u32 val, u32 filter)
{
	aq_hw_write_reg_bit(aq_hw, HW_ATL2_RPF_L3_L4_ACTF_ADR(filter),
			HW_ATL2_RPF_L3_L4_ACTF_MSK,
			HW_ATL2_RPF_L3_L4_ACTF_SHIFT, val);
}

void hw_atl2_rpf_l3_l4_rxqf_set(struct aq_hw_s *aq_hw, u32 val, u32 filter)
{
	aq_hw_write_reg_bit(aq_hw, HW_ATL2_RPF_L3_L4_RXQF_ADR(filter),
			HW_ATL2_RPF_L3_L4_RXQF_MSK,
			HW_ATL2_RPF_L3_L4_RXQF_SHIFT, val);
}

void hw_atl2_rpf_l3_v4_dest_addr_set(struct aq_hw_s *aq_hw, u32 filter, u32 val)
{
	u32 dword = filter % 4;
	u32 addr_set = 6 + ((filter < 4)?(0):(1));
	aq_hw_write_reg(aq_hw,
			HW_ATL2_RPF_L3_DA_DW_ADR(addr_set, dword),
			val);

}

void hw_atl2_rpf_l3_v4_src_addr_set(struct aq_hw_s *aq_hw, u32 filter, u32 val)
{
	u32 dword = filter % 4;
	u32 addr_set = 6 + ((filter < 4)?(0):(1));
	aq_hw_write_reg(aq_hw,
			HW_ATL2_RPF_L3_SA_DW_ADR(addr_set, dword),
			val);
}

void hw_atl2_rpf_l3_v6_dest_addr_set(struct aq_hw_s *aq_hw, u8 location,
				      u32 *ipv6_dst)
{
	int i;

	for (i = 0; i < 4; ++i)
		aq_hw_write_reg(aq_hw,
				HW_ATL2_RPF_L3_DA_DW_ADR(location, 3 - i),
				ipv6_dst[i]);
}

void hw_atl2_rpf_l3_v6_src_addr_set(struct aq_hw_s *aq_hw, u8 location,
				      u32 *ipv6_src)
{
	int i;

	for (i = 0; i < 4; ++i)
		aq_hw_write_reg(aq_hw,
				HW_ATL2_RPF_L3_SA_DW_ADR(location, 3 - i),
				ipv6_src[i]);
}

void hw_atl2_rpf_l3_cmd_set(struct aq_hw_s *aq_hw, u32 val, u32 filter)
{
	aq_hw_write_reg_bit(aq_hw, HW_ATL2_RPF_L3_CMD_ADR(filter),
			HW_ATL2_RPF_L3_CMD_MSK,
			HW_ATL2_RPF_L3_CMD_SHIFT, val);
}

void hw_atl2_rpf_l3_v6_v4_select_set(struct aq_hw_s *aq_hw, u32 val)
{
	aq_hw_write_reg_bit(aq_hw, HW_ATL2_RPF_L3_V6_V4_SELECT_ADR,
			HW_ATL2_RPF_L3_V6_V4_SELECT_MSK,
			HW_ATL2_RPF_L3_V6_V4_SELECT_SHIFT, val);
}

void hw_atl2_rpf_l3_v4_tag_set(struct aq_hw_s *aq_hw, u32 val, u32 filter)
{
	aq_hw_write_reg_bit(aq_hw, HW_ATL2_RPF_L3_V4_TAG_ADR(filter),
			HW_ATL2_RPF_L3_V4_TAG_MSK,
			HW_ATL2_RPF_L3_V4_TAG_SHIFT, val);
}

void hw_atl2_rpf_l3_v6_tag_set(struct aq_hw_s *aq_hw, u32 val, u32 filter)
{
	aq_hw_write_reg_bit(aq_hw, HW_ATL2_RPF_L3_V6_TAG_ADR(filter),
			HW_ATL2_RPF_L3_V6_TAG_MSK,
			HW_ATL2_RPF_L3_V6_TAG_SHIFT, val);
}

void hw_atl2_rpf_l4_tag_set(struct aq_hw_s *aq_hw, u32 val, u32 filter)
{
	aq_hw_write_reg_bit(aq_hw, HW_ATL2_RPF_L4_TAG_ADR(filter),
			HW_ATL2_RPF_L4_TAG_MSK,
			HW_ATL2_RPF_L4_TAG_SHIFT, val);
}
void hw_atl2_rpf_l4_cmd_set(struct aq_hw_s *aq_hw, u32 val, u32 filter)
{
	aq_hw_write_reg_bit(aq_hw, HW_ATL2_RPF_L4_CMD_ADR(filter),
			HW_ATL2_RPF_L4_CMD_MSK,
			HW_ATL2_RPF_L4_CMD_SHIFT, val);
}

void hw_atl2_rpf_l4_protf_set(struct aq_hw_s *aq_hw, u32 val, u32 filter)
{
	aq_hw_write_reg_bit(aq_hw, HW_ATL2_RPF_L4_PROTF_ADR(filter),
			HW_ATL2_RPF_L4_PROTF_MSK,
			HW_ATL2_RPF_L4_PROTF_SHIFT, val);
}

/* tsg */
void hw_atl2_tsg_clock_en(struct aq_hw_s *aq_hw, u32 clock_sel, u32 clock_enable)
{
	aq_hw_write_reg_bit(aq_hw, HW_ATL2_TIMESTAMPGENERATOR1_ADR,
		clock_sel == ATL_PTP_CLOCK_SEL_PTM ?
			HW_ATL2_PTMCLOCKENABLE_MSK :
			HW_ATL2_PTPCLOCKENABLE_MSK,
		clock_sel == ATL_PTP_CLOCK_SEL_PTM ?
			HW_ATL2_PTMCLOCKENABLE_SHIFT :
			HW_ATL2_PTPCLOCKENABLE_SHIFT,
		clock_enable);
}

inline static void _hw_atl2_tsg_clock_reset(struct aq_hw_s *aq_hw,
					    u32 clock_sel, u32 val)
{
	aq_hw_write_reg_bit(aq_hw, HW_ATL2_TIMESTAMPGENERATOR1_ADR,
		clock_sel == ATL_PTP_CLOCK_SEL_PTM ?
			HW_ATL2_PTMRESET_MSK : HW_ATL2_PTPRESET_MSK,
		clock_sel == ATL_PTP_CLOCK_SEL_PTM ?
			HW_ATL2_PTMRESET_SHIFT : HW_ATL2_PTPRESET_SHIFT,
		val);
}

void hw_atl2_tsg_clock_reset(struct aq_hw_s *aq_hw, u32 clock_sel)
{
	_hw_atl2_tsg_clock_reset(aq_hw, clock_sel, 1);
	_hw_atl2_tsg_clock_reset(aq_hw, clock_sel, 0);
}

inline static void _hw_atl2_tsg_clock_read(struct aq_hw_s *aq_hw, u32 clock_sel, u32 val)
{
	aq_hw_write_reg_bit(aq_hw, HW_ATL2_TIMESTAMPGENERATOR2_ADR,
		clock_sel == ATL_PTP_CLOCK_SEL_PTM ?
			HW_ATL2_PTMDIGITALCLOCKREAD_MSK :
			HW_ATL2_PTPDIGITALCLOCKREAD_MSK,
		clock_sel == ATL_PTP_CLOCK_SEL_PTM ?
			HW_ATL2_PTMDIGITALCLOCKREAD_SHIFT :
			HW_ATL2_PTPDIGITALCLOCKREAD_SHIFT,
		val);
}

u64 hw_atl2_tsg_clock_read(struct aq_hw_s *aq_hw, u32 clock_sel)
{
	_hw_atl2_tsg_clock_read(aq_hw, clock_sel, 1);
	_hw_atl2_tsg_clock_read(aq_hw, clock_sel, 0);

	return ((u64)aq_hw_read_reg(aq_hw,
			clock_sel == ATL_PTP_CLOCK_SEL_PTM ?
				HW_ATL2_PTMCLOCKNANOSECONDCOUNTBIT_LSW_ADR :
				HW_ATL2_PTPCLOCKNANOSECONDCOUNTBIT_LSW_ADR )
		<< 0 ) |
		((u64)aq_hw_read_reg(aq_hw,
			clock_sel == ATL_PTP_CLOCK_SEL_PTM ?
				HW_ATL2_PTMCLOCKNANOSECONDCOUNTBIT_MSW_ADR :
				HW_ATL2_PTPCLOCKNANOSECONDCOUNTBIT_MSW_ADR )
		<< 32 );
}

static inline void _hw_atl2_tsg_clock_modify_value(struct aq_hw_s *aq_hw,
						   u32 clock_sel,
						   u64 ns, u32 fns)
{
	aq_hw_write_reg(aq_hw, clock_sel == ATL_PTP_CLOCK_SEL_PTM ?
		HW_ATL2_MODIFYPTMNANOSECONDCOUNTERVALUE1_ADR :
		HW_ATL2_MODIFYPTPNANOSECONDCOUNTERVALUE1_ADR, (ns >> 32) & 0xffffffff);
	aq_hw_write_reg(aq_hw, clock_sel == ATL_PTP_CLOCK_SEL_PTM ?
		HW_ATL2_MODIFYPTMNANOSECONDCOUNTERVALUE0_ADR :
		HW_ATL2_MODIFYPTPNANOSECONDCOUNTERVALUE0_ADR, (ns >>  0) & 0xffffffff);
	aq_hw_write_reg(aq_hw, clock_sel == ATL_PTP_CLOCK_SEL_PTM ?
		HW_ATL2_MODIFYPTMFRACTIONALNANOSECONDCOUNTERVALUE1_ADR :
		HW_ATL2_MODIFYPTPFRACTIONALNANOSECONDCOUNTERVALUE1_ADR, fns);
}

inline static void _hw_atl2_tsg_clock_set(struct aq_hw_s *aq_hw,
					  u32 clock_sel, u32 val)
{
	aq_hw_write_reg_bit(aq_hw, HW_ATL2_TIMESTAMPGENERATOR2_ADR,
			clock_sel == ATL_PTP_CLOCK_SEL_PTM ?
				HW_ATL2_SETPTMTIMERCOUNTERS_MSK :
				HW_ATL2_SETPTPTIMERCOUNTERS_MSK,
			clock_sel == ATL_PTP_CLOCK_SEL_PTM ?
				HW_ATL2_SETPTMTIMERCOUNTERS_SHIFT :
				HW_ATL2_SETPTPTIMERCOUNTERS_SHIFT,
			val);
}

void hw_atl2_tsg_clock_set(struct aq_hw_s *aq_hw, u32 clock_sel, u64 ns, u32 fns)
{
	_hw_atl2_tsg_clock_modify_value(aq_hw, clock_sel, ns, fns);
	_hw_atl2_tsg_clock_set(aq_hw, clock_sel, 1);
	_hw_atl2_tsg_clock_set(aq_hw, clock_sel, 0);
}

inline static void _hw_atl2_tsg_clock_add(struct aq_hw_s *aq_hw, u32 clock_sel, u32 val)
{
	aq_hw_write_reg_bit(aq_hw, HW_ATL2_TIMESTAMPGENERATOR2_ADR,
		clock_sel == ATL_PTP_CLOCK_SEL_PTM ?
			HW_ATL2_ADDPTMTIMERCOUNTERS_MSK :
			HW_ATL2_ADDPTPTIMERCOUNTERS_MSK,
		clock_sel == ATL_PTP_CLOCK_SEL_PTM ?
			HW_ATL2_ADDPTMTIMERCOUNTERS_SHIFT :
			HW_ATL2_ADDPTPTIMERCOUNTERS_SHIFT,
		val);
}

void hw_atl2_tsg_clock_add(struct aq_hw_s *aq_hw, u32 clock_sel, u64 ns, u32 fns)
{
	_hw_atl2_tsg_clock_modify_value(aq_hw, clock_sel, ns, fns);
	_hw_atl2_tsg_clock_add(aq_hw, clock_sel, 1);
	_hw_atl2_tsg_clock_add(aq_hw, clock_sel, 0);
}

inline static void _hw_atl2_tsg_clock_sub(struct aq_hw_s *aq_hw, u32 clock_sel,
					  u32 val)
{
	aq_hw_write_reg_bit(aq_hw, HW_ATL2_TIMESTAMPGENERATOR2_ADR,
		clock_sel == ATL_PTP_CLOCK_SEL_PTM ?
			HW_ATL2_SUBTRACTPTMTIMERCOUNTERS_MSK :
			HW_ATL2_SUBTRACTPTPTIMERCOUNTERS_MSK,
		clock_sel == ATL_PTP_CLOCK_SEL_PTM ?
			HW_ATL2_SUBTRACTPTMTIMERCOUNTERS_SHIFT :
			HW_ATL2_SUBTRACTPTPTIMERCOUNTERS_SHIFT,
		val);
}

void hw_atl2_tsg_clock_sub(struct aq_hw_s *aq_hw, u32 clock_sel, u64 ns, u32 fns)
{
	_hw_atl2_tsg_clock_modify_value(aq_hw, clock_sel, ns, fns);
	_hw_atl2_tsg_clock_sub(aq_hw, clock_sel, 1);
	_hw_atl2_tsg_clock_sub(aq_hw, clock_sel, 0);
}

inline static void _hw_atl2_tsg_clock_set_inc_cfg(struct aq_hw_s *aq_hw, u32 clock_sel, u32 nsfns)
{
	aq_hw_write_reg(aq_hw, clock_sel == ATL_PTP_CLOCK_SEL_PTM ?
			HW_ATL2_PTMCOUNTERINCREMENTVALUE_ADR :
			HW_ATL2_PTPCOUNTERINCREMENTVALUE_ADR, nsfns);
}

inline static void _hw_atl2_tsg_clock_inc_set(struct aq_hw_s *aq_hw, u32 clock_sel, u32 val)
{
	aq_hw_write_reg_bit(aq_hw, HW_ATL2_TIMESTAMPGENERATOR2_ADR,
		clock_sel == ATL_PTP_CLOCK_SEL_PTM ?
			HW_ATL2_PTMLOADPERCLOCKINCREMENTVALUE_MSK :
			HW_ATL2_PTPLOADPERCLOCKINCREMENTVALUE_MSK,
		clock_sel == ATL_PTP_CLOCK_SEL_PTM ?
			HW_ATL2_PTMLOADPERCLOCKINCREMENTVALUE_SHIFT :
			HW_ATL2_PTPLOADPERCLOCKINCREMENTVALUE_SHIFT,
		val);
}

void hw_atl2_tsg_clock_increment_set(struct aq_hw_s *aq_hw, u32 clock_sel, u32 ns, u32 fns)
{
	u32 nsfns = (ns & 0xff) | (fns & 0xffffff00);
	_hw_atl2_tsg_clock_set_inc_cfg(aq_hw, clock_sel, nsfns);
	_hw_atl2_tsg_clock_inc_set(aq_hw, clock_sel, 1);
	_hw_atl2_tsg_clock_inc_set(aq_hw, clock_sel, 0);
}

void hw_atl2_rpf_rx_desc_timestamp_req_set(struct aq_hw_s *aq_hw, u32 request,
					    u32 descriptor)
{
	aq_hw_write_reg_bit(aq_hw,
			    HW_ATL2_RPF_TIMESTAMP_REQ_DESCD_ADR(descriptor),
			    HW_ATL2_RPF_TIMESTAMP_REQ_DESCD_MSK,
			    HW_ATL2_RPF_TIMESTAMP_REQ_DESCD_SHIFT, request);
}

u32 hw_atl2_rpf_rx_desc_timestamp_req_get(struct aq_hw_s *aq_hw,
					   u32 descriptor)
{
	return aq_hw_read_reg_bit(aq_hw,
				  HW_ATL2_RPF_TIMESTAMP_REQ_DESCD_ADR(descriptor),
				  HW_ATL2_RPF_TIMESTAMP_REQ_DESCD_MSK,
				  HW_ATL2_RPF_TIMESTAMP_REQ_DESCD_SHIFT);
}

/* TX */

void hw_atl2_tpb_tx_buf_clk_gate_en_set(struct aq_hw_s *aq_hw, u32 clk_gate_en)
{
	aq_hw_write_reg_bit(aq_hw, HW_ATL2_TPB_TX_BUF_CLK_GATE_EN_ADR,
			    HW_ATL2_TPB_TX_BUF_CLK_GATE_EN_MSK,
			    HW_ATL2_TPB_TX_BUF_CLK_GATE_EN_SHIFT,
			    clk_gate_en);
}

void hw_atl2_reg_tx_intr_moder_ctrl_set(struct aq_hw_s *aq_hw,
				       u32 tx_intr_moderation_ctl,
				       u32 queue)
{
	aq_hw_write_reg(aq_hw, HW_ATL2_TX_INTR_MODERATION_CTL_ADR(queue),
			tx_intr_moderation_ctl);
}

void hw_atl2_tdm_tx_desc_timestamp_writeback_en_set(struct aq_hw_s *aq_hw,
						     u32 enable, u32 descriptor)
{
	aq_hw_write_reg_bit(aq_hw, HW_ATL2_TDM_DESCD_TS_WRB_EN_ADR(descriptor),
			    HW_ATL2_TDM_DESCD_TS_WRB_EN_MSK,
			    HW_ATL2_TDM_DESCD_TS_WRB_EN_SHIFT, enable);
}

u32 hw_atl2_tdm_tx_desc_timestamp_writeback_en_get(struct aq_hw_s *aq_hw,
						    u32 descriptor)
{
	return aq_hw_read_reg_bit(aq_hw,
				  HW_ATL2_TDM_DESCD_TS_WRB_EN_ADR(descriptor),
				  HW_ATL2_TDM_DESCD_TS_WRB_EN_MSK,
				  HW_ATL2_TDM_DESCD_TS_WRB_EN_SHIFT);
}


void hw_atl2_tdm_tx_desc_timestamp_en_set(struct aq_hw_s *aq_hw, u32 enable,
					   u32 descriptor)
{
	aq_hw_write_reg_bit(aq_hw, HW_ATL2_TDM_DESCD_TS_EN_ADR(descriptor),
			    HW_ATL2_TDM_DESCD_TS_EN_MSK,
			    HW_ATL2_TDM_DESCD_TS_EN_SHIFT, enable);
}

u32 hw_atl2_tdm_tx_desc_timestamp_en_get(struct aq_hw_s *aq_hw, u32 descriptor)
{
	return aq_hw_read_reg_bit(aq_hw,
				  HW_ATL2_TDM_DESCD_TS_EN_ADR(descriptor),
				  HW_ATL2_TDM_DESCD_TS_EN_MSK,
				  HW_ATL2_TDM_DESCD_TS_EN_SHIFT);
}


void hw_atl2_tdm_tx_desc_avb_en_set(struct aq_hw_s *aq_hw, u32 enable,
				     u32 descriptor)
{
	aq_hw_write_reg_bit(aq_hw, HW_ATL2_TDM_DESCD_AVB_EN_ADR(descriptor),
			    HW_ATL2_TDM_DESCD_AVB_EN_MSK,
			    HW_ATL2_TDM_DESCD_AVB_EN_SHIFT, enable);
}

u32 hw_atl2_tdm_tx_desc_avb_en_get(struct aq_hw_s *aq_hw, u32 descriptor)
{
	return aq_hw_read_reg_bit(aq_hw,
				  HW_ATL2_TDM_DESCD_AVB_EN_ADR(descriptor),
				  HW_ATL2_TDM_DESCD_AVB_EN_MSK,
				  HW_ATL2_TDM_DESCD_AVB_EN_SHIFT);
}

void hw_atl2_tps_tx_pkt_shed_tc_data_max_credit_set(struct aq_hw_s *aq_hw,
						    u32 max_credit,
						    u32 tc)
{
	aq_hw_write_reg_bit(aq_hw, HW_ATL2_TPS_DATA_TCTCREDIT_MAX_ADR(tc),
			    HW_ATL2_TPS_DATA_TCTCREDIT_MAX_MSK,
			    HW_ATL2_TPS_DATA_TCTCREDIT_MAX_SHIFT,
			    max_credit);
}

void hw_atl2_tps_tx_pkt_shed_tc_data_weight_set(struct aq_hw_s *aq_hw,
						u32 tx_pkt_shed_tc_data_weight,
						u32 tc)
{
	aq_hw_write_reg_bit(aq_hw, HW_ATL2_TPS_DATA_TCTWEIGHT_ADR(tc),
			    HW_ATL2_TPS_DATA_TCTWEIGHT_MSK,
			    HW_ATL2_TPS_DATA_TCTWEIGHT_SHIFT,
			    tx_pkt_shed_tc_data_weight);
}

void _hw_atl2_tsg_ptp_gpio_en_pulse(struct aq_hw_s *aq_hw, u32 clk_sel, bool en)
{
	/* Enable GPIO and set "GTorEQ" mode */
	u32 val = en ? (2 << HW_ATL2_PTPCLOCKEVENTOUTPUTMODE_SHIFT) | 1 : 0;
	aq_hw_write_reg_bit(aq_hw, HW_ATL2_TSGPTPGPIOCTRL_ADR,
		clk_sel == ATL_PTP_CLOCK_SEL_PTM ?
			HW_ATL2_PTMCLOCKEVENTOUTPUTMODE_MSK |
				HW_ATL2_PTMENCLOCKEVENTOUTPUT_MSK :
			HW_ATL2_PTPCLOCKEVENTOUTPUTMODE_MSK |
				HW_ATL2_PTPENCLOCKEVENTOUTPUT_MSK,
		clk_sel == ATL_PTP_CLOCK_SEL_PTM ?
			HW_ATL2_PTMENCLOCKEVENTOUTPUT_SHIFT :
			HW_ATL2_PTPENCLOCKEVENTOUTPUT_SHIFT,
		val);
}

void _hw_atl2_tsg_ptp_gpio_ts(struct aq_hw_s *aq_hw, u32 clk_sel, u64 ts)
{
	aq_hw_write_reg(aq_hw, clk_sel == ATL_PTP_CLOCK_SEL_PTM ?
				HW_ATL2_MODIFYPTMNANOSECONDCOUNTERVALUE0_ADR :
				HW_ATL2_MODIFYPTPNANOSECONDCOUNTERVALUE0_ADR,
				ts & 0xffffffff);
	aq_hw_write_reg(aq_hw, clk_sel == ATL_PTP_CLOCK_SEL_PTM ?
				HW_ATL2_MODIFYPTMNANOSECONDCOUNTERVALUE1_ADR :
				HW_ATL2_MODIFYPTPNANOSECONDCOUNTERVALUE1_ADR,
				(ts >> 32) & 0xffffffff);
}

void hw_atl2_tsg_ptp_gpio_gen_pulse(struct aq_hw_s *aq_hw, u32 clk_sel, u64 ts)
{
	aq_pr_trace("%s clk %d, ts %010llu\n", __FUNCTION__, clk_sel, ts);
	_hw_atl2_tsg_ptp_gpio_ts(aq_hw, clk_sel, ts);
	_hw_atl2_tsg_ptp_gpio_en_pulse(aq_hw, clk_sel, !!ts);
}

void hw_atl2_tdm_tx_data_read_req_limit_set(struct aq_hw_s *aq_hw, u32 limit)
{
	aq_hw_write_reg_bit(aq_hw, HW_ATL2_TDM_TX_DATA_RD_REQ_LIMIT_ADR,
			    HW_ATL2_TDM_TX_DATA_RD_REQ_LIMIT_MSK,
			    HW_ATL2_TDM_TX_DATA_RD_REQ_LIMIT_SHIFT, limit);
}

void hw_atl2_tdm_tx_desc_read_req_limit_set(struct aq_hw_s *aq_hw, u32 limit)
{
	aq_hw_write_reg_bit(aq_hw, HW_ATL2_TDM_TX_DESC_RD_REQ_LIMIT_ADR,
			    HW_ATL2_TDM_TX_DESC_RD_REQ_LIMIT_MSK,
			    HW_ATL2_TDM_TX_DESC_RD_REQ_LIMIT_SHIFT, limit);
}

u32 hw_atl2_get_hw_version(struct aq_hw_s *aq_hw)
{
	return aq_hw_read_reg(aq_hw, HW_ATL2_FPGA_VER_ADR);
}

void hw_atl2_init_launchtime(struct aq_hw_s *aq_hw)
{
	u32 hw_ver = hw_atl2_get_hw_version(aq_hw);
	// TODO: do we need this correction for real HW?
	aq_hw_write_reg_bit(aq_hw, HW_ATL2_LT_CTRL_ADR,
		HW_ATL2_LT_CTRL_CLK_RATIO_MSK,
		HW_ATL2_LT_CTRL_CLK_RATIO_SHIFT,
		hw_ver  < HW_ATL2_FPGA_VER_U32(1, 0, 0, 0) ?
			HW_ATL2_LT_CTRL_CLK_RATIO_FULL_SPEED :
		hw_ver >= HW_ATL2_FPGA_VER_U32(1, 0, 85, 2) ?
			HW_ATL2_LT_CTRL_CLK_RATIO_HALF_SPEED :
		 	HW_ATL2_LT_CTRL_CLK_RATIO_QUATER_SPEED);
}

void hw_atl2_rpfl3l4_ipv4_dest_addr_clear(struct aq_hw_s *aq_hw, u8 location)
{
	aq_hw_write_reg(aq_hw, HW_ATL2_RX_GET_ADDR_DESTA_FL3L4(location), 0U);
}

void hw_atl2_rpfl3l4_ipv4_src_addr_clear(struct aq_hw_s *aq_hw, u8 location)
{
	aq_hw_write_reg(aq_hw, HW_ATL2_RX_GET_ADDR_SRCA_FL3L4(location), 0U);
}

void hw_atl2_rpfl3l4_cmd_clear(struct aq_hw_s *aq_hw, u8 location)
{
	aq_hw_write_reg(aq_hw, HW_ATL2_RX_GET_ADDR_CTRL_FL3L4(location), 0U);
}

void hw_atl2_rpfl3l4_ipv6_dest_addr_clear(struct aq_hw_s *aq_hw, u8 location)
{
	int i;

	for (i = 0; i < 4; ++i)
		aq_hw_write_reg(aq_hw,
				HW_ATL2_RX_GET_ADDR_DESTA_FL3L4(location + i),
				0U);
}

void hw_atl2_rpfl3l4_ipv6_src_addr_clear(struct aq_hw_s *aq_hw, u8 location)
{
	int i;

	for (i = 0; i < 4; ++i)
		aq_hw_write_reg(aq_hw,
				HW_ATL2_RX_GET_ADDR_SRCA_FL3L4(location + i),
				0U);
}

void hw_atl2_rpfl3l4_ipv4_dest_addr_set(struct aq_hw_s *aq_hw, u8 location,
				       u32 ipv4_dest)
{
	aq_hw_write_reg(aq_hw, HW_ATL2_RX_GET_ADDR_DESTA_FL3L4(location),
			ipv4_dest);
}

void hw_atl2_rpfl3l4_ipv4_src_addr_set(struct aq_hw_s *aq_hw, u8 location,
				      u32 ipv4_src)
{
	aq_hw_write_reg(aq_hw,
			HW_ATL2_RX_GET_ADDR_SRCA_FL3L4(location),
			ipv4_src);
}

void hw_atl2_rpfl3l4_cmd_set(struct aq_hw_s *aq_hw, u8 location, u32 cmd)
{
	aq_hw_write_reg(aq_hw, HW_ATL2_RX_GET_ADDR_CTRL_FL3L4(location), cmd);
}

void hw_atl2_rpfl3l4_ipv6_src_addr_set(struct aq_hw_s *aq_hw, u8 location,
				      u32 *ipv6_src)
{
	int i;

	for (i = 0; i < 4; ++i)
		aq_hw_write_reg(aq_hw,
				HW_ATL2_RX_GET_ADDR_SRCA_FL3L4(location + i),
				ipv6_src[i]);
}

void hw_atl2_rpfl3l4_ipv6_dest_addr_set(struct aq_hw_s *aq_hw, u8 location,
				       u32 *ipv6_dest)
{
	int i;

	for (i = 0; i < 4; ++i)
		aq_hw_write_reg(aq_hw,
				HW_ATL2_RX_GET_ADDR_DESTA_FL3L4(location + i),
				ipv6_dest[i]);
}

/* set action resolver record */
void hw_atl2_rpf_act_rslvr_record_set(struct aq_hw_s *aq_hw, u8 location,
				      u32 tag, u32 mask, u32 action)
{
	aq_hw_write_reg(aq_hw,
			HW_ATL2_RPF_ACT_RSLVR_REQ_TAG_ADR(location),
			tag);
	aq_hw_write_reg(aq_hw,
			HW_ATL2_RPF_ACT_RSLVR_TAG_MASK_ADR(location),
			mask);
	aq_hw_write_reg(aq_hw,
			HW_ATL2_RPF_ACT_RSLVR_ACTN_ADR(location),
			action);
}

/* get action resolver tag record */
void hw_atl2_rpf_act_rslvr_record_get(struct aq_hw_s *aq_hw, u8 location,
				      u32 *tag, u32 *mask, u32 *action)
{
	*tag = aq_hw_read_reg(aq_hw,
			      HW_ATL2_RPF_ACT_RSLVR_REQ_TAG_ADR(location));
	*mask = aq_hw_read_reg(aq_hw,
			       HW_ATL2_RPF_ACT_RSLVR_TAG_MASK_ADR(location));
	*action = aq_hw_read_reg(aq_hw,
				 HW_ATL2_RPF_ACT_RSLVR_ACTN_ADR(location));
}

void hw_atl2_rpf_act_rslvr_section_en_set(struct aq_hw_s *aq_hw, u32 sections)
{
	aq_hw_write_reg_bit(aq_hw, HW_ATL2_RPF_REC_TAB_EN_ADR,
			HW_ATL2_RPF_REC_TAB_EN_MSK,
			HW_ATL2_RPF_REC_TAB_EN_SHIFT,
			sections);
}

void hw_atl2_mif_shared_buf_get(struct aq_hw_s *aq_hw, int offset,
				  u32 *data, int len)
{
	int i;
	int j = 0;

	for (i = offset; i < offset + len; i++, j++)
		data[j] = aq_hw_read_reg(aq_hw,
					 HW_ATL2_MIF_SHARED_BUFFER_IN_ADR(i));
}

void hw_atl2_mif_shared_buf_write(struct aq_hw_s *aq_hw, int offset,
				  u32 *data, int len)
{
	int i;
	int j = 0;

	for (i = offset; i < offset + len; i++, j++)
		aq_hw_write_reg(aq_hw, HW_ATL2_MIF_SHARED_BUFFER_IN_ADR(i),
				data[j]);
}

void hw_atl2_mif_shared_buf_read(struct aq_hw_s *aq_hw, int offset,
				  u32 *data, int len)
{
	int i;
	int j = 0;

	for (i = offset; i < offset + len; i++, j++)
		data[j] = aq_hw_read_reg(aq_hw,
					 HW_ATL2_MIF_SHARED_BUFFER_OUT_ADR(i));
}

void hw_atl2_mif_host_finished_write_set(struct aq_hw_s *aq_hw, u32 finish)
{
	aq_hw_write_reg_bit(aq_hw, HW_ATL2_MIF_HOST_FINISHED_WRITE_ADR,
			    HW_ATL2_MIF_HOST_FINISHED_WRITE_MSK,
			    HW_ATL2_MIF_HOST_FINISHED_WRITE_SHIFT,
			    finish);
}

u32 hw_atl2_mif_mcp_finished_read_get(struct aq_hw_s *aq_hw)
{
	return aq_hw_read_reg_bit(aq_hw, HW_ATL2_MIF_MCP_FINISHED_READ_ADR,
				  HW_ATL2_MIF_MCP_FINISHED_READ_MSK,
				  HW_ATL2_MIF_MCP_FINISHED_READ_SHIFT);
}

u32 hw_atl2_mif_mcp_boot_reg_get(struct aq_hw_s *aq_hw)
{
	return aq_hw_read_reg(aq_hw, HW_ATL2_MIF_BOOT_REG_ADR);
}

void hw_atl2_mif_mcp_boot_reg_set(struct aq_hw_s *aq_hw, u32 val)
{
	return aq_hw_write_reg(aq_hw, HW_ATL2_MIF_BOOT_REG_ADR, val);
}

u32 hw_atl2_mif_host_req_int_get(struct aq_hw_s *aq_hw)
{
	return aq_hw_read_reg(aq_hw, HW_ATL2_MCP_HOST_REQ_INT_ADR);
}

void hw_atl2_mif_host_req_int_clr(struct aq_hw_s *aq_hw, u32 val)
{
	return aq_hw_write_reg(aq_hw, HW_ATL2_MCP_HOST_REQ_INT_CLR_ADR,
			       val);
}
