/******************************************************************************
 *                    Copyright 2014-2017 aQuantia Corporation
 *                    Confidential and Proprietary
 *
 * $File: aqdiag-interrupts.c $
 *
 * $Revision: $
 *
 * $DateTime: $
 *
 * $Author:   $
 *
 ******************************************************************************/
#include "aqdiag.h"
#include <linux/interrupt.h>
#include <linux/sched.h>
#include <linux/version.h>
#if LINUX_VERSION_CODE > KERNEL_VERSION(3,4,0) //FIX VALUE
#include <linux/module.h>
#endif
#if LINUX_VERSION_CODE >= KERNEL_VERSION(4,11,0) //FIX VALUE
#include <linux/uaccess.h>
#endif
#ifdef __arm__
#include <asm/div64.h>
#endif


static int requested_vectors = 8;
module_param_named(vectors, requested_vectors, int, 0644);
static int requested_inttype = MSIX;
module_param_named(inttype, requested_inttype, int, 0644);

static inline unsigned vec_irq(struct aqdiag_dev *dev, int i)
{
	switch (dev->interrupts.type) {
	case MSIX:
		return dev->interrupts.msix_entries[i].vector;
	case MSI:
		return dev->pdev->irq + i;
	case LEGACY_INTERRUPT:
		return dev->pdev->irq;
	}
	__builtin_unreachable();
}

static irqreturn_t aqdiag_int(int irq, void *priv)
{
	struct aqdiag_vector *vec = priv;
	uint64_t now, delta;
	int count;

#ifdef __arm__
	now = (uint64_t) ktime_to_ns(ktime_get());
	do_div(now, 100); // prevents warnings about undefined __aeabi_ldivmod and __aeabi_uldivmod symbols
#else
	now = ktime_to_ns(ktime_get()) / 100; //TODO
#endif
	spin_lock(&vec->lock);
	count = atomic_inc_return(&vec->count);
	if (count == 1) {
		vec->min = UINT64_MAX;
		vec->max = 0;
	} else {
		delta = now - vec->last;
		if (delta < vec->min)
			vec->min = delta;
		if (delta > vec->max)
			vec->max = delta;
	}
	vec->last = now;
	spin_unlock(&vec->lock);
	wake_up(&vec->wait);
	return IRQ_HANDLED;
}

#if !defined(RHEL_RELEASE) && LINUX_VERSION_CODE < KERNEL_VERSION(3,14,0) // not defined in older linux, so define it ourselves from newer linux sources
static int pci_enable_msix_range(struct pci_dev *dev,
                                   struct msix_entry *entries, int minvec,
                                   int maxvec)
{
        int rc, nvec = maxvec;

        if (maxvec < minvec)
                return -ERANGE;

        for (;;) {
                rc = pci_enable_msix(dev, entries, nvec);
                if (rc == 0)
                        return nvec;

                if (rc < 0)
                        return rc;
                if (rc < minvec)
                        return -ENOSPC;

                nvec = rc;
        }
}
#endif

#if !defined(RHEL_RELEASE) && LINUX_VERSION_CODE < KERNEL_VERSION(3,14,0) || LINUX_VERSION_CODE >= KERNEL_VERSION(4,11,0) // not defined in older linux, so define it ourselves from newer linux sources
static int pci_enable_msi_range(struct pci_dev *dev, int minvec,
                                   int maxvec)
{
        int rc, nvec = maxvec;

        if (maxvec < minvec)
                return -ERANGE;

        for (;;) {
#if LINUX_VERSION_CODE >= KERNEL_VERSION(4,11,0) //FIX VALUE
                rc = pci_enable_msi(dev);
#else
                rc = pci_enable_msi_block(dev, nvec);
#endif
                if (rc == 0)
                        return nvec;

                if (rc < 0)
                        return rc;
                if (rc < minvec)
                        return -ENOSPC;

                nvec = rc;
        }
}
#endif

int aqdiag_init_interrupts(struct aqdiag_dev *dev)
{
	int ret, nvecs, i;
	struct aqdiag_interrupts *ints = &dev->interrupts;

	switch (requested_inttype) {
		struct msix_entry *entries;

	case MSIX:
		entries = kcalloc(requested_vectors, sizeof(struct msix_entry), GFP_KERNEL);
		if (!entries)
			return -ENOMEM;

		for (i = 0; i < requested_vectors; i++)
			entries[i].entry = i;

		nvecs = pci_enable_msix_range(dev->pdev, entries, 1,
						requested_vectors);

		if (nvecs > 0) {
			ints->num = nvecs;
			ints->msix_entries = entries;
			ints->disable = &pci_disable_msix;
			ints->type = MSIX;
			break;
		}
		dev_warn(&dev->pdev->dev, "Can't init MSI-X (%d), falling back to MSI\n", nvecs);
		kfree(entries);
		// fall-through
	case MSI:
		nvecs = pci_enable_msi_range(dev->pdev, 1, requested_vectors);

		if (nvecs > 0) {
			ints->num = nvecs;
			ints->disable = &pci_disable_msi;
			ints->type = MSI;
			break;
		}
		dev_warn(&dev->pdev->dev, "Can't init MSI (%d), falling back to legacy\n", nvecs);
		// fall-through
	case LEGACY_INTERRUPT:
		ints->num = 1;
		ints->type = LEGACY_INTERRUPT;
		break;
	}

	ints->vecs = kcalloc(ints->num, sizeof(struct aqdiag_vector), GFP_KERNEL);
	if (!ints->vecs) {
		ret = -ENOMEM;
		goto err_alloc_vecs;
	}

	for (i = 0; i < ints->num; i++) {
		struct aqdiag_vector *vec = &ints->vecs[i];
		vec->irq = vec_irq(dev, i);
		vec->index = i;
		snprintf(vec->name, sizeof(vec->name), "aqdiag%d-%d", dev->minor, i);
		init_waitqueue_head(&vec->wait);
		atomic_set(&vec->count, 0);
		spin_lock_init(&vec->lock);
		ret = request_irq(vec->irq, aqdiag_int, 0, vec->name, vec);
		if (ret) {
			dev_err(&dev->pdev->dev, "request_irq failed for vector %d: %d", i, ret);
			goto err_req_irq;
		}

	}

	return 0;

err_req_irq:
	while (i) {
		struct aqdiag_vector *vec = &ints->vecs[--i];
		free_irq(vec->irq, vec);
	}
	kfree(ints->vecs);
err_alloc_vecs:
	if (ints->disable)
		ints->disable(dev->pdev);
	kfree(ints->msix_entries);

	return ret;
}

void aqdiag_free_irqs(struct aqdiag_dev *dev)
{
	struct aqdiag_interrupts *ints = &dev->interrupts;
	int i;

	for (i = 0; i < ints->num; i++) {
		struct aqdiag_vector *vec = &ints->vecs[i];
		free_irq(vec->irq, vec);
	}
	kfree(ints->vecs);
	if (ints->disable)
		ints->disable(dev->pdev);
	kfree(ints->msix_entries);
}

int aqdiag_req_int(struct aqdiag_dev *dev, struct aqdiag_req_int __user *ureq)
{
	struct aqdiag_req_int req;
	struct aqdiag_vector *vec;
	unsigned long flags;
	long timeout;
#ifdef __arm__
	uint64_t temp;
#endif

	if (copy_from_user(&req, ureq, sizeof(req)))
		return -EFAULT;

	if (req.vector > dev->interrupts.num)
		return -EINVAL;

	vec = &dev->interrupts.vecs[req.vector];
	if (req.reset_count)
		atomic_set(&vec->count, 0);

#ifdef __arm__
	temp = req.timeout;
	do_div(temp, 10); // prevents warnings about undefined __aeabi_ldivmod and __aeabi_uldivmod symbols
	timeout = usecs_to_jiffies(temp);
#else
	timeout = usecs_to_jiffies(req.timeout / 10);
#endif
	req.count = atomic_xchg(&vec->count, 0);
	if( !req.count )
	{
		timeout = wait_event_interruptible_timeout(vec->wait,
							   (req.count = atomic_xchg(&vec->count, 0)),
							   timeout);
	}
	spin_lock_irqsave(&vec->lock, flags);
	req.time_min = vec->min;
	req.time_max = vec->max;
	req.time_last = vec->last;
	spin_unlock_irqrestore(&vec->lock, flags);

	if(copy_to_user(ureq, &req, sizeof(req)))
		return -EFAULT;

	return timeout > 0 ? 0 : timeout == 0 ? -ETIME : timeout;
}

void aqdiag_cancel_req(struct aqdiag_dev *dev)
{
	int i;
	for (i = 0; i < requested_vectors; i++)
	{
		struct aqdiag_vector *vec = &dev->interrupts.vecs[i];
		wake_up(&vec->wait);
	}
}
