using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading;
using System.Threading.Channels;
using System.Threading.Tasks;
using System.Windows.Forms;
using Intel.Cst.Client;
using Microsoft.Extensions.Logging;
using Microsoft.Win32;
using WALWOASampleApp.CallBackFunctions;
using WALWOASampleApp.Properties;

namespace SampleApp
{
    public partial class FormWALWOA : Form
    {
        private static Logger logger = new Logger();

        private ClientBackendLink _commClient = null;
        private readonly int[] preDimIntervalSeconds = new int[] { 3, 5, 10, 15, 20, 25, 30, 35, 40, 45, 50, 55, 60 };
        private readonly int[] userPresentWaitIntervalsSeconds = new int[] { 5, 10 };
        private readonly int[] lockAfterDimIntervalsSeconds = new int[] { 3, 5, 10, 15, 20, 25, 30, 35, 40, 45, 50, 55, 60 };
        private readonly int[] mispredictionDimWaitSeconds = new int[] { 15, 20, 25, 30, 35, 40, 45, 50, 55, 60, 75, 90, 105, 120, 150, 180, 210, 240, 270, 300 };
        private readonly int[] mispredictionTimeWindowSeconds = new int[] { 3, 5, 7, 10 };
        private readonly int[] batteryRemainingPercentage = new int[] { 5, 10, 15, 20, 25, 30, 35, 40, 45, 50 };
        private readonly int[] dimTarget = new int[] { 0, 5, 10, 15, 20, 25 };
        private readonly int[] dimmingIntervalSeconds = new int[] { 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20 };
        private readonly int[] failSafeIntervalSeconds = new int[] { 1800, 3600 };
        private readonly int[] detectionAbsenceIntervalSeconds = new int[] { 1, 2, 3, 5, 10, 15, 20, 25, 30, 35, 40, 45, 50, 55, 60 };
        private readonly int[] correlationResetSeconds = new int[] { 0, 15, 20, 25, 30, 35, 40, 45, 50, 55, 60, 75, 90, 105, 120 };
        private readonly double[] proximityCooldownSeconds = new double[] { 0.5, 0, 1, 5, 10, 30 };
        private readonly double[] hidActiveDelaySeconds = new double[] { 0.5, 1, 1.5, 2, 2.5, 3, 3.5, 4, 4.5, 5, 10, 15, 20 };
        private bool serviceReconnectRequired = false;
        private bool connectedRemotely = false;
        private IFeatureCallback featureCallback;
        private SettingsAccessType globalMode = SettingsAccessType.LOCAL;
        private readonly string[] appNames = new string[] { "Intel.ContextSensingSampleApp", "Intel User Awareness Sample App", "Dell.DellOptimizer", "DellExpressSignIn" };
        private readonly string[] eventNames = EventTypeMap.EventSubscriptionNames.Values.ToArray();
        private CancellationTokenSource humanPresenceTaskTokenSource;
        private Task humanPresenceBackgroundTask = Task.CompletedTask;
        private const int HUMAN_PRESENCE_UPDATE_INTERVAL = 1500;
        const string DATA_CONNECTION_ERROR_TEXT = "FAULT"; // When displayed data depends on an active client connection (poll) but there is no connection
        const string DATA_UNAVAILABLE_TEXT = "UNAVAILABLE"; // When displayed data depends on an event but the event is not subscribed
        private bool snoozeRequested = false;
        private bool visionCapable = true;

        private readonly List<EventType> defaultEvents = new List<EventType>()
        {
            EventType.EVENT_CST_CAPABILITY_CHANGED,
            EventType.EVENT_CST_EXTENDED_CAPABILITY_CHANGED,
            EventType.EVENT_HUMAN_PRESENCE_CHANGED
        };

        public bool CheckedInitialOnlookerState { get; private set; } = false;
        public bool CheckedInitialAdaptiveDimmingState { get; private set; } = false;
        public bool HumanPresencePollEnabled => this.humanPresencePollCheckbox.Checked;
        private readonly SemaphoreSlim humanPresenceUpdateLock = new SemaphoreSlim(1, 1);

        // Queue up human presence status packets so we never apply them out of the order we receive them in.
        // This is important because we lock the human presence status while polling (takes 3 calls to get all of the information)
        // so it is possible multiple events coming in shortly after another could lead to semaphore pileup. Using a Channel
        // (which is essentially a producer-consumer pattern queue) makes sure we never end up in that situation.
        private readonly Channel<HumanPresenceStatusPacket> humanPresenceChannel = Channel.CreateUnbounded<HumanPresenceStatusPacket>();
        public ChannelWriter<HumanPresenceStatusPacket> HumanPresenceChannelWriter => this.humanPresenceChannel.Writer;

        #region constructor
        public FormWALWOA(bool bHidePrimaryGui = false)
        {
            InitializeComponent();
            this.icstSampleAppVersionLabel.Text = "Intel Context Sensing SampleApp Version " + Application.ProductVersion.ToString();
            this.featureCallback = new FeatureCallback();
            SystemEvents.SessionSwitch += new SessionSwitchEventHandler(SystemEvents_SessionSwitch);
            if (bHidePrimaryGui)
            {
                this.WindowState = FormWindowState.Minimized;
            }

            this.humanPresenceTaskTokenSource = new CancellationTokenSource();
            this.humanPresenceTaskTokenSource.Cancel();

            // Run human presence event handler in background
            // In a modern application, this should be a hosted service or similar, but since we don't have
            // that infrastructure in this SampleApp, we run it as a background task. Do not use this method
            // in a production application.
            _ = HumanPresenceEventHandler(default);
        }
        #endregion constructor

        #region system event handlers
        private async void SystemEvents_SessionSwitch(object sender, SessionSwitchEventArgs e)
        {
            logger.Debug("SessionSwitch event occurred with reason = " + e.Reason);
            if (this.serviceReconnectRequired && !this.connectedRemotely &&
                (e.Reason == SessionSwitchReason.SessionLogon ||
                 e.Reason == SessionSwitchReason.SessionUnlock))
            {
                this.serviceReconnectRequired = false;
                bool success = await RefreshFeaturesViaGetOptions();
                _ = GetAllFeatureCapabilities();

                if (success)
                {
                    SubscribeToDefaultEvents();
                }

                if (this.HumanPresencePollEnabled)
                {
                    StartHumanPresenceUpdateTask();
                }
                else
                {
                    // Since our HP is out of date, call the SDK once to make sure we have the latest data.
                    // After this, we'll get events if it changes as long as the app is subscribed.
                    UpdatePresenceStatus();
                }
            }
            else if (!this.serviceReconnectRequired &&
                     (e.Reason == SessionSwitchReason.SessionLogoff ||
                      e.Reason == SessionSwitchReason.SessionLock))
            {
                logger.Debug("SessionSwitch: Service reconnect is required");
                this.serviceReconnectRequired = true;
                StopHumanPresenceUpdateTask();
            }
            else if (e.Reason == SessionSwitchReason.RemoteConnect)
            {
                this.connectedRemotely = true;
                this.walCheckBox.Enabled = false;
                this.woaCheckBox.Enabled = false;
                this.nlopCheckBox.Enabled = false;
                this.adCheckBox.Enabled = false;
                this.onlookerDetectionCheckBox.Enabled = false;
                this.privacyCheckbox.Enabled = false;
                this.monitorAppCheckbox.Enabled = false;
                StopHumanPresenceUpdateTask();
            }
            else if (e.Reason == SessionSwitchReason.RemoteDisconnect)
            {
                this.connectedRemotely = false;
                this.walCheckBox.Enabled = true;
                this.woaCheckBox.Enabled = true;
                this.nlopCheckBox.Enabled = true;
                this.adCheckBox.Enabled = true;
                this.onlookerDetectionCheckBox.Enabled = true;
                this.privacyCheckbox.Enabled = true;
                this.monitorAppCheckbox.Enabled = true;
                // Intentionally not starting the human presence update task here
                // Defer starting it until the SessionLogon/Unlock to avoid spamming ICST with requests it will reject
            }
        }
        #endregion system event handlers

        #region form event handlers
        private async void FormWALWOA_Load(object sender, EventArgs e)
        {
            CenterToScreen();
            InitializeControls();
            this._commClient = new ClientBackendLink(this.featureCallback);

            if (this.HumanPresencePollEnabled)
            {
                StartHumanPresenceUpdateTask();
            }
            else
            {
                UpdatePresenceStatus();
            }

            bool success = await RefreshFeaturesViaGetOptions();

            if (success)
            {
                SetAppName();
                SubscribeToDefaultEvents();
            }

            _ = Task.Run(async () => await GetAllFeatureCapabilities());
        }

        private void FormWALWOA_FormClosing(object sender, FormClosingEventArgs e)
        {
            StopHumanPresenceUpdateTask();
            SystemEvents.SessionSwitch -= SystemEvents_SessionSwitch;
            Settings.Default.Save();
        }
        #endregion form event handlers

        #region control event handler
        internal async void checkBoxWAL_CheckedChanged(object sender, EventArgs e)
        {
            Cursor.Current = Cursors.WaitCursor;
            this.walCheckBox.Enabled = false;

            if (this.walCheckBox.Checked)
            {
                try
                {
                    if (this.globalMode == SettingsAccessType.GLOBAL)
                    {
                        logger.Info("Showing default settings for Walk Away Lock");
                        await this._commClient.SetOption(FeatureType.LOCK, FeatureProperty.FeatureEnabled, true, this.globalMode);
                    }
                    else
                    {
                        logger.Info("Walk Away Lock feature is being enabled");
                        this._commClient.Enable(FeatureType.LOCK);
                    }

                    FeatureSetting featureSetting = await this._commClient.GetOptions(FeatureType.LOCK, this.globalMode);
                    UpdateTimerControlSettings(featureSetting);
                }
                catch (Exception ex)
                {
                    this.walCheckBox.CheckedChanged -= checkBoxWAL_CheckedChanged;
                    this.walCheckBox.CheckState = CheckState.Unchecked;
                    this.walCheckBox.CheckedChanged += checkBoxWAL_CheckedChanged;
                    logger.Error("Error enabling Walk Away Lock " + ex.Message);
                    MessageBox.Show(ex.Message, "OnError", MessageBoxButtons.OK, MessageBoxIcon.Error);
                    this.walCheckBox.Enabled = true;
                }
            }
            else
            {
                try
                {
                    if (this.globalMode == SettingsAccessType.GLOBAL)
                    {
                        logger.Info("Setting global disable for Walk Away Lock");
                        await this._commClient.SetOption(FeatureType.LOCK, FeatureProperty.FeatureEnabled, false, this.globalMode);
                        FeatureSetting featureSetting = await this._commClient.GetOptions(FeatureType.LOCK, this.globalMode);
                        UpdateTimerControlSettings(featureSetting);
                    }
                    else
                    {
                        logger.Info("Walk Away Lock feature is being disabled");
                        this._commClient.Disable(FeatureType.LOCK);
                    }
                }
                catch (Exception ex)
                {
                    this.walCheckBox.CheckedChanged -= checkBoxWAL_CheckedChanged;
                    this.walCheckBox.CheckState = CheckState.Checked;
                    this.walCheckBox.CheckedChanged += checkBoxWAL_CheckedChanged;
                    logger.Error("Error disabling Walk Away Lock " + ex.Message);
                    MessageBox.Show(ex.Message, "OnError", MessageBoxButtons.OK, MessageBoxIcon.Error);
                    this.walCheckBox.Enabled = true;
                }
            }

            Cursor.Current = Cursors.Default;
        }

        internal async void checkBoxWOA_CheckedChanged(object sender, EventArgs e)
        {
            Cursor.Current = Cursors.WaitCursor;
            this.woaCheckBox.Enabled = false;

            if (this.woaCheckBox.Checked)
            {
                try
                {
                    if (this.globalMode == SettingsAccessType.GLOBAL)
                    {
                        logger.Info("Showing default settings for Wake On Approach");
                        await this._commClient.SetOption(FeatureType.WAKE, FeatureProperty.FeatureEnabled, true, this.globalMode);
                    }
                    else
                    {
                        logger.Info("Wake On Approach feature is being enabled");
                        this._commClient.Enable(FeatureType.WAKE);
                    }

                    FeatureSetting featureSetting = await this._commClient.GetOptions(FeatureType.WAKE, this.globalMode);
                    UpdateTimerControlSettings(featureSetting);
                }
                catch (Exception ex)
                {
                    this.woaCheckBox.CheckedChanged -= checkBoxWOA_CheckedChanged;
                    this.woaCheckBox.CheckState = CheckState.Unchecked;
                    this.woaCheckBox.CheckedChanged += checkBoxWOA_CheckedChanged;
                    logger.Error("Error enabling Wake On Approach " + ex.Message);
                    MessageBox.Show(ex.Message, "OnError", MessageBoxButtons.OK, MessageBoxIcon.Error);
                    this.woaCheckBox.Enabled = true;
                }
            }
            else
            {
                try
                {
                    if (this.globalMode == SettingsAccessType.GLOBAL)
                    {
                        logger.Info("Setting global disable for Wake On Approach");
                        await this._commClient.SetOption(FeatureType.WAKE, FeatureProperty.FeatureEnabled, false, this.globalMode);
                        FeatureSetting featureSetting = await this._commClient.GetOptions(FeatureType.WAKE, this.globalMode);
                        UpdateTimerControlSettings(featureSetting);
                    }
                    else
                    {
                        logger.Info("Wake On Approach feature is being disabled");
                        this._commClient.Disable(FeatureType.WAKE);
                    }
                }
                catch (Exception ex)
                {
                    this.woaCheckBox.CheckedChanged -= checkBoxWOA_CheckedChanged;
                    this.woaCheckBox.CheckState = CheckState.Checked;
                    this.woaCheckBox.CheckedChanged += checkBoxWOA_CheckedChanged;
                    logger.Error("Error disabling Wake On Approach " + ex.Message);
                    MessageBox.Show(ex.Message, "OnError", MessageBoxButtons.OK, MessageBoxIcon.Error);
                    this.woaCheckBox.Enabled = true;
                }
            }

            Cursor.Current = Cursors.Default;
        }

        internal async void checkBoxNlop_CheckedChanged(object sender, EventArgs e)
        {
            Cursor.Current = Cursors.WaitCursor;
            this.nlopCheckBox.Enabled = false;

            if (this.nlopCheckBox.Checked)
            {
                try
                {
                    if (this.globalMode == SettingsAccessType.GLOBAL)
                    {
                        logger.Info("Showing default settings for No Lock On Presence");
                        await this._commClient.SetOption(FeatureType.NOLOCKONPRESENCE, FeatureProperty.FeatureEnabled, true, this.globalMode);
                    }
                    else
                    {
                        logger.Info("No lock on presence feature is being enabled");
                        this._commClient.Enable(FeatureType.NOLOCKONPRESENCE);
                    }

                    FeatureSetting featureSetting = await this._commClient.GetOptions(FeatureType.NOLOCKONPRESENCE, this.globalMode);
                    UpdateTimerControlSettings(featureSetting);
                }
                catch (Exception ex)
                {
                    this.nlopCheckBox.CheckedChanged -= checkBoxNlop_CheckedChanged;
                    this.nlopCheckBox.CheckState = CheckState.Unchecked;
                    this.nlopCheckBox.CheckedChanged += checkBoxNlop_CheckedChanged;
                    logger.Error("Error enabling No Lock on Presence " + ex.Message);
                    MessageBox.Show(ex.Message, "OnError", MessageBoxButtons.OK, MessageBoxIcon.Error);
                    this.nlopCheckBox.Enabled = true;
                }
            }
            else
            {
                try
                {
                    if (this.globalMode == SettingsAccessType.GLOBAL)
                    {
                        logger.Info("Showing default settings for No Lock On Presence");
                        await this._commClient.SetOption(FeatureType.NOLOCKONPRESENCE, FeatureProperty.FeatureEnabled, false, this.globalMode);
                        FeatureSetting featureSetting = await this._commClient.GetOptions(FeatureType.NOLOCKONPRESENCE, this.globalMode);
                        UpdateTimerControlSettings(featureSetting);
                    }
                    else
                    {
                        logger.Info("No lock on presence feature is being disabled");
                        this._commClient.Disable(FeatureType.NOLOCKONPRESENCE);
                    }
                }
                catch (Exception ex)
                {
                    this.nlopCheckBox.CheckedChanged -= checkBoxNlop_CheckedChanged;
                    this.nlopCheckBox.CheckState = CheckState.Checked;
                    this.nlopCheckBox.CheckedChanged += checkBoxNlop_CheckedChanged;
                    logger.Error("Error disabling No Lock on Presence " + ex.Message);
                    MessageBox.Show(ex.Message, "OnError", MessageBoxButtons.OK, MessageBoxIcon.Error);
                    this.nlopCheckBox.Enabled = true;
                }
            }

            Cursor.Current = Cursors.Default;
        }

        internal async void checkBoxAd_CheckedChanged(object sender, EventArgs e)
        {
            Cursor.Current = Cursors.WaitCursor;
            this.adCheckBox.Enabled = false;

            if (this.adCheckBox.Checked)
            {
                try
                {
                    if (this.globalMode == SettingsAccessType.GLOBAL)
                    {
                        logger.Info("Showing default settings for Adaptive Dimming");
                        await this._commClient.SetOption(FeatureType.ADAPTIVEDIMMING, FeatureProperty.FeatureEnabled, true, this.globalMode);
                    }
                    else
                    {
                        logger.Info("Adaptive dimming feature is being enabled");
                        this._commClient.Enable(FeatureType.ADAPTIVEDIMMING);
                    }

                    FeatureSetting featureSetting = await this._commClient.GetOptions(FeatureType.ADAPTIVEDIMMING, this.globalMode);
                    UpdateTimerControlSettings(featureSetting);
                }
                catch (Exception ex)
                {
                    this.adCheckBox.CheckedChanged -= checkBoxAd_CheckedChanged;
                    this.adCheckBox.CheckState = CheckState.Unchecked;
                    this.adCheckBox.CheckedChanged += checkBoxAd_CheckedChanged;
                    logger.Error("Error enabling Adaptive Dimming " + ex.Message);
                    MessageBox.Show(ex.Message, "OnError", MessageBoxButtons.OK, MessageBoxIcon.Error);
                    this.adCheckBox.Enabled = true;
                }
            }
            else
            {
                try
                {
                    if (this.globalMode == SettingsAccessType.GLOBAL)
                    {
                        logger.Info("Showing default settings for Adaptive Dimming");
                        await this._commClient.SetOption(FeatureType.ADAPTIVEDIMMING, FeatureProperty.FeatureEnabled, false, this.globalMode);
                        FeatureSetting featureSetting = await this._commClient.GetOptions(FeatureType.ADAPTIVEDIMMING, this.globalMode);
                        UpdateTimerControlSettings(featureSetting);
                    }
                    else
                    {
                        logger.Info("Adaptive dimming feature is being disabled");
                        this._commClient.Disable(FeatureType.ADAPTIVEDIMMING);
                    }
                }
                catch (Exception ex)
                {
                    this.adCheckBox.CheckedChanged -= checkBoxAd_CheckedChanged;
                    this.adCheckBox.CheckState = CheckState.Checked;
                    this.adCheckBox.CheckedChanged += checkBoxAd_CheckedChanged;
                    logger.Error("Error disabling Adaptive Dimming " + ex.Message);
                    MessageBox.Show(ex.Message, "OnError", MessageBoxButtons.OK, MessageBoxIcon.Error);
                    this.adCheckBox.Enabled = true;
                }
            }

            Cursor.Current = Cursors.Default;
        }

        internal async void onlookerDetectionCheckBox_CheckedChanged(object sender, EventArgs e)
        {
            Cursor.Current = Cursors.WaitCursor;
            this.onlookerDetectionCheckBox.Enabled = false;

            if (this.onlookerDetectionCheckBox.Checked)
            {
                try
                {
                    if (this.globalMode == SettingsAccessType.GLOBAL)
                    {
                        logger.Info("Showing default settings for Onlooker Detection");
                        await this._commClient.SetOption(FeatureType.ONLOOKERDETECTION, FeatureProperty.FeatureEnabled, true, this.globalMode);
                    }
                    else
                    {
                        logger.Info("Onlooker Detection feature is being enabled");
                        this._commClient.Enable(FeatureType.ONLOOKERDETECTION);
                    }

                    FeatureSetting featureSetting = await this._commClient.GetOptions(FeatureType.ONLOOKERDETECTION, this.globalMode);
                    UpdateTimerControlSettings(featureSetting);
                }
                catch (Exception ex)
                {
                    this.onlookerDetectionCheckBox.CheckedChanged -= onlookerDetectionCheckBox_CheckedChanged;
                    this.onlookerDetectionCheckBox.CheckState = CheckState.Unchecked;
                    this.onlookerDetectionCheckBox.CheckedChanged += onlookerDetectionCheckBox_CheckedChanged;
                    logger.Error("Error enabling onlooker detection " + ex.Message);
                    MessageBox.Show(ex.Message, "OnError", MessageBoxButtons.OK, MessageBoxIcon.Error);
                    this.onlookerDetectionCheckBox.Enabled = true;
                }
            }
            else
            {
                try
                {
                    if (this.globalMode == SettingsAccessType.GLOBAL)
                    {
                        logger.Info("Setting global disable for Onlooker Detection");
                        await this._commClient.SetOption(FeatureType.ONLOOKERDETECTION, FeatureProperty.FeatureEnabled, false, this.globalMode);
                        FeatureSetting featureSetting = await this._commClient.GetOptions(FeatureType.ONLOOKERDETECTION, this.globalMode);
                        UpdateTimerControlSettings(featureSetting);
                    }
                    else
                    {
                        logger.Info("Onlooker Detection feature is being disabled");
                        this._commClient.Disable(FeatureType.ONLOOKERDETECTION);
                    }
                }
                catch (Exception ex)
                {
                    this.onlookerDetectionCheckBox.CheckedChanged -= onlookerDetectionCheckBox_CheckedChanged;
                    this.onlookerDetectionCheckBox.CheckState = CheckState.Checked;
                    this.onlookerDetectionCheckBox.CheckedChanged += onlookerDetectionCheckBox_CheckedChanged;
                    logger.Error("Error disabling Onlooker Detection " + ex.Message);
                    MessageBox.Show(ex.Message, "OnError", MessageBoxButtons.OK, MessageBoxIcon.Error);
                    this.onlookerDetectionCheckBox.Enabled = true;
                }
            }

            Cursor.Current = Cursors.Default;
        }

        private void checkBoxWAL_EnabledChanged(object sender, EventArgs e)
        {
            if (this.globalMode == SettingsAccessType.LOCAL)
            {
                if (this.walCheckBox.Enabled && this.walCheckBox.Checked)
                {
                    UpdateWalControlsVisibility(true);
                }
                else
                {
                    UpdateWalControlsVisibility(false);
                }
            }
        }

        private void checkBoxWOA_EnabledChanged(object sender, EventArgs e)
        {
            if (this.globalMode == SettingsAccessType.LOCAL)
            {
                if (this.woaCheckBox.Enabled && this.woaCheckBox.Checked)
                {
                    UpdateWoaControlsVisibility(true);
                }
                else
                {
                    UpdateWoaControlsVisibility(false);
                }
            }
        }

        private void checkBoxNlop_EnabledChanged(object sender, EventArgs e)
        {
            if (this.globalMode == SettingsAccessType.LOCAL)
            {
                if (this.nlopCheckBox.Enabled && this.nlopCheckBox.Checked)
                {
                    UpdateNLoPControlsVisibility(true);
                }
                else
                {
                    UpdateNLoPControlsVisibility(false);
                }
            }
        }

        private void checkBoxAd_EnabledChanged(object sender, EventArgs e)
        {
            if (this.globalMode == SettingsAccessType.LOCAL)
            {
                if (this.adCheckBox.Enabled && this.adCheckBox.Checked)
                {
                    UpdateAdControlsVisibility(true);
                }
                else
                {
                    UpdateAdControlsVisibility(false);
                }
            }
        }

        private void onlookerDetectionCheckBox_EnabledChanged(object sender, EventArgs e)
        {
            if (this.globalMode == SettingsAccessType.LOCAL)
            {
                if (this.onlookerDetectionCheckBox.Enabled && this.onlookerDetectionCheckBox.Checked)
                {
                    UpdateOnlookerDetectionControlsVisibility(true);
                }
                else
                {
                    UpdateOnlookerDetectionControlsVisibility(false);
                }
            }
        }

        private void UpdateTimerControlSettings(FeatureSetting featureSetting)
        {
            if (featureSetting is LockFeatures)
            {
                this.preDimIntervalComboBox.SelectedIndexChanged -= new System.EventHandler(preDimIntervalComboBox_SelectedIndexChanged);
                this.preDimIntervalComboBox.SelectedIndex = Array.IndexOf(this.preDimIntervalSeconds, ((LockFeatures)featureSetting).PreDimInterval / 1000);
                this.preDimIntervalComboBox.SelectedIndexChanged += new System.EventHandler(preDimIntervalComboBox_SelectedIndexChanged);

                this.notPresentDimmingTimeComboBox.SelectedIndexChanged -= new System.EventHandler(notPresentDimmingTimeComboBox_SelectedIndexChanged);
                this.notPresentDimmingTimeComboBox.SelectedIndex = Array.IndexOf(this.dimmingIntervalSeconds, ((LockFeatures)featureSetting).DimmingInterval / 1000);
                this.notPresentDimmingTimeComboBox.SelectedIndexChanged += new System.EventHandler(notPresentDimmingTimeComboBox_SelectedIndexChanged);

                this.userPresentWaitIntervalComboBox.SelectedIndexChanged -= new System.EventHandler(userPresentWaitIntervalComboBox_SelectedIndexChanged);
                this.userPresentWaitIntervalComboBox.SelectedIndex = Array.IndexOf(this.userPresentWaitIntervalsSeconds, ((LockFeatures)featureSetting).WaitInterval / 1000);
                this.userPresentWaitIntervalComboBox.SelectedIndexChanged += new System.EventHandler(userPresentWaitIntervalComboBox_SelectedIndexChanged);

                this.lockAfterDimIntervalComboBox.SelectedIndexChanged -= new System.EventHandler(lockIntervalComboBox_SelectedIndexChanged);
                this.lockAfterDimIntervalComboBox.SelectedIndex = Array.IndexOf(this.lockAfterDimIntervalsSeconds, ((LockFeatures)featureSetting).LockAfterDimInterval / 1000);
                this.lockAfterDimIntervalComboBox.SelectedIndexChanged += new System.EventHandler(lockIntervalComboBox_SelectedIndexChanged);
            }
            else if (featureSetting is NoLockOnPresenceFeatures)
            {
                this.comboBoxWaitTimer.SelectedIndexChanged -= new System.EventHandler(waitIntervalComboBox_SelectedIndexChanged);
                this.comboBoxWaitTimer.SelectedIndex = Array.IndexOf(this.preDimIntervalSeconds, ((NoLockOnPresenceFeatures)featureSetting).WaitInterval / 1000);
                this.comboBoxWaitTimer.SelectedIndexChanged += new System.EventHandler(waitIntervalComboBox_SelectedIndexChanged);

                this.nlopFailSafeTimeoutComboBox.SelectedIndexChanged -= new System.EventHandler(failSafeTimeoutComboBox_SelectedIndexChanged);
                this.nlopFailSafeTimeoutComboBox.SelectedIndex = Array.IndexOf(this.failSafeIntervalSeconds, ((NoLockOnPresenceFeatures)featureSetting).FailSafeTimeout / 1000);
                this.nlopFailSafeTimeoutComboBox.SelectedIndexChanged += new System.EventHandler(failSafeTimeoutComboBox_SelectedIndexChanged);
            }
            else if (featureSetting is AdaptiveDimmingFeatures)
            {
                this.adNotPresentDimWaitTimeComboBox.SelectedIndexChanged -= new System.EventHandler(adNotPresentDimWaitTimeComboBox_SelectedIndexChanged);
                this.adNotPresentDimWaitTimeComboBox.SelectedIndex = Array.IndexOf(this.preDimIntervalSeconds, ((AdaptiveDimmingFeatures)featureSetting).NotPresentDimWaitTime / 1000);
                this.adNotPresentDimWaitTimeComboBox.SelectedIndexChanged += new System.EventHandler(adNotPresentDimWaitTimeComboBox_SelectedIndexChanged);

                this.mispredictionTimeWindowCombobox.SelectedIndexChanged -= new System.EventHandler(mispredictionTimeWindowCombobox_SelectedIndexChanged);
                this.mispredictionTimeWindowCombobox.SelectedIndex = Array.IndexOf(this.mispredictionTimeWindowSeconds, ((AdaptiveDimmingFeatures)featureSetting).MispredictionTimeWindow / 1000);
                this.mispredictionTimeWindowCombobox.SelectedIndexChanged += new System.EventHandler(mispredictionTimeWindowCombobox_SelectedIndexChanged);

                this.MP1DimWaitComboBox.SelectedIndexChanged -= new System.EventHandler(MP1DimWaitComboBox_SelectedIndexChanged);
                this.MP1DimWaitComboBox.SelectedIndex = Array.IndexOf(this.mispredictionDimWaitSeconds, ((AdaptiveDimmingFeatures)featureSetting).Misprediction1DimWaitTime / 1000);
                this.MP1DimWaitComboBox.SelectedIndexChanged += new System.EventHandler(MP1DimWaitComboBox_SelectedIndexChanged);

                this.MP2DimWaitComboBox.SelectedIndexChanged -= new System.EventHandler(MP2DimWaitComboBox_SelectedIndexChanged);
                this.MP2DimWaitComboBox.SelectedIndex = Array.IndexOf(this.mispredictionDimWaitSeconds, ((AdaptiveDimmingFeatures)featureSetting).Misprediction2DimWaitTime / 1000);
                this.MP2DimWaitComboBox.SelectedIndexChanged += new System.EventHandler(MP2DimWaitComboBox_SelectedIndexChanged);

                this.MP3DimWaitComboBox.SelectedIndexChanged -= new System.EventHandler(MP3DimWaitComboBox_SelectedIndexChanged);
                this.MP3DimWaitComboBox.SelectedIndex = Array.IndexOf(this.mispredictionDimWaitSeconds, ((AdaptiveDimmingFeatures)featureSetting).Misprediction3DimWaitTime / 1000);
                this.MP3DimWaitComboBox.SelectedIndexChanged += new System.EventHandler(MP3DimWaitComboBox_SelectedIndexChanged);

                this.MP4DimWaitComboBox.SelectedIndexChanged -= new System.EventHandler(MP4DimWaitComboBox_SelectedIndexChanged);
                this.MP4DimWaitComboBox.SelectedIndex = Array.IndexOf(this.mispredictionDimWaitSeconds, ((AdaptiveDimmingFeatures)featureSetting).Misprediction4DimWaitTime / 1000);
                this.MP4DimWaitComboBox.SelectedIndexChanged += new System.EventHandler(MP4DimWaitComboBox_SelectedIndexChanged);

                this.adDimmingIntervalComboBox.SelectedIndexChanged -= new System.EventHandler(adDimmingIntervalComboBox_SelectedIndexChanged);
                this.adDimmingIntervalComboBox.SelectedIndex = Array.IndexOf(this.dimmingIntervalSeconds, ((AdaptiveDimmingFeatures)featureSetting).DimmingInterval / 1000);
                this.adDimmingIntervalComboBox.SelectedIndexChanged += new System.EventHandler(adDimmingIntervalComboBox_SelectedIndexChanged);

                this.adDisengagedDimWaitTimeComboBox.SelectedIndexChanged -= new System.EventHandler(adDisengagedDimWaitTimeComboBox_SelectedIndexChanged);
                this.adDisengagedDimWaitTimeComboBox.SelectedIndex = Array.IndexOf(this.preDimIntervalSeconds, ((AdaptiveDimmingFeatures)featureSetting).DisengagedDimWaitTime / 1000);
                this.adDisengagedDimWaitTimeComboBox.SelectedIndexChanged += new System.EventHandler(adDisengagedDimWaitTimeComboBox_SelectedIndexChanged);
            }
            else if (featureSetting is OnlookerDetectionFeatures)
            {
                this.onlookerDetectionDelayIntervalComboBox.SelectedIndexChanged -= new System.EventHandler(onlookerDetectionDelayIntervalComboBox_SelectedIndexChanged);
                this.onlookerDetectionDelayIntervalComboBox.SelectedIndex = Array.IndexOf(this.preDimIntervalSeconds, ((OnlookerDetectionFeatures)featureSetting).DetectionDelayInterval / 1000);
                this.onlookerDetectionDelayIntervalComboBox.SelectedIndexChanged += new System.EventHandler(onlookerDetectionDelayIntervalComboBox_SelectedIndexChanged);

                this.onlookerDetectionDimmingTimeComboBox.SelectedIndexChanged -= new System.EventHandler(onlookerDetectionDimmingTimeComboBox_SelectedIndexChanged);
                this.onlookerDetectionDimmingTimeComboBox.SelectedIndex = Array.IndexOf(this.dimmingIntervalSeconds, ((OnlookerDetectionFeatures)featureSetting).DimmingInterval / 1000);
                this.onlookerDetectionDimmingTimeComboBox.SelectedIndexChanged += new System.EventHandler(onlookerDetectionDimmingTimeComboBox_SelectedIndexChanged);

                this.onlookerDetectionAbsenceIntervalComboBox.SelectedIndexChanged -= new System.EventHandler(onlookerDetectionAbsenceIntervalComboBox_SelectedIndexChanged);
                this.onlookerDetectionAbsenceIntervalComboBox.SelectedIndex = Array.IndexOf(this.detectionAbsenceIntervalSeconds, ((OnlookerDetectionFeatures)featureSetting).OnlookerAbsenceWaitInterval / 1000);
                this.onlookerDetectionAbsenceIntervalComboBox.SelectedIndexChanged += new System.EventHandler(onlookerDetectionAbsenceIntervalComboBox_SelectedIndexChanged);
            }
            else if (featureSetting is AppFeatures)
            {
                this.correlationResetTimeComboBox.SelectedIndexChanged -= new System.EventHandler(correlationResetTimeComboBox_SelectedIndexChanged);
                this.correlationResetTimeComboBox.SelectedIndex = Array.IndexOf(this.correlationResetSeconds, ((AppFeatures)featureSetting).CorrelationResetTime / 1000);
                this.correlationResetTimeComboBox.SelectedIndexChanged += new System.EventHandler(correlationResetTimeComboBox_SelectedIndexChanged);

                this.proximityCooldownComboBox.SelectedIndexChanged -= new System.EventHandler(proximityCooldownComboBox_SelectedIndexChanged);
                this.proximityCooldownComboBox.SelectedIndex = Array.IndexOf(this.proximityCooldownSeconds, ((AppFeatures)featureSetting).ProximityCooldownTime / 1000.0);
                this.proximityCooldownComboBox.SelectedIndexChanged += new System.EventHandler(proximityCooldownComboBox_SelectedIndexChanged);

                this.hidActiveDelayComboBox.SelectedIndexChanged -= new System.EventHandler(hidActiveDelayComboBox_SelectedIndexChanged);
                this.hidActiveDelayComboBox.SelectedIndex = Array.IndexOf(this.hidActiveDelaySeconds, ((AppFeatures)featureSetting).HidActiveDelay / 1000.0);
                this.hidActiveDelayComboBox.SelectedIndexChanged += new System.EventHandler(hidActiveDelayComboBox_SelectedIndexChanged);
            }
        }
        #endregion control event handler

        #region GUI initialization
        private void InitializeControls()
        {
            LoadControlPresets();
            AddToolTipsToControls();
            UpdateAppFeatureControlsVisibility(this.testModeCheckbox.Checked); // Make sure that the default visibility matches the default state of the test mode checkbox
        }

        private void AddToolTipsToControls()
        {
            string awayString = "Have PC lock and turn off display when you move away from your PC";
            string awayExternalMonitorString = "Keep WAL enabled while connected to external monitor(s)";
            string preDimIntervalString = "Time to dim after you leave";
            string lockAfterDimString = "Time to lock after dim";
            string nearString = "Have PC wake as you approach";
            string nearExternalMonitorString = "Keep WOA enabled while connected to external monitor(s)";
            string proximityDistanceString = "Real Time Proximity Sensor reading (in mm)";
            using (var toolTip = new ToolTip())
            {
                toolTip.AutoPopDelay = 5000;
                toolTip.ReshowDelay = 500;
                toolTip.ShowAlways = true;
                toolTip.SetToolTip(this.walLabel, awayString);
                toolTip.SetToolTip(this.walCheckBox, awayString);
                toolTip.SetToolTip(this.walPanel, awayString);
                toolTip.SetToolTip(this.walExternalMonitorLabel, awayExternalMonitorString);
                toolTip.SetToolTip(this.walExternalMonitorOnRadioButton, awayExternalMonitorString);
                toolTip.SetToolTip(this.walExternalMonitorOffRadioButton, awayExternalMonitorString);
                toolTip.SetToolTip(this.preDimIntervalLabel, preDimIntervalString);
                toolTip.SetToolTip(this.preDimIntervalComboBox, preDimIntervalString);
                toolTip.SetToolTip(this.lockAfterDimIntervalLabel, lockAfterDimString);
                toolTip.SetToolTip(this.lockAfterDimIntervalComboBox, lockAfterDimString);
                toolTip.SetToolTip(this.woaLabel, nearString);
                toolTip.SetToolTip(this.woaCheckBox, nearString);
                toolTip.SetToolTip(this.woaPanel, nearString);
                toolTip.SetToolTip(this.panelWoa, nearExternalMonitorString);
                toolTip.SetToolTip(this.woaExternalMonitorLabel, nearExternalMonitorString);
                toolTip.SetToolTip(this.woaExternalMonitorOnRadioButton, nearExternalMonitorString);
                toolTip.SetToolTip(this.woaExternalMonitorOffRadioButton, nearExternalMonitorString);
                toolTip.SetToolTip(this.proximityDistanceLabel, proximityDistanceString);
                toolTip.SetToolTip(this.proximityDistancePanel, proximityDistanceString);
                toolTip.SetToolTip(this.proximityDistanceValueLabel, proximityDistanceString);
            }
        }

        private void AddControlEventHandlers()
        {
            /* Do not remove event handler assignments! At this point, the event handlers for the controls have not been added. */

            this.walCheckBox.CheckedChanged += new System.EventHandler(checkBoxWAL_CheckedChanged);
            this.walExternalMonitorOnRadioButton.CheckedChanged += new System.EventHandler(awayExternalMonitorOnRadioButton_CheckedChanged);
            this.walExternalMonitorOffRadioButton.CheckedChanged += new System.EventHandler(awayExternalMonitorOffRadioButton_CheckedChanged);
            this.preDimIntervalComboBox.SelectedIndexChanged += new System.EventHandler(preDimIntervalComboBox_SelectedIndexChanged);
            this.notPresentDimmingTimeComboBox.SelectedIndexChanged += new System.EventHandler(notPresentDimmingTimeComboBox_SelectedIndexChanged);
            this.notPresentDimTargetComboBox.SelectedIndexChanged += new System.EventHandler(notPresentDimTargetComboBox_SelectedIndexChanged);
            this.userPresentWaitIntervalComboBox.SelectedIndexChanged += new System.EventHandler(userPresentWaitIntervalComboBox_SelectedIndexChanged);
            this.dimScreenOnRadioButton.CheckedChanged += new System.EventHandler(dimScreenOnRadioButton_CheckedChanged);
            this.dimScreenOffRadioButton.CheckedChanged += new System.EventHandler(dimScreenOffRadioButton_CheckedChanged);
            this.honorPowerRequestsOnRadioButton.CheckedChanged += new System.EventHandler(honorPowerRequestsOnRadioButton_CheckedChanged);
            this.honorPowerRequestsOffRadioButton.CheckedChanged += new System.EventHandler(honorPowerRequestsOffRadioButton_CheckedChanged);
            this.honorUserInCallOnRadioButton.CheckedChanged += new System.EventHandler(honorUserInCallOnRadioButton_CheckedChanged);
            this.honorUserInCallOffRadioButton.CheckedChanged += new System.EventHandler(honorUserInCallOffRadioButton_CheckedChanged);
            this.lockAfterDimIntervalComboBox.SelectedIndexChanged += new System.EventHandler(lockIntervalComboBox_SelectedIndexChanged);
            this.displayOffOnRadioButton.CheckedChanged += new System.EventHandler(displayOffOnRadioButton_CheckedChanged);
            this.displayOffOffRadioButton.CheckedChanged += new System.EventHandler(displayOffOffRadioButton_CheckedChanged);

            this.woaCheckBox.CheckedChanged += new System.EventHandler(checkBoxWOA_CheckedChanged);
            this.woaExternalMonitorOffRadioButton.CheckedChanged += new System.EventHandler(nearExternalMonitorOffRadioButton_CheckedChanged);
            this.woaExternalMonitorOnRadioButton.CheckedChanged += new System.EventHandler(nearExternalMonitorOnRadioButton_CheckedChanged);
            this.woaEnableOnBatteryTrueRadioButton.CheckedChanged += new System.EventHandler(woaEnableOnBatteryTrue_radio_CheckedChanged);
            this.woaEnableOnBatteryFalseRadioButton.CheckedChanged += new System.EventHandler(woaEnableOnBatteryFalse_radio_CheckedChanged);
            this.woaBatteryRemainingComboBox.SelectedIndexChanged += new System.EventHandler(woaBatteryComboBox_SelectedIndexChanged);

            this.nlopCheckBox.CheckedChanged += new System.EventHandler(checkBoxNlop_CheckedChanged);
            this.nlopExternalMonitorOffRadioButton.CheckedChanged += new System.EventHandler(nlopExternalMonitorOffRadioButton_CheckedChanged);
            this.nlopExternalMonitorOnRadioButton.CheckedChanged += new System.EventHandler(nlopExternalMonitorOnRadioButton_CheckedChanged);
            this.nlopEnableOnBatteryTrueRadioButton.CheckedChanged += new System.EventHandler(nlopEnableOnBatteryTrue_radio_CheckedChanged);
            this.nlopEnableOnBatteryFalseRadioButton.CheckedChanged += new System.EventHandler(nlopEnableOnBatteryFalse_radio_CheckedChanged);
            this.nlopBatteryRemainingComboBox.SelectedIndexChanged += new System.EventHandler(nlopBatteryComboBox_SelectedIndexChanged);
            this.comboBoxWaitTimer.SelectedIndexChanged += new System.EventHandler(waitIntervalComboBox_SelectedIndexChanged);
            this.nlopFailSafeTimeoutComboBox.SelectedIndexChanged += new System.EventHandler(failSafeTimeoutComboBox_SelectedIndexChanged);

            this.adCheckBox.CheckedChanged += new System.EventHandler(checkBoxAd_CheckedChanged);
            this.adNotPresentDimWaitTimeComboBox.SelectedIndexChanged += new System.EventHandler(adNotPresentDimWaitTimeComboBox_SelectedIndexChanged);
            this.adExternalMonitorOffRadioButton.CheckedChanged += new System.EventHandler(adExternalMonitorOffRadioButton_CheckedChanged);
            this.adExternalMonitorOnRadioButton.CheckedChanged += new System.EventHandler(adExternalMonitorOnRadioButton_CheckedChanged);
            this.adHonorPowerRequestOnRadioButton.CheckedChanged += new System.EventHandler(adHonorPowerRequestOnRadioButton_CheckedChanged);
            this.adHonorPowerRequestOffRadioButton.CheckedChanged += new System.EventHandler(adHonorPowerRequestOffRadioButton_CheckedChanged);
            this.ignoreHidInputRadioOn.CheckedChanged += new System.EventHandler(ignoreHidInputRadioOn_CheckedChanged);
            this.ignoreHidInputRadioOff.CheckedChanged += new System.EventHandler(ignoreHidInputRadioOff_CheckedChanged);
            this.ignoreHidOnExternalMonitorCheckBox.CheckedChanged += new System.EventHandler(ignoreHidOnExternalMonitorCheckBox_CheckedChanged);
            this.honorUserInCallADRadioButtonOn.CheckedChanged += new System.EventHandler(honorUserInCallADRadioButtonOn_CheckedChanged);
            this.honorUserInCallADRadioButtonOff.CheckedChanged += new System.EventHandler(honorUserInCallADRadioButtonOff_CheckedChanged);
            this.zeroRefreshRateEnableRadioButton.CheckedChanged += new System.EventHandler(zeroRefreshRateEnableRadioButton_CheckedChanged);
            this.zeroRefreshRateDisableRadioButton.CheckedChanged += new System.EventHandler(zeroRefreshRateDisableRadioButton_CheckedChanged);
            this.lowRefreshRateEnableRadioButton.CheckedChanged += new System.EventHandler(lowRefreshRateEnableRadioButton_CheckedChanged);
            this.lowRefreshRateDisableRadioButton.CheckedChanged += new System.EventHandler(lowRefreshRateDisableRadioButton_CheckedChanged);
            this.mispredictionWithFaceDetectionCheckbox.CheckedChanged += new System.EventHandler(mispredictionWithFaceDetectionCheckbox_CheckedChanged);
            this.mispredictionTimeWindowCombobox.SelectedIndexChanged += new System.EventHandler(mispredictionTimeWindowCombobox_SelectedIndexChanged);
            this.MP1DimWaitComboBox.SelectedIndexChanged += new System.EventHandler(MP1DimWaitComboBox_SelectedIndexChanged);
            this.MP2DimWaitComboBox.SelectedIndexChanged += new System.EventHandler(MP2DimWaitComboBox_SelectedIndexChanged);
            this.MP3DimWaitComboBox.SelectedIndexChanged += new System.EventHandler(MP3DimWaitComboBox_SelectedIndexChanged);
            this.MP4DimWaitComboBox.SelectedIndexChanged += new System.EventHandler(MP4DimWaitComboBox_SelectedIndexChanged);
            this.adDisengagedDimTargetComboBox.SelectedIndexChanged += new System.EventHandler(adDisengagedDimTargetComboBox_SelectedIndexChanged);
            this.adNotPresentDimTargetComboBox.SelectedIndexChanged += new System.EventHandler(adNotPresentDimTargetComboBox_SelectedIndexChanged);
            this.adDimmingIntervalComboBox.SelectedIndexChanged += new System.EventHandler(adDimmingIntervalComboBox_SelectedIndexChanged);
            this.adDisengagedDimWaitTimeComboBox.SelectedIndexChanged += new System.EventHandler(adDisengagedDimWaitTimeComboBox_SelectedIndexChanged);
            this.touchPauseScanEnableRadioButton.CheckedChanged += new System.EventHandler(touchPauseScanEnableRadioButton_CheckedChanged);
            this.touchPauseScanDisableRadioButton.CheckedChanged += new System.EventHandler(touchPauseScanDisableRadioButton_CheckedChanged);

            this.onlookerDetectionCheckBox.CheckedChanged += new System.EventHandler(onlookerDetectionCheckBox_CheckedChanged);
            this.onlookerDetectionExternalMonitorOnRadioButton.CheckedChanged += new System.EventHandler(onlookerDetectionExternalMonitorOnRadioButton_CheckedChanged);
            this.onlookerDetectionExternalMonitorOffRadioButton.CheckedChanged += new System.EventHandler(onlookerDetectionExternalMonitorOffRadioButton_CheckedChanged);
            this.onlookerDetectionDelayIntervalComboBox.SelectedIndexChanged += new System.EventHandler(onlookerDetectionDelayIntervalComboBox_SelectedIndexChanged);
            this.onlookerDetectionDimWhenDetectedOnRadioButton.CheckedChanged += new System.EventHandler(onlookerDetectionDimWhenDetectedOnRadioButton_CheckedChanged);
            this.onlookerDetectionDimWhenDetectedOffRadioButton.CheckedChanged += new System.EventHandler(onlookerDetectionDimWhenDetectedOffRadioButton_CheckedChanged);
            this.onlookerDetectionDimmingTimeComboBox.SelectedIndexChanged += new System.EventHandler(onlookerDetectionDimmingTimeComboBox_SelectedIndexChanged);
            this.onlookerDetectionDimTargetComboBox.SelectedIndexChanged += new System.EventHandler(onlookerDetectionDimTargetComboBox_SelectedIndexChanged);
            this.onlookerDetectionEnableOnLowBatteryOnRadioButton.CheckedChanged += new System.EventHandler(onlookerDetectionEnableOnLowBatteryOnRadioButton_CheckedChanged);
            this.onlookerDetectionEnableOnLowBatteryOffRadioButton.CheckedChanged += new System.EventHandler(onlookerDetectionEnableOnLowBatteryOffRadioButton_CheckedChanged);
            this.onlookerDetectionLowBatteryLimitComboBox.SelectedIndexChanged += new System.EventHandler(onlookerDetectionLowBatteryLimitComboBox_SelectedIndexChanged);
            this.onlookerDetectionAbsenceIntervalComboBox.SelectedIndexChanged += new System.EventHandler(onlookerDetectionAbsenceIntervalComboBox_SelectedIndexChanged);
            this.onlookerDetectionUseFaceAttentionOnRadioButton.CheckedChanged += new System.EventHandler(onlookerDetectionUseFaceAttentionOnRadioButton_CheckedChanged);
            this.onlookerDetectionUseFaceAttentionOffRadioButton.CheckedChanged += new System.EventHandler(onlookerDetectionUseFaceAttentionOffRadioButton_CheckedChanged);

            this.privacyCheckbox.CheckedChanged += new System.EventHandler(privacyCheckBox_CheckedChanged);
            this.appNameComboBox.SelectedIndexChanged += new System.EventHandler(appNameComboBox_SelectedIndexChanged);
            this.appNameComboBox.SelectedIndexChanged += new System.EventHandler(appNameComboBox_SelectedIndexChanged);
            this.monitorAppCheckbox.CheckedChanged += new System.EventHandler(monitorAppCheckbox_CheckedChanged);
            this.enabledEventsListBox.SelectedIndexChanged += new System.EventHandler(enabledEventsListBox_SelectedIndexChanged);
            this.correlationResetTimeComboBox.SelectedIndexChanged += new System.EventHandler(correlationResetTimeComboBox_SelectedIndexChanged);
            this.proximityCooldownComboBox.SelectedIndexChanged += new System.EventHandler(proximityCooldownComboBox_SelectedIndexChanged);
            this.hidActiveDelayComboBox.SelectedIndexChanged += new System.EventHandler(hidActiveDelayComboBox_SelectedIndexChanged);

            this.hupLrrOffRadioButton.CheckedChanged += new System.EventHandler(hupLrrOffRadioButton_CheckedChanged);
            this.hupLrrOnRadioButton.CheckedChanged += new System.EventHandler(hupLrrOnRadioButton_CheckedChanged);
            this.hupZrrOffRadioButton.CheckedChanged += new System.EventHandler(hupZrrOffRadioButton_CheckedChanged);
            this.hupZrrOnRadioButton.CheckedChanged += new System.EventHandler(hupZrrOnRadioButton_CheckedChanged);
            this.hupIgnoreUpNoRadioButton.CheckedChanged += new System.EventHandler(hupIgnoreUpNoRadioButton_CheckedChanged);
            this.hupIgnoreUpYesRadioButton.CheckedChanged += new System.EventHandler(hupIgnoreUpYesRadioButton_CheckedChanged);
            this.hupTpsOffRadioButton.CheckedChanged += new System.EventHandler(hupTpsOffRadioButton_CheckedChanged);
            this.hupTpsOnRadioButton.CheckedChanged += new System.EventHandler(hupTpsOnRadioButton_CheckedChanged);
            this.hupSetOsDimLevelOffRadioButton.CheckedChanged += new System.EventHandler(hupSetOsDimLevelOffRadioButton_CheckedChanged);
            this.hupSetOsDimLevelOnRadioButton.CheckedChanged += new System.EventHandler(hupSetOsDimLevelOnRadioButton_CheckedChanged);
        }

        private async void HandleSetOptionExceptions(Task setTask)
        {
            try
            {
                await setTask;
            }
            catch (ErrorException ee)
            {
                logger.Debug(ee.ErrorObject.Description);
                MessageBox.Show(ee.ErrorObject.Description, ee.ErrorType.ToString(), MessageBoxButtons.OK, MessageBoxIcon.Error);
            }
            catch (AggregateException ae)
            {
                foreach (Exception ie in ae.InnerExceptions)
                {
                    var ee = (ErrorException)ie;
                    logger.Debug(ee.ErrorObject.Description);
                    MessageBox.Show(ee.ErrorObject.Description, ee.ErrorType.ToString(), MessageBoxButtons.OK, MessageBoxIcon.Error);
                }
            }
            catch (Exception ex)
            {
                logger.Debug(ex.Message);
            }
        }

        private void RemoveControlEventHandlers()
        {
            this.walCheckBox.CheckedChanged -= new System.EventHandler(checkBoxWAL_CheckedChanged);
            this.walExternalMonitorOnRadioButton.CheckedChanged -= new System.EventHandler(awayExternalMonitorOnRadioButton_CheckedChanged);
            this.walExternalMonitorOffRadioButton.CheckedChanged -= new System.EventHandler(awayExternalMonitorOffRadioButton_CheckedChanged);
            this.preDimIntervalComboBox.SelectedIndexChanged -= new System.EventHandler(preDimIntervalComboBox_SelectedIndexChanged);
            this.notPresentDimmingTimeComboBox.SelectedIndexChanged -= new System.EventHandler(notPresentDimmingTimeComboBox_SelectedIndexChanged);
            this.notPresentDimTargetComboBox.SelectedIndexChanged -= new System.EventHandler(notPresentDimTargetComboBox_SelectedIndexChanged);
            this.userPresentWaitIntervalComboBox.SelectedIndexChanged -= new System.EventHandler(userPresentWaitIntervalComboBox_SelectedIndexChanged);
            this.dimScreenOnRadioButton.CheckedChanged -= new System.EventHandler(dimScreenOnRadioButton_CheckedChanged);
            this.dimScreenOffRadioButton.CheckedChanged -= new System.EventHandler(dimScreenOffRadioButton_CheckedChanged);
            this.honorPowerRequestsOnRadioButton.CheckedChanged -= new System.EventHandler(honorPowerRequestsOnRadioButton_CheckedChanged);
            this.honorPowerRequestsOffRadioButton.CheckedChanged -= new System.EventHandler(honorPowerRequestsOffRadioButton_CheckedChanged);
            this.honorUserInCallOnRadioButton.CheckedChanged -= new System.EventHandler(honorUserInCallOnRadioButton_CheckedChanged);
            this.honorUserInCallOffRadioButton.CheckedChanged -= new System.EventHandler(honorUserInCallOffRadioButton_CheckedChanged);
            this.lockAfterDimIntervalComboBox.SelectedIndexChanged -= new System.EventHandler(lockIntervalComboBox_SelectedIndexChanged);
            this.displayOffOnRadioButton.CheckedChanged -= new System.EventHandler(displayOffOnRadioButton_CheckedChanged);
            this.displayOffOffRadioButton.CheckedChanged -= new System.EventHandler(displayOffOffRadioButton_CheckedChanged);

            this.woaCheckBox.CheckedChanged -= new System.EventHandler(checkBoxWOA_CheckedChanged);
            this.woaExternalMonitorOffRadioButton.CheckedChanged -= new System.EventHandler(nearExternalMonitorOffRadioButton_CheckedChanged);
            this.woaExternalMonitorOnRadioButton.CheckedChanged -= new System.EventHandler(nearExternalMonitorOnRadioButton_CheckedChanged);
            this.woaEnableOnBatteryTrueRadioButton.CheckedChanged -= new System.EventHandler(woaEnableOnBatteryTrue_radio_CheckedChanged);
            this.woaEnableOnBatteryFalseRadioButton.CheckedChanged -= new System.EventHandler(woaEnableOnBatteryFalse_radio_CheckedChanged);
            this.woaBatteryRemainingComboBox.SelectedIndexChanged -= new System.EventHandler(woaBatteryComboBox_SelectedIndexChanged);

            this.nlopCheckBox.CheckedChanged -= new System.EventHandler(checkBoxNlop_CheckedChanged);
            this.nlopExternalMonitorOffRadioButton.CheckedChanged -= new System.EventHandler(nlopExternalMonitorOffRadioButton_CheckedChanged);
            this.nlopExternalMonitorOnRadioButton.CheckedChanged -= new System.EventHandler(nlopExternalMonitorOnRadioButton_CheckedChanged);
            this.nlopEnableOnBatteryTrueRadioButton.CheckedChanged -= new System.EventHandler(nlopEnableOnBatteryTrue_radio_CheckedChanged);
            this.nlopEnableOnBatteryFalseRadioButton.CheckedChanged -= new System.EventHandler(nlopEnableOnBatteryFalse_radio_CheckedChanged);
            this.nlopBatteryRemainingComboBox.SelectedIndexChanged -= new System.EventHandler(nlopBatteryComboBox_SelectedIndexChanged);
            this.comboBoxWaitTimer.SelectedIndexChanged -= new System.EventHandler(waitIntervalComboBox_SelectedIndexChanged);
            this.nlopFailSafeTimeoutComboBox.SelectedIndexChanged -= new System.EventHandler(failSafeTimeoutComboBox_SelectedIndexChanged);

            this.adCheckBox.CheckedChanged -= new System.EventHandler(checkBoxAd_CheckedChanged);
            this.adNotPresentDimWaitTimeComboBox.SelectedIndexChanged -= new System.EventHandler(adNotPresentDimWaitTimeComboBox_SelectedIndexChanged);
            this.adExternalMonitorOffRadioButton.CheckedChanged -= new System.EventHandler(adExternalMonitorOffRadioButton_CheckedChanged);
            this.adExternalMonitorOnRadioButton.CheckedChanged -= new System.EventHandler(adExternalMonitorOnRadioButton_CheckedChanged);
            this.adHonorPowerRequestOnRadioButton.CheckedChanged -= new System.EventHandler(adHonorPowerRequestOnRadioButton_CheckedChanged);
            this.adHonorPowerRequestOffRadioButton.CheckedChanged -= new System.EventHandler(adHonorPowerRequestOffRadioButton_CheckedChanged);
            this.ignoreHidInputRadioOn.CheckedChanged -= new System.EventHandler(ignoreHidInputRadioOn_CheckedChanged);
            this.ignoreHidInputRadioOff.CheckedChanged -= new System.EventHandler(ignoreHidInputRadioOff_CheckedChanged);
            this.ignoreHidOnExternalMonitorCheckBox.CheckedChanged -= new System.EventHandler(ignoreHidOnExternalMonitorCheckBox_CheckedChanged);
            this.honorUserInCallADRadioButtonOn.CheckedChanged -= new System.EventHandler(honorUserInCallADRadioButtonOn_CheckedChanged);
            this.honorUserInCallADRadioButtonOff.CheckedChanged -= new System.EventHandler(honorUserInCallADRadioButtonOff_CheckedChanged);
            this.zeroRefreshRateEnableRadioButton.CheckedChanged -= new System.EventHandler(zeroRefreshRateEnableRadioButton_CheckedChanged);
            this.zeroRefreshRateDisableRadioButton.CheckedChanged -= new System.EventHandler(zeroRefreshRateDisableRadioButton_CheckedChanged);
            this.lowRefreshRateEnableRadioButton.CheckedChanged -= new System.EventHandler(lowRefreshRateEnableRadioButton_CheckedChanged);
            this.lowRefreshRateDisableRadioButton.CheckedChanged -= new System.EventHandler(lowRefreshRateDisableRadioButton_CheckedChanged);
            this.mispredictionWithFaceDetectionCheckbox.CheckedChanged -= new System.EventHandler(mispredictionWithFaceDetectionCheckbox_CheckedChanged);
            this.mispredictionTimeWindowCombobox.SelectedIndexChanged -= new System.EventHandler(mispredictionTimeWindowCombobox_SelectedIndexChanged);
            this.MP1DimWaitComboBox.SelectedIndexChanged -= new System.EventHandler(MP1DimWaitComboBox_SelectedIndexChanged);
            this.MP2DimWaitComboBox.SelectedIndexChanged -= new System.EventHandler(MP2DimWaitComboBox_SelectedIndexChanged);
            this.MP3DimWaitComboBox.SelectedIndexChanged -= new System.EventHandler(MP3DimWaitComboBox_SelectedIndexChanged);
            this.MP4DimWaitComboBox.SelectedIndexChanged -= new System.EventHandler(MP4DimWaitComboBox_SelectedIndexChanged);
            this.adDisengagedDimTargetComboBox.SelectedIndexChanged -= new System.EventHandler(adDisengagedDimTargetComboBox_SelectedIndexChanged);
            this.adNotPresentDimTargetComboBox.SelectedIndexChanged -= new System.EventHandler(adNotPresentDimTargetComboBox_SelectedIndexChanged);
            this.adDimmingIntervalComboBox.SelectedIndexChanged -= new System.EventHandler(adDimmingIntervalComboBox_SelectedIndexChanged);
            this.adDisengagedDimWaitTimeComboBox.SelectedIndexChanged -= new System.EventHandler(adDisengagedDimWaitTimeComboBox_SelectedIndexChanged);
            this.touchPauseScanEnableRadioButton.CheckedChanged -= new System.EventHandler(touchPauseScanEnableRadioButton_CheckedChanged);
            this.touchPauseScanDisableRadioButton.CheckedChanged -= new System.EventHandler(touchPauseScanDisableRadioButton_CheckedChanged);

            this.onlookerDetectionCheckBox.CheckedChanged -= new System.EventHandler(onlookerDetectionCheckBox_CheckedChanged);
            this.onlookerDetectionExternalMonitorOnRadioButton.CheckedChanged -= new System.EventHandler(onlookerDetectionExternalMonitorOnRadioButton_CheckedChanged);
            this.onlookerDetectionExternalMonitorOffRadioButton.CheckedChanged -= new System.EventHandler(onlookerDetectionExternalMonitorOffRadioButton_CheckedChanged);
            this.onlookerDetectionDelayIntervalComboBox.SelectedIndexChanged -= new System.EventHandler(onlookerDetectionDelayIntervalComboBox_SelectedIndexChanged);
            this.onlookerDetectionDimWhenDetectedOnRadioButton.CheckedChanged -= new System.EventHandler(onlookerDetectionDimWhenDetectedOnRadioButton_CheckedChanged);
            this.onlookerDetectionDimWhenDetectedOffRadioButton.CheckedChanged -= new System.EventHandler(onlookerDetectionDimWhenDetectedOffRadioButton_CheckedChanged);
            this.onlookerDetectionDimmingTimeComboBox.SelectedIndexChanged -= new System.EventHandler(onlookerDetectionDimmingTimeComboBox_SelectedIndexChanged);
            this.onlookerDetectionDimTargetComboBox.SelectedIndexChanged -= new System.EventHandler(onlookerDetectionDimTargetComboBox_SelectedIndexChanged);
            this.onlookerDetectionEnableOnLowBatteryOnRadioButton.CheckedChanged -= new System.EventHandler(onlookerDetectionEnableOnLowBatteryOnRadioButton_CheckedChanged);
            this.onlookerDetectionEnableOnLowBatteryOffRadioButton.CheckedChanged -= new System.EventHandler(onlookerDetectionEnableOnLowBatteryOffRadioButton_CheckedChanged);
            this.onlookerDetectionLowBatteryLimitComboBox.SelectedIndexChanged -= new System.EventHandler(onlookerDetectionLowBatteryLimitComboBox_SelectedIndexChanged);
            this.onlookerDetectionAbsenceIntervalComboBox.SelectedIndexChanged -= new System.EventHandler(onlookerDetectionAbsenceIntervalComboBox_SelectedIndexChanged);
            this.onlookerDetectionUseFaceAttentionOnRadioButton.CheckedChanged -= new System.EventHandler(onlookerDetectionUseFaceAttentionOnRadioButton_CheckedChanged);
            this.onlookerDetectionUseFaceAttentionOffRadioButton.CheckedChanged -= new System.EventHandler(onlookerDetectionUseFaceAttentionOffRadioButton_CheckedChanged);

            this.privacyCheckbox.CheckedChanged -= new System.EventHandler(privacyCheckBox_CheckedChanged);
            this.appNameComboBox.SelectedIndexChanged -= new System.EventHandler(appNameComboBox_SelectedIndexChanged);
            this.monitorAppCheckbox.CheckedChanged -= new System.EventHandler(monitorAppCheckbox_CheckedChanged);
            this.enabledEventsListBox.SelectedIndexChanged -= new System.EventHandler(enabledEventsListBox_SelectedIndexChanged);
            this.correlationResetTimeComboBox.SelectedIndexChanged -= new System.EventHandler(correlationResetTimeComboBox_SelectedIndexChanged);
            this.proximityCooldownComboBox.SelectedIndexChanged -= new System.EventHandler(proximityCooldownComboBox_SelectedIndexChanged);
            this.hidActiveDelayComboBox.SelectedIndexChanged -= new System.EventHandler(hidActiveDelayComboBox_SelectedIndexChanged);

            this.hupLrrOffRadioButton.CheckedChanged -= new System.EventHandler(hupLrrOffRadioButton_CheckedChanged);
            this.hupLrrOnRadioButton.CheckedChanged -= new System.EventHandler(hupLrrOnRadioButton_CheckedChanged);
            this.hupZrrOffRadioButton.CheckedChanged -= new System.EventHandler(hupZrrOffRadioButton_CheckedChanged);
            this.hupZrrOnRadioButton.CheckedChanged -= new System.EventHandler(hupZrrOnRadioButton_CheckedChanged);
            this.hupTpsOffRadioButton.CheckedChanged -= new System.EventHandler(hupTpsOffRadioButton_CheckedChanged);
            this.hupTpsOnRadioButton.CheckedChanged -= new System.EventHandler(hupTpsOnRadioButton_CheckedChanged);
            this.hupIgnoreUpNoRadioButton.CheckedChanged -= new System.EventHandler(hupIgnoreUpNoRadioButton_CheckedChanged);
            this.hupIgnoreUpYesRadioButton.CheckedChanged -= new System.EventHandler(hupIgnoreUpYesRadioButton_CheckedChanged);
            this.hupSetOsDimLevelOffRadioButton.CheckedChanged -= new System.EventHandler(hupSetOsDimLevelOffRadioButton_CheckedChanged);
            this.hupSetOsDimLevelOnRadioButton.CheckedChanged -= new System.EventHandler(hupSetOsDimLevelOnRadioButton_CheckedChanged);
        }

        private void LoadControlPresets()
        {
            this.preDimIntervalComboBox.Items.AddRange(Array.ConvertAll(this.preDimIntervalSeconds, x => x.ToString() + " Seconds"));
            this.notPresentDimmingTimeComboBox.Items.AddRange(Array.ConvertAll(this.dimmingIntervalSeconds, x => x.ToString() + " Seconds"));
            this.notPresentDimTargetComboBox.Items.AddRange(Array.ConvertAll(this.dimTarget, x => x.ToString() + " %"));
            this.userPresentWaitIntervalComboBox.Items.AddRange(Array.ConvertAll(this.userPresentWaitIntervalsSeconds, x => x.ToString() + " Seconds"));
            this.lockAfterDimIntervalComboBox.Items.AddRange(Array.ConvertAll(this.lockAfterDimIntervalsSeconds, x => x.ToString() + " Seconds"));

            this.woaBatteryRemainingComboBox.Items.AddRange(Array.ConvertAll(this.batteryRemainingPercentage, x => x.ToString() + " %"));

            this.nlopBatteryRemainingComboBox.Items.AddRange(Array.ConvertAll(this.batteryRemainingPercentage, x => x.ToString() + " %"));
            this.comboBoxWaitTimer.Items.AddRange(Array.ConvertAll(this.preDimIntervalSeconds, x => x.ToString() + " Seconds"));
            this.nlopFailSafeTimeoutComboBox.Items.AddRange(Array.ConvertAll(this.failSafeIntervalSeconds, x => x.ToString() + " Seconds"));

            this.adNotPresentDimWaitTimeComboBox.Items.AddRange(Array.ConvertAll(this.preDimIntervalSeconds, x => x.ToString() + " Seconds"));
            this.MP1DimWaitComboBox.Items.AddRange(Array.ConvertAll(this.mispredictionDimWaitSeconds, x => x.ToString() + " Seconds"));
            this.MP2DimWaitComboBox.Items.AddRange(Array.ConvertAll(this.mispredictionDimWaitSeconds, x => x.ToString() + " Seconds"));
            this.MP3DimWaitComboBox.Items.AddRange(Array.ConvertAll(this.mispredictionDimWaitSeconds, x => x.ToString() + " Seconds"));
            this.MP4DimWaitComboBox.Items.AddRange(Array.ConvertAll(this.mispredictionDimWaitSeconds, x => x.ToString() + " Seconds"));
            this.mispredictionTimeWindowCombobox.Items.AddRange(Array.ConvertAll(this.mispredictionTimeWindowSeconds, x => x.ToString() + " Seconds"));
            this.adDisengagedDimTargetComboBox.Items.AddRange(Array.ConvertAll(this.dimTarget, x => x.ToString() + " %"));
            this.adNotPresentDimTargetComboBox.Items.AddRange(Array.ConvertAll(this.dimTarget, x => x.ToString() + " %"));
            this.adDimmingIntervalComboBox.Items.AddRange(Array.ConvertAll(this.dimmingIntervalSeconds, x => x.ToString() + " Seconds"));
            this.adDisengagedDimWaitTimeComboBox.Items.AddRange(Array.ConvertAll(this.preDimIntervalSeconds, x => x.ToString() + " Seconds"));

            this.onlookerDetectionDelayIntervalComboBox.Items.AddRange(Array.ConvertAll(this.preDimIntervalSeconds, x => x.ToString() + " Seconds"));
            this.onlookerDetectionDimmingTimeComboBox.Items.AddRange(Array.ConvertAll(this.dimmingIntervalSeconds, x => x.ToString() + " Seconds"));
            this.onlookerDetectionDimTargetComboBox.Items.AddRange(Array.ConvertAll(this.dimTarget, x => x.ToString() + " %"));
            this.onlookerDetectionLowBatteryLimitComboBox.Items.AddRange(Array.ConvertAll(this.batteryRemainingPercentage, x => x.ToString() + " %"));
            this.onlookerDetectionAbsenceIntervalComboBox.Items.AddRange(Array.ConvertAll(this.detectionAbsenceIntervalSeconds, x => x.ToString() + " Seconds"));

            this.appNameComboBox.Items.AddRange(this.appNames);
            this.enabledEventsListBox.Items.AddRange(this.eventNames);
            this.correlationResetTimeComboBox.Items.AddRange(Array.ConvertAll(this.correlationResetSeconds, x => x.ToString() + " Seconds"));
            this.proximityCooldownComboBox.Items.AddRange(Array.ConvertAll(this.proximityCooldownSeconds, x => x.ToString() + " Seconds"));
            this.hidActiveDelayComboBox.Items.AddRange(Array.ConvertAll(this.hidActiveDelaySeconds, x => x.ToString() + " Seconds"));
        }

        internal void MakeIntervalControlsVisible(bool visible)
        {
            //labelMinLockInterval.Visible = visible;
            //todo:add new controls
        }
        #endregion GUI initialization

        #region helpers
        internal void StartHumanPresenceUpdateTask()
        {
            if (this.humanPresenceBackgroundTask.IsCompleted && this.HumanPresencePollEnabled)
            {
                this.humanPresenceTaskTokenSource.Cancel();
                this.humanPresenceTaskTokenSource = new CancellationTokenSource();
                this.humanPresenceBackgroundTask = HumanPresenceUpdateBackgroundTask(this.humanPresenceTaskTokenSource.Token);
            }
        }

        internal void StopHumanPresenceUpdateTask()
        {
            this.humanPresenceTaskTokenSource.Cancel();
        }

        internal async Task HumanPresenceUpdateBackgroundTask(CancellationToken stoppingToken)
        {
            try
            {
                while (!stoppingToken.IsCancellationRequested)
                {
                    UpdatePresenceStatus();
                    await Task.Delay(HUMAN_PRESENCE_UPDATE_INTERVAL, stoppingToken);
                }
            }
            catch (OperationCanceledException)
            {
                return;
            }
            catch (Exception ex)
            {
                logger.Error($"Human Presence background task stopped due to error: {ex}");
            }
            finally
            {
                logger.Debug("Human Presence background task stopped.");
            }
        }

        internal void UpdateWalControlsVisibility(bool visible)
        {
            this.panelWal.Enabled = visible;
            this.panelWal.Visible = visible;
        }

        internal void UpdateWoaControlsVisibility(bool visible)
        {
            this.panelWoa.Enabled = visible;
            this.panelWoa.Visible = visible;
        }

        internal void UpdateNLoPControlsVisibility(bool visible)
        {
            this.panelNlop.Enabled = visible;
            this.panelNlop.Visible = visible;
        }

        internal void UpdateAdControlsVisibility(bool visible)
        {
            this.panelAd.Enabled = visible;
            this.panelAd.Visible = visible;
        }

        internal void UpdateOnlookerDetectionControlsVisibility(bool visible)
        {
            this.onlookerDetectionSettingsPanel.Enabled = visible;
            this.onlookerDetectionSettingsPanel.Visible = visible;
        }

        internal void UpdateAppFeatureControlsVisibility(bool visible)
        {
            this.appFeaturesPanel.Enabled = visible;
            this.appFeaturesPanel.Visible = visible;
            this.appEnabledPanel.Visible = visible;
            this.extendedCapabilitiesPanel.Visible = visible;
            this.ipfConnectionActivePanel.Visible = visible;
        }

        private string CollectionToString(System.Collections.Specialized.StringCollection collection)
        {
            var sb = new StringBuilder();
            foreach (string str in collection)
            {
                sb.Append(str);
                sb.Append(",");
            }

            return sb.ToString();
        }

        private void SetAppName()
        {
            string appName = "Intel.ContextSensingSampleApp";
            Task setTask = this._commClient.SetOption(FeatureType.APPLICATION, FeatureProperty.AppName, appName);
            HandleSetOptionExceptions(setTask);
        }

        private void SubscribeToDefaultEvents()
        {
            List<string> subscribedEvents = GetSubscribedEvents(this.enabledEventsListBox.SelectedItems);

            foreach (EventType eventType in this.defaultEvents)
            {
                subscribedEvents.Add(EventTypeMap.EventSubscriptionNames[eventType]);
            }

            subscribedEvents = subscribedEvents.Distinct().ToList();

            Task setTask = this._commClient.SetOption(FeatureType.APPLICATION, FeatureProperty.EnabledEvents, subscribedEvents);
            HandleSetOptionExceptions(setTask);

            this.enabledEventsListBox.SelectedIndexChanged -= enabledEventsListBox_SelectedIndexChanged;
            UpdateEventListBox(subscribedEvents);
            this.enabledEventsListBox.SelectedIndexChanged += enabledEventsListBox_SelectedIndexChanged;
        }

        private List<string> GetSubscribedEvents(ListBox.SelectedObjectCollection selectedEvents)
        {
            var subscribedEventsList = new List<string>();
            foreach (string eventName in selectedEvents)
            {
                if (this.eventNames.Contains(eventName))
                {
                    // Sanity check to make sure we don't send bad data to the service
                    subscribedEventsList.Add(eventName);
                }
            }

            return subscribedEventsList;
        }

        public void SetServiceSdkVersion(string sdkVersion)
        {
            this.icstSdkVersionLabel.Text = "ICST SDK Version: " + sdkVersion;
        }
        #endregion helpers

        private void awayExternalMonitorOnRadioButton_CheckedChanged(object sender, EventArgs e)
        {
            if (sender is RadioButton && ((RadioButton)sender).Checked)
            {
                Task setTask = this._commClient.SetOption(FeatureType.LOCK, FeatureProperty.EnableWithExtMonitor, true, this.globalMode);
                HandleSetOptionExceptions(setTask);
            }
        }

        private void awayExternalMonitorOffRadioButton_CheckedChanged(object sender, EventArgs e)
        {
            if (sender is RadioButton && ((RadioButton)sender).Checked)
            {
                Task setTask = this._commClient.SetOption(FeatureType.LOCK, FeatureProperty.EnableWithExtMonitor, false, this.globalMode);
                HandleSetOptionExceptions(setTask);
            }
        }

        private void preDimIntervalComboBox_SelectedIndexChanged(object sender, EventArgs e)
        {
            if (sender is ComboBox)
            {
                if (((ComboBox)sender).SelectedIndex < this.preDimIntervalSeconds.Length)
                {
                    Task setTask = this._commClient.SetOption(FeatureType.LOCK, FeatureProperty.PreDimInterval, this.preDimIntervalSeconds[((ComboBox)sender).SelectedIndex] * 1000, this.globalMode);
                    HandleSetOptionExceptions(setTask);
                }
            }
        }

        private void notPresentDimmingTimeComboBox_SelectedIndexChanged(object sender, EventArgs e)
        {
            if (sender is ComboBox)
            {
                if (((ComboBox)sender).SelectedIndex < this.dimmingIntervalSeconds.Length)
                {
                    Task setTask = this._commClient.SetOption(FeatureType.LOCK, FeatureProperty.DimmingInterval, this.dimmingIntervalSeconds[((ComboBox)sender).SelectedIndex] * 1000, this.globalMode);
                    HandleSetOptionExceptions(setTask);
                }
            }
        }

        private void notPresentDimTargetComboBox_SelectedIndexChanged(object sender, EventArgs e)
        {
            if (sender is ComboBox)
            {
                if (((ComboBox)sender).SelectedIndex < this.dimTarget.Length)
                {
                    Task setTask = this._commClient.SetOption(FeatureType.LOCK, FeatureProperty.NotPresentDimTarget, this.dimTarget[((ComboBox)sender).SelectedIndex], this.globalMode);
                    HandleSetOptionExceptions(setTask);
                }
            }
        }

        private void userPresentWaitIntervalComboBox_SelectedIndexChanged(object sender, EventArgs e)
        {
            if (sender is ComboBox && ((ComboBox)sender).SelectedIndex < this.userPresentWaitIntervalsSeconds.Length)
            {
                Task setTask = this._commClient.SetOption(FeatureType.LOCK, FeatureProperty.WaitInterval, this.userPresentWaitIntervalsSeconds[((ComboBox)sender).SelectedIndex] * 1000, this.globalMode);
                HandleSetOptionExceptions(setTask);
            }
        }

        private void dimScreenOnRadioButton_CheckedChanged(object sender, EventArgs e)
        {
            if (sender is RadioButton && ((RadioButton)sender).Checked)
            {
                Task setTask = this._commClient.SetOption(FeatureType.LOCK, FeatureProperty.DimScreen, true, this.globalMode);
                HandleSetOptionExceptions(setTask);
            }
        }

        private void dimScreenOffRadioButton_CheckedChanged(object sender, EventArgs e)
        {
            if (sender is RadioButton && ((RadioButton)sender).Checked)
            {
                Task setTask = this._commClient.SetOption(FeatureType.LOCK, FeatureProperty.DimScreen, false, this.globalMode);
                HandleSetOptionExceptions(setTask);
            }
        }

        private void honorPowerRequestsOnRadioButton_CheckedChanged(object sender, EventArgs e)
        {
            if (sender is RadioButton && ((RadioButton)sender).Checked)
            {
                Task setTask = this._commClient.SetOption(FeatureType.LOCK, FeatureProperty.PowerRequest, true, this.globalMode);
                HandleSetOptionExceptions(setTask);
            }
        }

        private void honorPowerRequestsOffRadioButton_CheckedChanged(object sender, EventArgs e)
        {
            if (sender is RadioButton && ((RadioButton)sender).Checked)
            {
                Task setTask = this._commClient.SetOption(FeatureType.LOCK, FeatureProperty.PowerRequest, false, this.globalMode);
                HandleSetOptionExceptions(setTask);
            }
        }

        private void honorUserInCallOnRadioButton_CheckedChanged(object sender, EventArgs e)
        {
            if (sender is RadioButton && ((RadioButton)sender).Checked)
            {
                Task setTask = this._commClient.SetOption(FeatureType.LOCK, FeatureProperty.EnableAudioMode, true, this.globalMode);
                HandleSetOptionExceptions(setTask);
            }
        }

        private void honorUserInCallOffRadioButton_CheckedChanged(object sender, EventArgs e)
        {
            if (sender is RadioButton && ((RadioButton)sender).Checked)
            {
                Task setTask = this._commClient.SetOption(FeatureType.LOCK, FeatureProperty.EnableAudioMode, false, this.globalMode);
                HandleSetOptionExceptions(setTask);
            }
        }

        private void lockIntervalComboBox_SelectedIndexChanged(object sender, EventArgs e)
        {
            if (sender is ComboBox && ((ComboBox)sender).SelectedIndex < this.lockAfterDimIntervalsSeconds.Length)
            {
                Task setTask = this._commClient.SetOption(FeatureType.LOCK, FeatureProperty.LockAfterDimInterval, this.lockAfterDimIntervalsSeconds[((ComboBox)sender).SelectedIndex] * 1000, this.globalMode);
                HandleSetOptionExceptions(setTask);
            }
        }

        private void displayOffOnRadioButton_CheckedChanged(object sender, EventArgs e)
        {
            if (sender is RadioButton && ((RadioButton)sender).Checked)
            {
                Task setTask = this._commClient.SetOption(FeatureType.LOCK, FeatureProperty.DisplayOffAfterLock, true, this.globalMode);
                HandleSetOptionExceptions(setTask);
            }
        }

        private void displayOffOffRadioButton_CheckedChanged(object sender, EventArgs e)
        {
            if (sender is RadioButton && ((RadioButton)sender).Checked)
            {
                Task setTask = this._commClient.SetOption(FeatureType.LOCK, FeatureProperty.DisplayOffAfterLock, false, this.globalMode);
                HandleSetOptionExceptions(setTask);
            }
        }

        private void nearExternalMonitorOnRadioButton_CheckedChanged(object sender, EventArgs e)
        {
            if (sender is RadioButton && ((RadioButton)sender).Checked)
            {
                Task setTask = this._commClient.SetOption(FeatureType.WAKE, FeatureProperty.EnableWithExtMonitor, true, this.globalMode);
                HandleSetOptionExceptions(setTask);
            }
        }

        private void nearExternalMonitorOffRadioButton_CheckedChanged(object sender, EventArgs e)
        {
            if (sender is RadioButton && ((RadioButton)sender).Checked)
            {
                Task setTask = this._commClient.SetOption(FeatureType.WAKE, FeatureProperty.EnableWithExtMonitor, false, this.globalMode);
                HandleSetOptionExceptions(setTask);
            }
        }

        private void woaEnableOnBatteryTrue_radio_CheckedChanged(object sender, EventArgs e)
        {
            if (sender is RadioButton && ((RadioButton)sender).Checked)
            {
                Task setTask = this._commClient.SetOption(FeatureType.WAKE, FeatureProperty.WakeOnLowBattery, true, this.globalMode);
                HandleSetOptionExceptions(setTask);
            }
        }

        private void woaEnableOnBatteryFalse_radio_CheckedChanged(object sender, EventArgs e)
        {
            if (sender is RadioButton && ((RadioButton)sender).Checked)
            {
                Task setTask = this._commClient.SetOption(FeatureType.WAKE, FeatureProperty.WakeOnLowBattery, false, this.globalMode);
                HandleSetOptionExceptions(setTask);
            }
        }

        private void woaBatteryComboBox_SelectedIndexChanged(object sender, EventArgs e)
        {
            if (sender is ComboBox)
            {
                if (((ComboBox)sender).SelectedIndex < this.batteryRemainingPercentage.Length)
                {
                    Task setTask = this._commClient.SetOption(FeatureType.WAKE, FeatureProperty.LowBatteryRemainingPercentageLimit, this.batteryRemainingPercentage[((ComboBox)sender).SelectedIndex], this.globalMode);
                    HandleSetOptionExceptions(setTask);
                }
            }
        }

        private void nlopExternalMonitorOffRadioButton_CheckedChanged(object sender, EventArgs e)
        {
            if (sender is RadioButton && ((RadioButton)sender).Checked)
            {
                Task setTask = this._commClient.SetOption(FeatureType.NOLOCKONPRESENCE, FeatureProperty.EnableWithExtMonitor, false, this.globalMode);
                HandleSetOptionExceptions(setTask);
            }
        }

        private void nlopExternalMonitorOnRadioButton_CheckedChanged(object sender, EventArgs e)
        {
            if (sender is RadioButton && ((RadioButton)sender).Checked)
            {
                Task setTask = this._commClient.SetOption(FeatureType.NOLOCKONPRESENCE, FeatureProperty.EnableWithExtMonitor, true, this.globalMode);
                HandleSetOptionExceptions(setTask);
            }
        }

        private void nlopEnableOnBatteryTrue_radio_CheckedChanged(object sender, EventArgs e)
        {
            if (sender is RadioButton && ((RadioButton)sender).Checked)
            {
                Task setTask = this._commClient.SetOption(FeatureType.NOLOCKONPRESENCE, FeatureProperty.EnableOnBattery, true, this.globalMode);
                HandleSetOptionExceptions(setTask);
            }
        }

        private void nlopEnableOnBatteryFalse_radio_CheckedChanged(object sender, EventArgs e)
        {
            if (sender is RadioButton && ((RadioButton)sender).Checked)
            {
                Task setTask = this._commClient.SetOption(FeatureType.NOLOCKONPRESENCE, FeatureProperty.EnableOnBattery, false, this.globalMode);
                HandleSetOptionExceptions(setTask);
            }
        }

        private void nlopBatteryComboBox_SelectedIndexChanged(object sender, EventArgs e)
        {
            if (sender is ComboBox)
            {
                if (((ComboBox)sender).SelectedIndex < this.batteryRemainingPercentage.Length)
                {
                    Task setTask = this._commClient.SetOption(FeatureType.NOLOCKONPRESENCE, FeatureProperty.LowBatteryRemainingPercentageLimit, this.batteryRemainingPercentage[((ComboBox)sender).SelectedIndex], this.globalMode);
                    HandleSetOptionExceptions(setTask);
                }
            }
        }

        private void waitIntervalComboBox_SelectedIndexChanged(object sender, EventArgs e)
        {
            if (sender is ComboBox && ((ComboBox)sender).SelectedIndex < this.preDimIntervalSeconds.Length)
            {
                Task setTask = this._commClient.SetOption(FeatureType.NOLOCKONPRESENCE, FeatureProperty.WaitInterval, this.preDimIntervalSeconds[((ComboBox)sender).SelectedIndex] * 1000, this.globalMode);
                HandleSetOptionExceptions(setTask);
            }
        }

        private void failSafeTimeoutComboBox_SelectedIndexChanged(object sender, EventArgs e)
        {
            if (sender is ComboBox && ((ComboBox)sender).SelectedIndex < this.preDimIntervalSeconds.Length)
            {
                Task setTask = this._commClient.SetOption(FeatureType.NOLOCKONPRESENCE, FeatureProperty.FailSafeTimeout, this.failSafeIntervalSeconds[((ComboBox)sender).SelectedIndex] * 1000, this.globalMode);
                HandleSetOptionExceptions(setTask);
            }
        }

        private void adExternalMonitorOnRadioButton_CheckedChanged(object sender, EventArgs e)
        {
            if (sender is RadioButton && ((RadioButton)sender).Checked)
            {
                Task setTask = this._commClient.SetOption(FeatureType.ADAPTIVEDIMMING, FeatureProperty.EnableWithExtMonitor, true, this.globalMode);
                HandleSetOptionExceptions(setTask);
            }
        }

        private void adExternalMonitorOffRadioButton_CheckedChanged(object sender, EventArgs e)
        {
            if (sender is RadioButton && ((RadioButton)sender).Checked)
            {
                Task setTask = this._commClient.SetOption(FeatureType.ADAPTIVEDIMMING, FeatureProperty.EnableWithExtMonitor, false, this.globalMode);
                HandleSetOptionExceptions(setTask);
            }
        }

        private void adNotPresentDimWaitTimeComboBox_SelectedIndexChanged(object sender, EventArgs e)
        {
            if (sender is ComboBox && ((ComboBox)sender).SelectedIndex < this.preDimIntervalSeconds.Length)
            {
                Task setTask = this._commClient.SetOption(FeatureType.ADAPTIVEDIMMING, FeatureProperty.NotPresentDimWaitTime, this.preDimIntervalSeconds[((ComboBox)sender).SelectedIndex] * 1000, this.globalMode);
                HandleSetOptionExceptions(setTask);
            }
        }

        private void mispredictionWithFaceDetectionCheckbox_CheckedChanged(object sender, EventArgs e)
        {
            if (sender is CheckBox)
            {
                if (((CheckBox)sender).CheckState == CheckState.Checked)
                {
                    Task setTask = this._commClient.SetOption(FeatureType.ADAPTIVEDIMMING, FeatureProperty.MispredictionFaceDetection, true, this.globalMode);
                    HandleSetOptionExceptions(setTask);
                }
                else
                {
                    Task setTask = this._commClient.SetOption(FeatureType.ADAPTIVEDIMMING, FeatureProperty.MispredictionFaceDetection, false, this.globalMode);
                    HandleSetOptionExceptions(setTask);
                }
            }
        }

        private void honorUserInCallADRadioButtonOff_CheckedChanged(object sender, EventArgs e)
        {
            if (sender is RadioButton && ((RadioButton)sender).Checked)
            {
                Task setTask = this._commClient.SetOption(FeatureType.ADAPTIVEDIMMING, FeatureProperty.EnableAudioMode, false, this.globalMode);
                HandleSetOptionExceptions(setTask);
            }
        }

        private void honorUserInCallADRadioButtonOn_CheckedChanged(object sender, EventArgs e)
        {
            if (sender is RadioButton && ((RadioButton)sender).Checked)
            {
                Task setTask = this._commClient.SetOption(FeatureType.ADAPTIVEDIMMING, FeatureProperty.EnableAudioMode, true, this.globalMode);
                HandleSetOptionExceptions(setTask);
            }
        }

        private void zeroRefreshRateDisableRadioButton_CheckedChanged(object sender, EventArgs e)
        {
            if (sender is RadioButton && ((RadioButton)sender).Checked)
            {
                Task setTask = this._commClient.SetOption(FeatureType.ADAPTIVEDIMMING, FeatureProperty.ZeroRefreshRateEnabled, false, this.globalMode);
                HandleSetOptionExceptions(setTask);
            }
        }

        private void zeroRefreshRateEnableRadioButton_CheckedChanged(object sender, EventArgs e)
        {
            if (sender is RadioButton && ((RadioButton)sender).Checked)
            {
                Task setTask = this._commClient.SetOption(FeatureType.ADAPTIVEDIMMING, FeatureProperty.ZeroRefreshRateEnabled, true, this.globalMode);
                HandleSetOptionExceptions(setTask);
            }
        }

        private void lowRefreshRateDisableRadioButton_CheckedChanged(object sender, EventArgs e)
        {
            if (sender is RadioButton && ((RadioButton)sender).Checked)
            {
                Task setTask = this._commClient.SetOption(FeatureType.ADAPTIVEDIMMING, FeatureProperty.LowRefreshRateEnabled, false, this.globalMode);
                HandleSetOptionExceptions(setTask);
            }
        }

        private void lowRefreshRateEnableRadioButton_CheckedChanged(object sender, EventArgs e)
        {
            if (sender is RadioButton && ((RadioButton)sender).Checked)
            {
                Task setTask = this._commClient.SetOption(FeatureType.ADAPTIVEDIMMING, FeatureProperty.LowRefreshRateEnabled, true, this.globalMode);
                HandleSetOptionExceptions(setTask);
            }
        }

        private void mispredictionTimeWindowCombobox_SelectedIndexChanged(object sender, EventArgs e)
        {
            if (sender is ComboBox && ((ComboBox)sender).SelectedIndex < this.mispredictionTimeWindowSeconds.Length)
            {
                Task setTask = this._commClient.SetOption(FeatureType.ADAPTIVEDIMMING, FeatureProperty.MispredictionTimeWindow, this.mispredictionTimeWindowSeconds[((ComboBox)sender).SelectedIndex] * 1000, this.globalMode);
                HandleSetOptionExceptions(setTask);
            }
        }

        private void MP1DimWaitComboBox_SelectedIndexChanged(object sender, EventArgs e)
        {
            if (sender is ComboBox && ((ComboBox)sender).SelectedIndex < this.mispredictionDimWaitSeconds.Length)
            {
                Task setTask = this._commClient.SetOption(FeatureType.ADAPTIVEDIMMING, FeatureProperty.Misprediction1DimWaitTime, this.mispredictionDimWaitSeconds[((ComboBox)sender).SelectedIndex] * 1000, this.globalMode);
                HandleSetOptionExceptions(setTask);
            }
        }

        private void MP2DimWaitComboBox_SelectedIndexChanged(object sender, EventArgs e)
        {
            if (sender is ComboBox && ((ComboBox)sender).SelectedIndex < this.mispredictionDimWaitSeconds.Length)
            {
                Task setTask = this._commClient.SetOption(FeatureType.ADAPTIVEDIMMING, FeatureProperty.Misprediction2DimWaitTime, this.mispredictionDimWaitSeconds[((ComboBox)sender).SelectedIndex] * 1000, this.globalMode);
                HandleSetOptionExceptions(setTask);
            }
        }

        private void MP3DimWaitComboBox_SelectedIndexChanged(object sender, EventArgs e)
        {
            if (sender is ComboBox && ((ComboBox)sender).SelectedIndex < this.mispredictionDimWaitSeconds.Length)
            {
                Task setTask = this._commClient.SetOption(FeatureType.ADAPTIVEDIMMING, FeatureProperty.Misprediction3DimWaitTime, this.mispredictionDimWaitSeconds[((ComboBox)sender).SelectedIndex] * 1000, this.globalMode);
                HandleSetOptionExceptions(setTask);
            }
        }

        private void MP4DimWaitComboBox_SelectedIndexChanged(object sender, EventArgs e)
        {
            if (sender is ComboBox && ((ComboBox)sender).SelectedIndex < this.mispredictionDimWaitSeconds.Length)
            {
                Task setTask = this._commClient.SetOption(FeatureType.ADAPTIVEDIMMING, FeatureProperty.Misprediction4DimWaitTime, this.mispredictionDimWaitSeconds[((ComboBox)sender).SelectedIndex] * 1000, this.globalMode);
                HandleSetOptionExceptions(setTask);
            }
        }

        private void adDisengagedDimTargetComboBox_SelectedIndexChanged(object sender, EventArgs e)
        {
            if (sender is ComboBox)
            {
                if (((ComboBox)sender).SelectedIndex < this.dimTarget.Length)
                {
                    Task setTask = this._commClient.SetOption(FeatureType.ADAPTIVEDIMMING, FeatureProperty.DisengagedDimTarget, this.dimTarget[((ComboBox)sender).SelectedIndex], this.globalMode);
                    HandleSetOptionExceptions(setTask);
                }
            }
        }

        private void adNotPresentDimTargetComboBox_SelectedIndexChanged(object sender, EventArgs e)
        {
            if (sender is ComboBox)
            {
                if (((ComboBox)sender).SelectedIndex < this.dimTarget.Length)
                {
                    Task setTask = this._commClient.SetOption(FeatureType.ADAPTIVEDIMMING, FeatureProperty.NotPresentDimTarget, this.dimTarget[((ComboBox)sender).SelectedIndex], this.globalMode);
                    HandleSetOptionExceptions(setTask);
                }
            }
        }

        private void adDimmingIntervalComboBox_SelectedIndexChanged(object sender, EventArgs e)
        {
            if (sender is ComboBox)
            {
                if (((ComboBox)sender).SelectedIndex < this.dimmingIntervalSeconds.Length)
                {
                    Task setTask = this._commClient.SetOption(FeatureType.ADAPTIVEDIMMING, FeatureProperty.DimmingInterval, this.dimmingIntervalSeconds[((ComboBox)sender).SelectedIndex] * 1000, this.globalMode);
                    HandleSetOptionExceptions(setTask);
                }
            }
        }

        private void adDisengagedDimWaitTimeComboBox_SelectedIndexChanged(object sender, EventArgs e)
        {
            if (sender is ComboBox)
            {
                if (((ComboBox)sender).SelectedIndex < this.preDimIntervalSeconds.Length)
                {
                    Task setTask = this._commClient.SetOption(FeatureType.ADAPTIVEDIMMING, FeatureProperty.DisengagedDimWaitTime, this.preDimIntervalSeconds[((ComboBox)sender).SelectedIndex] * 1000, this.globalMode);
                    HandleSetOptionExceptions(setTask);
                }
            }
        }

        private void onlookerDetectionDelayIntervalComboBox_SelectedIndexChanged(object sender, EventArgs e)
        {
            if (sender is ComboBox && ((ComboBox)sender).SelectedIndex < this.preDimIntervalSeconds.Length)
            {
                Task setTask = this._commClient.SetOption(FeatureType.ONLOOKERDETECTION, FeatureProperty.DetectionDelayInterval, this.preDimIntervalSeconds[((ComboBox)sender).SelectedIndex] * 1000, this.globalMode);
                HandleSetOptionExceptions(setTask);
            }
        }

        private void onlookerDetectionDimmingTimeComboBox_SelectedIndexChanged(object sender, EventArgs e)
        {
            if (sender is ComboBox && ((ComboBox)sender).SelectedIndex < this.dimmingIntervalSeconds.Length)
            {
                Task setTask = this._commClient.SetOption(FeatureType.ONLOOKERDETECTION, FeatureProperty.DimmingInterval, this.dimmingIntervalSeconds[((ComboBox)sender).SelectedIndex] * 1000, this.globalMode);
                HandleSetOptionExceptions(setTask);
            }
        }

        private void onlookerDetectionDimTargetComboBox_SelectedIndexChanged(object sender, EventArgs e)
        {
            if (sender is ComboBox && ((ComboBox)sender).SelectedIndex < this.dimTarget.Length)
            {
                Task setTask = this._commClient.SetOption(FeatureType.ONLOOKERDETECTION, FeatureProperty.DimTarget, this.dimTarget[((ComboBox)sender).SelectedIndex], this.globalMode);
                HandleSetOptionExceptions(setTask);
            }
        }

        private void onlookerDetectionLowBatteryLimitComboBox_SelectedIndexChanged(object sender, EventArgs e)
        {
            if (sender is ComboBox && ((ComboBox)sender).SelectedIndex < this.batteryRemainingPercentage.Length)
            {
                Task setTask = this._commClient.SetOption(FeatureType.ONLOOKERDETECTION, FeatureProperty.LowBatteryRemainingPercentageLimit, this.batteryRemainingPercentage[((ComboBox)sender).SelectedIndex], this.globalMode);
                HandleSetOptionExceptions(setTask);
            }
        }

        private void onlookerDetectionAbsenceIntervalComboBox_SelectedIndexChanged(object sender, EventArgs e)
        {
            if (sender is ComboBox && ((ComboBox)sender).SelectedIndex < this.detectionAbsenceIntervalSeconds.Length)
            {
                Task setTask = this._commClient.SetOption(FeatureType.ONLOOKERDETECTION, FeatureProperty.OnlookerAbsenceWaitInterval, this.detectionAbsenceIntervalSeconds[((ComboBox)sender).SelectedIndex] * 1000, this.globalMode);
                HandleSetOptionExceptions(setTask);
            }
        }

        private void onlookerDetectionExternalMonitorOnRadioButton_CheckedChanged(object sender, EventArgs e)
        {
            if (sender is RadioButton && ((RadioButton)sender).Checked)
            {
                Task setTask = this._commClient.SetOption(FeatureType.ONLOOKERDETECTION, FeatureProperty.EnableWithExtMonitor, true, this.globalMode);
                HandleSetOptionExceptions(setTask);
            }
        }

        private void onlookerDetectionExternalMonitorOffRadioButton_CheckedChanged(object sender, EventArgs e)
        {
            if (sender is RadioButton && ((RadioButton)sender).Checked)
            {
                Task setTask = this._commClient.SetOption(FeatureType.ONLOOKERDETECTION, FeatureProperty.EnableWithExtMonitor, false, this.globalMode);
                HandleSetOptionExceptions(setTask);
            }
        }

        private void onlookerDetectionDimWhenDetectedOnRadioButton_CheckedChanged(object sender, EventArgs e)
        {
            if (sender is RadioButton && ((RadioButton)sender).Checked)
            {
                Task setTask = this._commClient.SetOption(FeatureType.ONLOOKERDETECTION, FeatureProperty.DimWhenOnlookerDetected, true, this.globalMode);
                HandleSetOptionExceptions(setTask);
            }
        }

        private void onlookerDetectionDimWhenDetectedOffRadioButton_CheckedChanged(object sender, EventArgs e)
        {
            if (sender is RadioButton && ((RadioButton)sender).Checked)
            {
                Task setTask = this._commClient.SetOption(FeatureType.ONLOOKERDETECTION, FeatureProperty.DimWhenOnlookerDetected, false, this.globalMode);
                HandleSetOptionExceptions(setTask);
            }
        }

        private void onlookerDetectionEnableOnLowBatteryOnRadioButton_CheckedChanged(object sender, EventArgs e)
        {
            if (sender is RadioButton && ((RadioButton)sender).Checked)
            {
                Task setTask = this._commClient.SetOption(FeatureType.ONLOOKERDETECTION, FeatureProperty.EnableOnLowBattery, true, this.globalMode);
                HandleSetOptionExceptions(setTask);
            }
        }

        private void onlookerDetectionEnableOnLowBatteryOffRadioButton_CheckedChanged(object sender, EventArgs e)
        {
            if (sender is RadioButton && ((RadioButton)sender).Checked)
            {
                Task setTask = this._commClient.SetOption(FeatureType.ONLOOKERDETECTION, FeatureProperty.EnableOnLowBattery, false, this.globalMode);
                HandleSetOptionExceptions(setTask);
            }
        }

        private void appNameComboBox_SelectedIndexChanged(object sender, EventArgs e)
        {
            if (sender is ComboBox)
            {
                if (((ComboBox)sender).SelectedIndex < this.appNames.Length)
                {
                    Task setTask = this._commClient.SetOption(FeatureType.APPLICATION, FeatureProperty.AppName, this.appNames[((ComboBox)sender).SelectedIndex], this.globalMode);
                    HandleSetOptionExceptions(setTask);
                }
            }
        }

        private void globalModeCheckbox_CheckedChanged(object sender, EventArgs e)
        {
            if (sender is CheckBox)
            {
                if (((CheckBox)sender).CheckState == CheckState.Checked)
                {
                    this.globalMode = SettingsAccessType.GLOBAL;
                    UpdateWoaControlsVisibility(true);
                    UpdateWalControlsVisibility(true);
                    UpdateNLoPControlsVisibility(true);
                    UpdateAdControlsVisibility(true);
                    UpdateOnlookerDetectionControlsVisibility(true);
                    // Spp panel visibility is controlled separately, even in global mode
                    // See the "test mode" checkbox
                }
                else
                {
                    this.globalMode = SettingsAccessType.LOCAL;
                }

                _ = RefreshFeaturesViaGetOptions();
            }
        }

        internal void UpdateControlSettings(FeatureSetting featureSetting)
        {
            if (featureSetting is LockFeatures)
            {
                UpdateControlSettings((LockFeatures)featureSetting);
            }
            else if (featureSetting is WakeFeatures)
            {
                UpdateControlSettings((WakeFeatures)featureSetting);
            }
            else if (featureSetting is NoLockOnPresenceFeatures)
            {
                UpdateControlSettings((NoLockOnPresenceFeatures)featureSetting);
            }
            else if (featureSetting is AdaptiveDimmingFeatures)
            {
                UpdateControlSettings((AdaptiveDimmingFeatures)featureSetting);
            }
            else if (featureSetting is OnlookerDetectionFeatures)
            {
                UpdateControlSettings((OnlookerDetectionFeatures)featureSetting);
            }
            else if (featureSetting is AppFeatures)
            {
                UpdateControlSettings((AppFeatures)featureSetting);
            }
            else if (featureSetting is UserBasedRefreshRateFeatures)
            {
                UpdateControlSettings((UserBasedRefreshRateFeatures)featureSetting);
            }
            else if (featureSetting is TouchPauseScanFeatures)
            {
                UpdateControlSettings((TouchPauseScanFeatures)featureSetting);
            }

            this.globalModeCheckbox.Checked = this.globalMode == SettingsAccessType.GLOBAL;
        }

        string ConvertToPascalCase(string str)
        {
            return str.First().ToString().ToUpper() + str.Substring(1);
        }

        internal void UpdateControlSettings(LockFeatures lockFeatures)
        {
            RemoveControlEventHandlers();
            this.walCheckBox.Checked = lockFeatures.WALEnabled;
            this.walExternalMonitorOnRadioButton.Checked = lockFeatures.EnableLockWithExternalMonitor;
            this.walExternalMonitorOffRadioButton.Checked = !lockFeatures.EnableLockWithExternalMonitor;
            this.preDimIntervalComboBox.SelectedIndex = Array.IndexOf(this.preDimIntervalSeconds, lockFeatures.PreDimInterval / 1000);
            this.notPresentDimmingTimeComboBox.SelectedIndex = Array.IndexOf(this.dimmingIntervalSeconds, lockFeatures.DimmingInterval / 1000);
            this.notPresentDimTargetComboBox.SelectedIndex = Array.IndexOf(this.dimTarget, lockFeatures.NotPresentDimTarget);
            this.userPresentWaitIntervalComboBox.SelectedIndex = Array.IndexOf(this.userPresentWaitIntervalsSeconds, lockFeatures.WaitInterval / 1000);
            this.dimScreenOnRadioButton.Checked = lockFeatures.DimScreen;
            this.dimScreenOffRadioButton.Checked = !lockFeatures.DimScreen;
            this.honorPowerRequestsOnRadioButton.Checked = lockFeatures.PowerRequest;
            this.honorPowerRequestsOffRadioButton.Checked = !lockFeatures.PowerRequest;
            this.honorUserInCallOnRadioButton.Checked = lockFeatures.EnableAudioMode;
            this.honorUserInCallOffRadioButton.Checked = !lockFeatures.EnableAudioMode;
            this.lockAfterDimIntervalComboBox.SelectedIndex = Array.IndexOf(this.lockAfterDimIntervalsSeconds, lockFeatures.LockAfterDimInterval / 1000);
            this.displayOffOnRadioButton.Checked = lockFeatures.DisplayOffAfterLock;
            this.displayOffOffRadioButton.Checked = !lockFeatures.DisplayOffAfterLock;

            AddControlEventHandlers();
            UpdateWalControlsVisibility(this.walCheckBox.Checked);
        }

        internal void UpdateControlSettings(WakeFeatures wakeFeatures)
        {
            RemoveControlEventHandlers();
            this.woaCheckBox.Checked = wakeFeatures.WOAEnabled;
            this.woaExternalMonitorOnRadioButton.Checked = wakeFeatures.EnableWakeWithExternalMonitor;
            this.woaExternalMonitorOffRadioButton.Checked = !wakeFeatures.EnableWakeWithExternalMonitor;
            this.woaEnableOnBatteryTrueRadioButton.Checked = wakeFeatures.WakeOnLowBattery;
            this.woaEnableOnBatteryFalseRadioButton.Checked = !wakeFeatures.WakeOnLowBattery;
            this.woaBatteryRemainingComboBox.SelectedIndex = Array.IndexOf(this.batteryRemainingPercentage, wakeFeatures.LowBatteryRemainingPercentageLimit);

            AddControlEventHandlers();

            UpdateWoaControlsVisibility(this.woaCheckBox.Checked);
        }

        internal void UpdateControlSettings(NoLockOnPresenceFeatures nlopFeatures)
        {
            RemoveControlEventHandlers();
            this.nlopCheckBox.Checked = nlopFeatures.NLoPEnabled;
            this.nlopExternalMonitorOnRadioButton.Checked = nlopFeatures.EnableNLoPWithExternalMonitor;
            this.nlopExternalMonitorOffRadioButton.Checked = !nlopFeatures.EnableNLoPWithExternalMonitor;
            this.nlopEnableOnBatteryTrueRadioButton.Checked = nlopFeatures.EnableOnBattery;
            this.nlopEnableOnBatteryFalseRadioButton.Checked = !nlopFeatures.EnableOnBattery;
            this.nlopBatteryRemainingComboBox.SelectedIndex = Array.IndexOf(this.batteryRemainingPercentage, nlopFeatures.LowBatteryRemainingPercentageLimit);
            this.nlopBatteryRemainingComboBox.SelectedIndex = Array.IndexOf(this.batteryRemainingPercentage, nlopFeatures.LowBatteryRemainingPercentageLimit);
            this.comboBoxWaitTimer.SelectedIndex = Array.IndexOf(this.preDimIntervalSeconds, nlopFeatures.WaitInterval / 1000);
            this.nlopFailSafeTimeoutComboBox.SelectedIndex = Array.IndexOf(this.failSafeIntervalSeconds, nlopFeatures.FailSafeTimeout / 1000);
            UpdateNLoPControlsVisibility(this.nlopCheckBox.Checked);
            AddControlEventHandlers();
        }

        internal void UpdateControlSettings(AdaptiveDimmingFeatures adaptiveDimmingFeatures)
        {
            RemoveControlEventHandlers();
            this.adCheckBox.Checked = adaptiveDimmingFeatures.ADEnabled;
            this.adExternalMonitorOnRadioButton.Checked = adaptiveDimmingFeatures.EnableAdaptiveDimmingWithExternalMonitor;
            this.adExternalMonitorOffRadioButton.Checked = !adaptiveDimmingFeatures.EnableAdaptiveDimmingWithExternalMonitor;
            this.adHonorPowerRequestOnRadioButton.Checked = adaptiveDimmingFeatures.PowerRequest;
            this.adHonorPowerRequestOffRadioButton.Checked = !adaptiveDimmingFeatures.PowerRequest;
            this.honorUserInCallADRadioButtonOn.Checked = adaptiveDimmingFeatures.EnableAudioMode;
            this.honorUserInCallADRadioButtonOff.Checked = !adaptiveDimmingFeatures.EnableAudioMode;
            this.ignoreHidInputRadioOn.Checked = adaptiveDimmingFeatures.IgnoreHidInput;
            this.ignoreHidInputRadioOff.Checked = !adaptiveDimmingFeatures.IgnoreHidInput;
            this.adNotPresentDimWaitTimeComboBox.SelectedIndex = Array.IndexOf(this.preDimIntervalSeconds, adaptiveDimmingFeatures.NotPresentDimWaitTime / 1000);
            this.mispredictionTimeWindowCombobox.SelectedIndex = Array.IndexOf(this.mispredictionTimeWindowSeconds, adaptiveDimmingFeatures.MispredictionTimeWindow / 1000);
            this.mispredictionWithFaceDetectionCheckbox.Checked = adaptiveDimmingFeatures.MispredictionFaceDetection;
            this.MP1DimWaitComboBox.SelectedIndex = Array.IndexOf(this.mispredictionDimWaitSeconds, adaptiveDimmingFeatures.Misprediction1DimWaitTime / 1000);
            this.MP2DimWaitComboBox.SelectedIndex = Array.IndexOf(this.mispredictionDimWaitSeconds, adaptiveDimmingFeatures.Misprediction2DimWaitTime / 1000);
            this.MP3DimWaitComboBox.SelectedIndex = Array.IndexOf(this.mispredictionDimWaitSeconds, adaptiveDimmingFeatures.Misprediction3DimWaitTime / 1000);
            this.MP4DimWaitComboBox.SelectedIndex = Array.IndexOf(this.mispredictionDimWaitSeconds, adaptiveDimmingFeatures.Misprediction4DimWaitTime / 1000);
            this.adDisengagedDimTargetComboBox.SelectedIndex = Array.IndexOf(this.dimTarget, adaptiveDimmingFeatures.DisengagedDimTarget);
            this.adNotPresentDimTargetComboBox.SelectedIndex = Array.IndexOf(this.dimTarget, adaptiveDimmingFeatures.NotPresentDimTarget);
            this.adDimmingIntervalComboBox.SelectedIndex = Array.IndexOf(this.dimmingIntervalSeconds, adaptiveDimmingFeatures.DimmingInterval / 1000);
            this.adDisengagedDimWaitTimeComboBox.SelectedIndex = Array.IndexOf(this.preDimIntervalSeconds, adaptiveDimmingFeatures.DisengagedDimWaitTime / 1000);
            this.lowRefreshRateEnableRadioButton.Checked = adaptiveDimmingFeatures.LowRefreshRateEnabled;
            this.lowRefreshRateDisableRadioButton.Checked = !adaptiveDimmingFeatures.LowRefreshRateEnabled;
            this.zeroRefreshRateEnableRadioButton.Checked = adaptiveDimmingFeatures.ZeroRefreshRateEnabled;
            this.zeroRefreshRateDisableRadioButton.Checked = !adaptiveDimmingFeatures.ZeroRefreshRateEnabled;
            this.touchPauseScanEnableRadioButton.Checked = adaptiveDimmingFeatures.TouchPauseScanEnabled;
            this.touchPauseScanDisableRadioButton.Checked = !adaptiveDimmingFeatures.TouchPauseScanEnabled;
            AddControlEventHandlers();
            UpdateAdControlsVisibility(this.adCheckBox.Checked);
        }

        internal void UpdateControlSettings(AppFeatures appFeatures)
        {
            RemoveControlEventHandlers();
            this.privacyCheckbox.Checked = appFeatures.PrivacyEnabled;
            this.appNameComboBox.SelectedIndex = Array.IndexOf(this.appNames, appFeatures.AppName);
            this.monitorAppCheckbox.Checked = appFeatures.MonitorApp;
            UpdateEventListBox(appFeatures.EnabledEvents);
            this.correlationResetTimeComboBox.SelectedIndex = Array.IndexOf(this.correlationResetSeconds, appFeatures.CorrelationResetTime / 1000);
            this.proximityCooldownComboBox.SelectedIndex = Array.IndexOf(this.proximityCooldownSeconds, appFeatures.ProximityCooldownTime / 1000.0);
            this.hidActiveDelayComboBox.SelectedIndex = Array.IndexOf(this.hidActiveDelaySeconds, appFeatures.HidActiveDelay / 1000.0);
            AddControlEventHandlers();
        }

        private void UpdateEventListBox(List<string> eventList)
        {
            foreach (string eventName in this.eventNames)
            {
                int index = this.enabledEventsListBox.FindStringExact(eventName);
                if (index != ListBox.NoMatches)
                {
                    this.enabledEventsListBox.SetSelected(index, eventList.Contains(eventName));
                }
            }
        }

        internal void UpdateControlSettings(OnlookerDetectionFeatures onlookerDetectionFeatures)
        {
            RemoveControlEventHandlers();
            this.onlookerDetectionCheckBox.Checked = onlookerDetectionFeatures.OnlookerDetectionEnabled;
            this.onlookerDetectionExternalMonitorOnRadioButton.Checked = onlookerDetectionFeatures.EnableOnlookerDetectionWithExternalMonitor;
            this.onlookerDetectionExternalMonitorOffRadioButton.Checked = !onlookerDetectionFeatures.EnableOnlookerDetectionWithExternalMonitor;
            this.onlookerDetectionDelayIntervalComboBox.SelectedIndex = Array.IndexOf(this.preDimIntervalSeconds, onlookerDetectionFeatures.DetectionDelayInterval / 1000);
            this.onlookerDetectionDimWhenDetectedOnRadioButton.Checked = onlookerDetectionFeatures.DimWhenOnlookerDetected;
            this.onlookerDetectionDimWhenDetectedOffRadioButton.Checked = !onlookerDetectionFeatures.DimWhenOnlookerDetected;
            this.onlookerDetectionDimmingTimeComboBox.SelectedIndex = Array.IndexOf(this.dimmingIntervalSeconds, onlookerDetectionFeatures.DimmingInterval / 1000);
            this.onlookerDetectionDimTargetComboBox.SelectedIndex = Array.IndexOf(this.dimTarget, onlookerDetectionFeatures.DimTarget);
            this.onlookerDetectionEnableOnLowBatteryOnRadioButton.Checked = onlookerDetectionFeatures.EnableOnLowBattery;
            this.onlookerDetectionEnableOnLowBatteryOffRadioButton.Checked = !onlookerDetectionFeatures.EnableOnLowBattery;
            this.onlookerDetectionLowBatteryLimitComboBox.SelectedIndex = Array.IndexOf(this.batteryRemainingPercentage, onlookerDetectionFeatures.LowBatteryRemainingPercentageLimit);
            this.onlookerDetectionAbsenceIntervalComboBox.SelectedIndex = Array.IndexOf(this.detectionAbsenceIntervalSeconds, onlookerDetectionFeatures.OnlookerAbsenceWaitInterval / 1000);
            this.onlookerDetectionUseFaceAttentionOnRadioButton.Checked = onlookerDetectionFeatures.UseFaceAttention;
            this.onlookerDetectionUseFaceAttentionOffRadioButton.Checked = !onlookerDetectionFeatures.UseFaceAttention;

            AddControlEventHandlers();
            UpdateOnlookerDetectionControlsVisibility(this.onlookerDetectionCheckBox.Checked);
        }

        internal void UpdateControlSettings(UserBasedRefreshRateFeatures userBasedRefreshRateFeatures)
        {
            RemoveControlEventHandlers();
            this.hupLrrOffRadioButton.Checked = !userBasedRefreshRateFeatures.LowRefreshRateEnabled;
            this.hupLrrOnRadioButton.Checked = userBasedRefreshRateFeatures.LowRefreshRateEnabled;
            this.hupZrrOffRadioButton.Checked = !userBasedRefreshRateFeatures.ZeroRefreshRateEnabled;
            this.hupZrrOnRadioButton.Checked = userBasedRefreshRateFeatures.ZeroRefreshRateEnabled;
            this.hupIgnoreUpNoRadioButton.Checked = !userBasedRefreshRateFeatures.IgnoreUserPresenceStatus;
            this.hupIgnoreUpYesRadioButton.Checked = userBasedRefreshRateFeatures.IgnoreUserPresenceStatus;
            this.hupSetOsDimLevelOffRadioButton.Checked = !userBasedRefreshRateFeatures.OverrideVideoDimLevel;
            this.hupSetOsDimLevelOnRadioButton.Checked = userBasedRefreshRateFeatures.OverrideVideoDimLevel;

            AddControlEventHandlers();

            UpdateWoaControlsVisibility(this.woaCheckBox.Checked);
        }

        internal void UpdateControlSettings(TouchPauseScanFeatures touchPauseScanFeatures)
        {
            RemoveControlEventHandlers();
            this.hupTpsOffRadioButton.Checked = !touchPauseScanFeatures.TouchPauseScanEnabled;
            this.hupTpsOnRadioButton.Checked = touchPauseScanFeatures.TouchPauseScanEnabled;

            AddControlEventHandlers();
            UpdateWoaControlsVisibility(this.woaCheckBox.Checked);
        }

        internal void ClearFeatureCheckboxes()
        {
            this.nlopCheckBox.CheckedChanged -= checkBoxNlop_CheckedChanged;
            this.nlopCheckBox.CheckState = CheckState.Unchecked;
            this.nlopCheckBox.CheckedChanged += checkBoxNlop_CheckedChanged;
            UpdateNLoPControlsVisibility(false);

            this.walCheckBox.CheckedChanged -= checkBoxWAL_CheckedChanged;
            this.walCheckBox.CheckState = CheckState.Unchecked;
            this.walCheckBox.CheckedChanged += checkBoxWAL_CheckedChanged;
            UpdateWalControlsVisibility(false);

            this.woaCheckBox.CheckedChanged -= checkBoxWOA_CheckedChanged;
            this.woaCheckBox.CheckState = CheckState.Unchecked;
            this.woaCheckBox.CheckedChanged += checkBoxWOA_CheckedChanged;
            UpdateWoaControlsVisibility(false);

            this.adCheckBox.CheckedChanged -= checkBoxAd_CheckedChanged;
            this.adCheckBox.CheckState = CheckState.Unchecked;
            this.adCheckBox.CheckedChanged += checkBoxAd_CheckedChanged;
            UpdateAdControlsVisibility(false);

            this.onlookerDetectionCheckBox.CheckedChanged -= onlookerDetectionCheckBox_CheckedChanged;
            this.onlookerDetectionCheckBox.CheckState = CheckState.Unchecked;
            this.onlookerDetectionCheckBox.CheckedChanged += onlookerDetectionCheckBox_CheckedChanged;
            UpdateOnlookerDetectionControlsVisibility(false);

            this.privacyCheckbox.CheckedChanged -= privacyCheckBox_CheckedChanged;
            this.privacyCheckbox.CheckState = CheckState.Unchecked;
            this.privacyCheckbox.CheckedChanged += privacyCheckBox_CheckedChanged;

            this.monitorAppCheckbox.CheckedChanged -= monitorAppCheckbox_CheckedChanged;
            this.monitorAppCheckbox.CheckState = CheckState.Unchecked;
            this.monitorAppCheckbox.CheckedChanged += monitorAppCheckbox_CheckedChanged;
            UpdateAppFeatureControlsVisibility(this.testModeCheckbox.Checked);
        }

        internal void EnableFeatureCheckboxes(bool enabled = true)
        {
            this.nlopCheckBox.Enabled = enabled;
            this.walCheckBox.Enabled = enabled;
            this.woaCheckBox.Enabled = enabled;
            this.adCheckBox.Enabled = enabled;
            this.onlookerDetectionCheckBox.Enabled = enabled;
            this.privacyCheckbox.Enabled = enabled;
            this.monitorAppCheckbox.Enabled = enabled;
        }

        private async Task<bool> UpdateControlsViaGetOptions(FeatureType featureType)
        {
            bool getOptionsSucceeded = true;
            try
            {
                FeatureSetting featureSetting = await this._commClient.GetOptions(featureType, this.globalMode);
                UpdateControlSettings(featureSetting);
            }
            catch (ErrorException ee)
            {
                logger.Error(ee.ErrorObject.Description);
                MessageBox.Show(ee.ErrorObject.Description, "GetOptions Error", MessageBoxButtons.OK, MessageBoxIcon.Error);
                getOptionsSucceeded = false;
            }

            return getOptionsSucceeded;
        }

        internal async Task<bool> RefreshFeaturesViaGetOptions()
        {
            if (this._commClient is null)
            {
                logger.Debug("WCF client is null.");
                this._commClient = new ClientBackendLink(this.featureCallback);
            }

            bool success = await UpdateControlsViaGetOptions(FeatureType.LOCK);
            if (success)
            {
                success = await UpdateControlsViaGetOptions(FeatureType.WAKE);
            }

            if (success)
            {
                success = await UpdateControlsViaGetOptions(FeatureType.NOLOCKONPRESENCE);
            }

            if (success)
            {
                success = await UpdateControlsViaGetOptions(FeatureType.ADAPTIVEDIMMING);
            }

            if (success)
            {
                success = await UpdateControlsViaGetOptions(FeatureType.ONLOOKERDETECTION);
            }

            if (success)
            {
                success = await UpdateControlsViaGetOptions(FeatureType.APPLICATION);
            }

            if (success)
            {
                success = await UpdateControlsViaGetOptions(FeatureType.TOUCHPAUSESCAN);
            }

            if (success)
            {
                success = await UpdateControlsViaGetOptions(FeatureType.USERBASEDREFRESHRATE);
            }

            return success;
        }

        private async Task HumanPresenceEventHandler(CancellationToken cancellationToken)
        {
            while (await this.humanPresenceChannel.Reader.WaitToReadAsync(cancellationToken))
            {
                HumanPresenceStatusPacket nextEvent = await this.humanPresenceChannel.Reader.ReadAsync(cancellationToken);
                await this.humanPresenceUpdateLock.WaitAsync(cancellationToken);
                try
                {
                    UpdatePresenceStatusText(nextEvent.PlatformStatus.ToString(), nextEvent.PolicyStatus.ToString(),
                        nextEvent.SensorStatus.ToString());
                }
                catch (Exception ex)
                {
                    logger.LogError($"Exception updating Human Presence text: {ex}");
                }
                finally
                {
                    this.humanPresenceUpdateLock.Release();
                }
            }
        }

        public void UpdatePresenceStatusText(string platformStatus, string policyStatus, string sensorStatus)
        {
            this.PlatformUserPresenceValueLabel.Text = platformStatus;
            this.PolicyUserPresenceValueLabel.Text = policyStatus;
            this.SensorUserPresenceValueLabel.Text = sensorStatus;
        }

        private async void UpdatePresenceStatus()
        {
            if (this._commClient is null)
            {
                logger.Debug("CST client is null.");
                this._commClient = new ClientBackendLink(this.featureCallback);
            }

            await this.humanPresenceUpdateLock.WaitAsync();
            try
            {
                UpdatePresenceStatusText(
                    (await this._commClient.GetHumanPresenceStatus(HumanPresenceStatusType.PLATFORM)).ToString(),
                    (await this._commClient.GetHumanPresenceStatus(HumanPresenceStatusType.POLICY)).ToString(),
                    (await this._commClient.GetHumanPresenceStatus(HumanPresenceStatusType.SENSOR)).ToString());
            }
            catch (ErrorException ee)
            {
                logger.Error(ee.ErrorObject.Description);
                UpdatePresenceStatusText(DATA_CONNECTION_ERROR_TEXT, DATA_CONNECTION_ERROR_TEXT,
                    DATA_CONNECTION_ERROR_TEXT);
            }
            catch (Exception ex)
            {
                logger.Error($"Error getting the human presence status: {ex}");
                UpdatePresenceStatusText(DATA_CONNECTION_ERROR_TEXT, DATA_CONNECTION_ERROR_TEXT,
                    DATA_CONNECTION_ERROR_TEXT);
            }
            finally
            {
                this.humanPresenceUpdateLock.Release();
            }
        }

        internal async void privacyCheckBox_CheckedChanged(object sender, EventArgs e)
        {
            if (sender is CheckBox)
            {
                if (((CheckBox)sender).CheckState == CheckState.Checked)
                {
                    try
                    {
                        await this._commClient.SetOption(FeatureType.APPLICATION, FeatureProperty.PrivacyEnabled, true, this.globalMode);
                    }
                    catch (Exception ex)
                    {
                        this.privacyCheckbox.CheckedChanged -= privacyCheckBox_CheckedChanged;
                        this.privacyCheckbox.CheckState = CheckState.Unchecked;
                        this.privacyCheckbox.CheckedChanged += privacyCheckBox_CheckedChanged;
                        logger.Error("Error enabling privacy" + ex.Message);
                        MessageBox.Show(ex.Message, "OnError", MessageBoxButtons.OK, MessageBoxIcon.Error);
                    }
                }
                else
                {
                    try
                    {
                        await this._commClient.SetOption(FeatureType.APPLICATION, FeatureProperty.PrivacyEnabled, false, this.globalMode);
                    }
                    catch (Exception ex)
                    {
                        this.privacyCheckbox.CheckedChanged -= privacyCheckBox_CheckedChanged;
                        this.privacyCheckbox.CheckState = CheckState.Unchecked;
                        this.privacyCheckbox.CheckedChanged += privacyCheckBox_CheckedChanged;
                        logger.Error("Error disabling privacy" + ex.Message);
                        MessageBox.Show(ex.Message, "OnError", MessageBoxButtons.OK, MessageBoxIcon.Error);
                    }
                }
            }
        }

        private void adHonorPowerRequestOnRadioButton_CheckedChanged(object sender, EventArgs e)
        {
            if (sender is RadioButton && ((RadioButton)sender).Checked)
            {
                Task setTask = this._commClient.SetOption(FeatureType.ADAPTIVEDIMMING, FeatureProperty.PowerRequest, true, this.globalMode);
                HandleSetOptionExceptions(setTask);
            }
        }

        private void adHonorPowerRequestOffRadioButton_CheckedChanged(object sender, EventArgs e)
        {
            if (sender is RadioButton && ((RadioButton)sender).Checked)
            {
                Task setTask = this._commClient.SetOption(FeatureType.ADAPTIVEDIMMING, FeatureProperty.PowerRequest, false, this.globalMode);
                HandleSetOptionExceptions(setTask);
            }
        }

        private void enabledEventsListBox_SelectedIndexChanged(object sender, EventArgs e)
        {
            if (sender is ListBox)
            {
                var castSender = (ListBox)sender;
                List<string> requestedEvents = GetSubscribedEvents(castSender.SelectedItems);
                Task setTask = this._commClient.SetOption(FeatureType.APPLICATION, FeatureProperty.EnabledEvents, requestedEvents, this.globalMode);
                HandleSetOptionExceptions(setTask);

                // Reset subscribable displayed data to uninitialized values
                if (!requestedEvents.Contains(EventTypeMap.EventSubscriptionNames[EventType.EVENT_CST_EXTENDED_CAPABILITY_CHANGED]))
                {
                    this.activeSensorTypeValLabel.Text = DATA_UNAVAILABLE_TEXT;
                    this.correlationStatusValueLabel.Text = DATA_UNAVAILABLE_TEXT;
                    this.visionCapableValLabel.Text = DATA_UNAVAILABLE_TEXT;
                    this.mispredictionCountValLabel.Text = DATA_UNAVAILABLE_TEXT;
                    this.AttentionCapableValueLabel.Text = DATA_UNAVAILABLE_TEXT;
                    this.odSensorStatusVallabel.Text = DATA_UNAVAILABLE_TEXT;
                    this.zeroRefreshRateCapableValLabel.Text = DATA_UNAVAILABLE_TEXT;
                    this.lowRefreshRateCapableValLabel.Text = DATA_UNAVAILABLE_TEXT;
                    this.touchPauseScanCapableValLabel.Text = DATA_UNAVAILABLE_TEXT;

                    if (!requestedEvents.Contains(EventTypeMap.EventSubscriptionNames[EventType.EVENT_BIOMETRIC_SENSOR_STATUS_CHANGED]))
                    {
                        this.biometricPresenceSensorStatusValueLabel.Text = DATA_UNAVAILABLE_TEXT;
                    }
                }

                if (!requestedEvents.Contains(EventTypeMap.EventSubscriptionNames[EventType.PROXIMITY]))
                {
                    this.proximityDistanceValueLabel.Text = DATA_UNAVAILABLE_TEXT;
                }

                if (!requestedEvents.Contains(EventTypeMap.EventSubscriptionNames[EventType.EVENT_HUMAN_PRESENCE_CHANGED]))
                {
                    UpdatePresenceStatusText(DATA_UNAVAILABLE_TEXT, DATA_UNAVAILABLE_TEXT, DATA_UNAVAILABLE_TEXT);
                }
            }
        }

        private async void monitorAppCheckbox_CheckedChanged(object sender, EventArgs e)
        {
            if (sender is CheckBox)
            {
                bool newValue = ((CheckBox)sender).CheckState == CheckState.Checked;
                try
                {
                    await this._commClient.SetOption(FeatureType.APPLICATION, FeatureProperty.MonitorApp, newValue, this.globalMode);
                }
                catch (Exception ex)
                {
                    this.monitorAppCheckbox.CheckedChanged -= monitorAppCheckbox_CheckedChanged;
                    this.monitorAppCheckbox.CheckState = CheckState.Unchecked;
                    this.monitorAppCheckbox.CheckedChanged += monitorAppCheckbox_CheckedChanged;
                    logger.Error("Error " + (newValue ? "enabling" : "disabling") + " Monitor App setting: " + ex.Message);
                    MessageBox.Show(ex.Message, "OnError", MessageBoxButtons.OK, MessageBoxIcon.Error);
                }
            }
        }

        private void testModeCheckbox_CheckedChanged(object sender, EventArgs e)
        {
            if (sender is CheckBox)
            {
                // "Test Mode" controls the enable state of all app feature selections.
                UpdateAppFeatureControlsVisibility(((CheckBox)sender).CheckState == CheckState.Checked);
            }
        }

        internal async Task RefreshFeatureCapability(FeatureType featureType, EventType eventType)
        {
            if (this._commClient is null)
            {
                logger.Debug("Application client is null.");
                this._commClient = new ClientBackendLink(this.featureCallback);
            }

            try
            {
                FeatureCapability featureCapability = await this._commClient.GetFeatureCapability(featureType);

                if (featureCapability is null)
                {
                    logger.Debug($"Received null service capability");
                }
                else
                {
                    logger.Debug($"Updating feature cues for {featureType}");
                    _ = Task.Run(() => SampleAppProgram.walwoaForm.UpdateFeaturesCues(featureType, featureCapability, eventType));
                }
            }
            catch (ErrorException ee)
            {
                logger.Error(ee.ErrorObject.Description);
                MessageBox.Show(ee.ErrorObject.Description, "GetFeatureCapability Error", MessageBoxButtons.OK, MessageBoxIcon.Error);
            }

            return;
        }

        internal async Task GetAllFeatureCapabilities(IList<FeatureType> featuresToExclude = null)
        {
            if (this._commClient is null)
            {
                logger.Debug("Application client is null.");
                this._commClient = new ClientBackendLink(this.featureCallback);
            }

            try
            {
                IDictionary<FeatureType, FeatureCapability> featureCapabilities = await this._commClient.GetAllFeatureCapabilities();
                if (featureCapabilities is null)
                {
                    logger.Debug($"Obtained null response from the service");
                }
                else
                {
                    logger.Debug($"Received feature capabilities for {featureCapabilities.Count} features");
                    foreach (KeyValuePair<FeatureType, FeatureCapability> capability in featureCapabilities)
                    {
                        if (!(featuresToExclude?.Contains(capability.Key)).GetValueOrDefault(false))
                        {
                            logger.Debug($"Updating feature cues for {capability.Key}");
                            _ = Task.Run(() => UpdateFeaturesCues(capability.Key, capability.Value, EventType.EVENT_CST_CAPABILITY_CHANGED));
                            _ = Task.Run(() => UpdateFeaturesCues(capability.Key, capability.Value, EventType.EVENT_CST_EXTENDED_CAPABILITY_CHANGED));
                        }
                    }
                }
            }
            catch (ErrorException ee)
            {
                logger.Error(ee.ErrorObject.Description);
                MessageBox.Show(ee.ErrorObject.Description, "GetFeatureCapability Error", MessageBoxButtons.OK, MessageBoxIcon.Error);
            }
        }

        internal Task UpdateFeaturesCues(FeatureType featureType, FeatureCapability featureCapability, EventType eventType)
        {
            if (eventType == EventType.EVENT_CST_CAPABILITY_CHANGED)
            {
                UpdateFeatureCapableCue(featureType, featureCapability.Capable);
            }
            else if (eventType == EventType.EVENT_CST_EXTENDED_CAPABILITY_CHANGED)
            {
                if (featureCapability.ExtendedStatus != null)
                {
                    UpdateExtendedCapabilitiesCue(featureType, featureCapability.ExtendedStatus);
                }
            }

            return Task.CompletedTask;
        }

        private void UpdateFeatureCapableCue(FeatureType featureType, bool featureCapable)
        {
            // When the APPLICATION is not capable, the other values are a don't care, so we'll display them as not capable
            System.Drawing.Color textColor = featureCapable && this.appEnabledCheckBox.Checked ? System.Drawing.Color.Green : System.Drawing.Color.Black;
            switch (featureType)
            {
                case FeatureType.LOCK:
                    {
                        this.walLabel.ForeColor = textColor;
                    }
                    break;
                case FeatureType.WAKE:
                    {
                        this.woaLabel.ForeColor = textColor;
                    }
                    break;
                case FeatureType.NOLOCKONPRESENCE:
                    {
                        this.nlopLabel.ForeColor = textColor;
                    }
                    break;
                case FeatureType.ONLOOKERDETECTION:
                    {
                        this.onlookerDetectionLabel.ForeColor = textColor;
                    }
                    break;
                case FeatureType.ADAPTIVEDIMMING:
                    {
                        this.adLabel.ForeColor = textColor;
                    }
                    break;
                case FeatureType.TOUCHPAUSESCAN:
                    {
                        // For backwards compatibility reasons, this information is also still available in AD's status (see below)
                        this.touchPauseScanCapableValLabel.Text = featureCapable ? "Yes" : "No";
                    }
                    break;
                case FeatureType.USERBASEDREFRESHRATE:
                    {
                        logger.Info("Overall UBRR capability", featureCapable);
                    }
                    break;
                case FeatureType.APPLICATION:
                    {
                        if (featureCapable)
                        {
                            this.appEnabledCheckBox.Checked = true;

                            // When the APPLICATION feature becomes capable, the other capabilities stop
                            // being a don't care, so we should now refresh all other capabilities.
                            _ = GetAllFeatureCapabilities(new FeatureType[] { FeatureType.APPLICATION });
                        }
                        else
                        {
                            this.walLabel.ForeColor = textColor;
                            this.woaLabel.ForeColor = textColor;
                            this.nlopLabel.ForeColor = textColor;
                            this.onlookerDetectionLabel.ForeColor = textColor;
                            this.adLabel.ForeColor = textColor;
                            this.appEnabledCheckBox.Checked = false;
                        }
                    }
                    break;
            }
        }

        private void UpdateExtendedCapabilitiesCue(FeatureType featureType, IDictionary<string, string> extendedStatus)
        {
            switch (featureType)
            {
                case FeatureType.ONLOOKERDETECTION:
                    {
                        if (extendedStatus.TryGetValue("SensorStatus", out string sensorStatus))
                        {
                            this.odSensorStatusVallabel.Text = sensorStatus;
                        }

                        if (extendedStatus.TryGetValue("SnoozeActive", out string currentSnoozeState))
                        {
                            if (Convert.ToBoolean(currentSnoozeState))
                            {
                                this.odCancelSnoozeButton.Enabled = true;
                                this.snoozeRequested = true;
                            }
                            else
                            {
                                this.odCancelSnoozeButton.Enabled = false;
                                this.snoozeRequested = false;
                            }
                        }

                        if (!this.CheckedInitialOnlookerState)
                        {
                            // Only run this code once - once we have checked the initial state, we can just use the events
                            this.CheckedInitialOnlookerState = true;
                            if (extendedStatus.TryGetValue("OnlookerDetected", out string onlookerDetected))
                            {
                                if (Convert.ToBoolean(onlookerDetected) && !this.odCancelSnoozeButton.Enabled)
                                {
                                    ShowOnlookerPopup();
                                }
                            }
                        }
                    }
                    break;
                case FeatureType.ADAPTIVEDIMMING:
                    {
                        if (extendedStatus.TryGetValue("MispredictionCount", out string mispredictionCount))
                        {
                            this.mispredictionCountValLabel.Text = mispredictionCount;
                        }

                        if (extendedStatus.TryGetValue("TouchPauseScanCapable", out string touchPauseScanCapable))
                        {
                            logger.Info("AD reported TPS capability", touchPauseScanCapable);
                        }

                        if (!this.CheckedInitialAdaptiveDimmingState)
                        {
                            // Only run this code once - once we have checked the initial state, we can just use the events
                            this.CheckedInitialAdaptiveDimmingState = true;
                            if (extendedStatus.TryGetValue("AdaptiveDimmingActive", out string adaptiveDimmingActive))
                            {
                                if (Convert.ToBoolean(adaptiveDimmingActive))
                                {
                                    ShowAdaptiveDimmingPopup(true);
                                }
                            }

                            if (extendedStatus.TryGetValue("ZeroRefreshRateCapable", out string zeroRefreshRateCapable))
                            {
                                logger.Info("AD reported ZRR capability", zeroRefreshRateCapable);
                            }

                            if (extendedStatus.TryGetValue("LowRefreshRateCapable", out string lowRefreshRateCapable))
                            {
                                logger.Info("AD reported LRR capability", lowRefreshRateCapable);
                            }
                        }
                    }
                    break;
                case FeatureType.USERBASEDREFRESHRATE:
                    {
                        // For backwards compatibility reasons, this information is also still available in AD's status (see above)
                        if (extendedStatus.TryGetValue("ZeroRefreshRateCapable", out string zeroRefreshRateCapable))
                        {
                            this.zeroRefreshRateCapableValLabel.Text = bool.Parse(zeroRefreshRateCapable) ? "Yes" : "No";
                        }

                        if (extendedStatus.TryGetValue("LowRefreshRateCapable", out string lowRefreshRateCapable))
                        {
                            this.lowRefreshRateCapableValLabel.Text = bool.Parse(lowRefreshRateCapable) ? "Yes" : "No";
                        }
                    }
                    break;
                case FeatureType.APPLICATION:
                    {
                        if (extendedStatus.TryGetValue("ActiveBiometricPresenceSensor", out string activeSensorType))
                        {
                            this.activeSensorTypeValLabel.Text = activeSensorType;
                        }

                        if (extendedStatus.TryGetValue("CorrelationStatus", out string correlationStatus))
                        {
                            this.correlationStatusValueLabel.Text = bool.Parse(correlationStatus) ? "Positive" : "Negative";
                        }

                        if (extendedStatus.TryGetValue("ActiveBiometricPresenceSensorStatus", out string activeSensorStatus))
                        {
                            this.biometricPresenceSensorStatusValueLabel.Text = activeSensorStatus;
                        }

                        if (extendedStatus.TryGetValue("VisionCapable", out string visionCapableString))
                        {
                            this.visionCapable = bool.Parse(visionCapableString);
                            this.visionCapableValLabel.Text = this.visionCapable ? "Yes" : "No";
                        }

                        if (extendedStatus.TryGetValue("IpfConnectionActive", out string ipfConnectionActive))
                        {
                            SetIpfConnectionState(bool.Parse(ipfConnectionActive));
                        }

                        if (extendedStatus.TryGetValue("FaceIdCapable", out string faceIdCapable))
                        {
                            this.FaceIdCapableValueLabel.Text = faceIdCapable;
                        }

                        if (extendedStatus.TryGetValue("FaceIdEnrolled", out string faceIdEnrolled))
                        {
                            this.FaceIdEnrolledValueLabel.Text = faceIdEnrolled;
                        }

                        if (extendedStatus.TryGetValue("AttentionCapable", out string attentionCapable))
                        {
                            this.AttentionCapableValueLabel.Text = attentionCapable;
                        }

                        if (extendedStatus.TryGetValue("OsHumanPresenceFeatureConflictDetected", out string osHumanPresenceFeatureConflictDetected))
                        {
                            this.HuPExtensionsLabel.ForeColor = bool.Parse(osHumanPresenceFeatureConflictDetected) ? System.Drawing.Color.Green : System.Drawing.Color.Black;
                        }

                    }
                    break;
            }
        }

        internal async Task SnoozeOnlookerDetection(int snoozeVal)
        {
            if (this._commClient is null)
            {
                logger.Debug("Application client is null.");
                this._commClient = new ClientBackendLink(this.featureCallback);
            }

            try
            {
                this.snoozeRequested = false;
                string commandString = CommandType.set_value.ToString() + " " + RequestType.ONLOOKER_SNOOZE.ToString();
                string result = await this._commClient.ExecuteCommand(commandString, snoozeVal);
                this.snoozeRequested = Newtonsoft.Json.JsonConvert.DeserializeObject<bool>(result);
                if (this.snoozeRequested)
                {
                    this.odCancelSnoozeButton.Enabled = true;
                }
            }
            catch (ErrorException ee)
            {
                logger.Error(ee.ErrorObject.Description);
                MessageBox.Show(ee.ErrorObject.Description, "SnoozeOnlookerDetection Error", MessageBoxButtons.OK, MessageBoxIcon.Error);
            }
            catch (Exception ee)
            {
                logger.Error(ee.Message);
                MessageBox.Show(ee.Message, "SnoozeOnlookerDetection Error", MessageBoxButtons.OK, MessageBoxIcon.Error);
            }
        }

        public void HandleOnlookerClear(string message)
        {
            new Thread(() =>
            {
                MessageBox.Show($"Onlooker {message}", "Onlooker Presence Changed", MessageBoxButtons.OK);
            }).Start();
        }

        public void HandleSnoozeComplete()
        {
            this.snoozeRequested = false;
            this.odCancelSnoozeButton.Enabled = false;
        }

        private void odCancelSnoozeButton_Click(object sender, EventArgs e)
        {
            if (this.snoozeRequested)
            {
                _ = Task.Run(async () => await this._commClient.SnoozeOnlookerDetection(0));
            }
            else
            {
                new Thread(() =>
                {
                    MessageBox.Show("Onlooker Detection snooze is not currently active", "Snooze Onlooker Detection", MessageBoxButtons.OK);
                }).Start();
            }
        }

        public void ShowOnlookerPopup()
        {
            new Thread(() =>
            {
                var onlookerPopup = new OnlookerDetectedForm();
                DialogResult okVal = onlookerPopup.ShowDialog();

                if (okVal == DialogResult.OK)
                {
                    int snoozeVal = onlookerPopup.SnoozeTime;
                    new Thread(() =>
                    {
                        SampleAppProgram.walwoaForm.Invoke(new MethodInvoker(delegate
                        {
                            _ = Task.Run(async () => await SampleAppProgram.walwoaForm.SnoozeOnlookerDetection(snoozeVal));
                        }));
                    }).Start();
                }

                onlookerPopup.Dispose();
            }).Start();
        }

        public void ShowAdaptiveDimmingPopup(bool adaptiveDimmingActive)
        {
            string message = adaptiveDimmingActive ? "started" : "stopped";

            new Thread(() =>
            {
                MessageBox.Show($"Adaptive dimming {message}", "Adaptive Dimming", MessageBoxButtons.OK);
            }).Start();
        }

        public void SetIpfConnectionState(bool state)
        {
            this.ipfConnectionActiveCheckBox.Checked = state;
        }

        private void humanPresencePollCheckbox_CheckedChanged(object sender, EventArgs e)
        {
            if (sender is CheckBox pollCheckbox)
            {
                if (pollCheckbox.Checked)
                {
                    StartHumanPresenceUpdateTask();
                }
                else
                {
                    StopHumanPresenceUpdateTask();
                }
            }
        }

        private void correlationResetTimeComboBox_SelectedIndexChanged(object sender, EventArgs e)
        {
            if (sender is ComboBox && ((ComboBox)sender).SelectedIndex < this.correlationResetSeconds.Length)
            {
                Task setTask = this._commClient.SetOption(FeatureType.APPLICATION, FeatureProperty.CorrelationResetTime, this.correlationResetSeconds[((ComboBox)sender).SelectedIndex] * 1000, this.globalMode);
                HandleSetOptionExceptions(setTask);
            }
        }

        private void proximityCooldownComboBox_SelectedIndexChanged(object sender, EventArgs e)
        {
            if (sender is ComboBox && ((ComboBox)sender).SelectedIndex < this.proximityCooldownSeconds.Length)
            {
                Task setTask = this._commClient.SetOption(FeatureType.APPLICATION, FeatureProperty.ProximityCooldownTime, this.proximityCooldownSeconds[((ComboBox)sender).SelectedIndex] * 1000, this.globalMode);
                HandleSetOptionExceptions(setTask);
            }
        }

        private void hidActiveDelayComboBox_SelectedIndexChanged(object sender, EventArgs e)
        {
            if (sender is ComboBox && ((ComboBox)sender).SelectedIndex < this.hidActiveDelaySeconds.Length)
            {
                Task setTask = this._commClient.SetOption(FeatureType.APPLICATION, FeatureProperty.HidActiveDelay, this.hidActiveDelaySeconds[((ComboBox)sender).SelectedIndex] * 1000, this.globalMode);
                HandleSetOptionExceptions(setTask);
            }
        }

        private void UpdateIgnoreHidInput(bool value)
        {
            this.ignoreHidInputRadioOn.Checked = value;
            this.ignoreHidInputRadioOff.Checked = !value;
        }

        private void ignoreHidOnExternalMonitorCheckBox_CheckedChanged(object sender, EventArgs e)
        {
            if (sender is CheckBox hidCheckBox)
            {
                this.ignoreHidInputRadioOn.Enabled = !hidCheckBox.Checked;
                this.ignoreHidInputRadioOff.Enabled = !hidCheckBox.Checked;
                if (hidCheckBox.Checked)
                {
                    logger.LogInformation("Enabling external monitor/HID interaction, HID setting control will begin after the next external monitor event.");
                }
                else
                {
                    logger.LogInformation("Disabling external monitor/HID interaction, requesting CST to use HID input for Adaptive Dimming");
                    UpdateIgnoreHidInput(false);
                }
            }
        }

        public void HandleExternalMonitorEvent(bool externalMonitorConnected)
        {
            if (this.ignoreHidOnExternalMonitorCheckBox.Checked)
            {
                logger.LogInformation("External monitor is now {ConnectionStatus}, requesting CST to {HidEnableStatus} HID input for Adaptive Dimming", externalMonitorConnected ? "connected" : "disconnected", externalMonitorConnected ? "ignore" : "use");
                UpdateIgnoreHidInput(externalMonitorConnected);
            }
            else
            {
                logger.LogTrace("AD ignoring external monitor event because ignore HID on external monitor is disabled");
            }
        }

        private void ignoreHidInputRadioOn_CheckedChanged(object sender, EventArgs e)
        {
            if (sender is RadioButton button && button.Checked)
            {
                Task setTask = this._commClient.SetOption(FeatureType.ADAPTIVEDIMMING, FeatureProperty.IgnoreHidInput, true, this.globalMode);
                HandleSetOptionExceptions(setTask);
            }
        }

        private void ignoreHidInputRadioOff_CheckedChanged(object sender, EventArgs e)
        {
            if (sender is RadioButton button && button.Checked)
            {
                Task setTask = this._commClient.SetOption(FeatureType.ADAPTIVEDIMMING, FeatureProperty.IgnoreHidInput, false, this.globalMode);
                HandleSetOptionExceptions(setTask);
            }
        }

        private void touchPauseScanEnableRadioButton_CheckedChanged(object sender, EventArgs e)
        {
            if (sender is RadioButton button && button.Checked)
            {
                Task setTask = this._commClient.SetOption(FeatureType.ADAPTIVEDIMMING, FeatureProperty.TouchPauseScanEnabled, true, this.globalMode);
                HandleSetOptionExceptions(setTask);
            }
        }

        private void touchPauseScanDisableRadioButton_CheckedChanged(object sender, EventArgs e)
        {
            if (sender is RadioButton button && button.Checked)
            {
                Task setTask = this._commClient.SetOption(FeatureType.ADAPTIVEDIMMING, FeatureProperty.TouchPauseScanEnabled, false, this.globalMode);
                HandleSetOptionExceptions(setTask);
            }
        }

        private void hupLrrOnRadioButton_CheckedChanged(object sender, EventArgs e)
        {
            if (sender is RadioButton button && button.Checked)
            {
                Task setTask = this._commClient.SetOption(FeatureType.USERBASEDREFRESHRATE, FeatureProperty.LowRefreshRateEnabled, true, this.globalMode);
                HandleSetOptionExceptions(setTask);
            }
        }

        private void hupLrrOffRadioButton_CheckedChanged(object sender, EventArgs e)
        {
            if (sender is RadioButton button && button.Checked)
            {
                Task setTask = this._commClient.SetOption(FeatureType.USERBASEDREFRESHRATE, FeatureProperty.LowRefreshRateEnabled, false, this.globalMode);
                HandleSetOptionExceptions(setTask);
            }
        }

        private void hupZrrOnRadioButton_CheckedChanged(object sender, EventArgs e)
        {
            if (sender is RadioButton button && button.Checked)
            {
                Task setTask = this._commClient.SetOption(FeatureType.USERBASEDREFRESHRATE, FeatureProperty.ZeroRefreshRateEnabled, true, this.globalMode);
                HandleSetOptionExceptions(setTask);
            }
        }

        private void hupZrrOffRadioButton_CheckedChanged(object sender, EventArgs e)
        {
            if (sender is RadioButton button && button.Checked)
            {
                Task setTask = this._commClient.SetOption(FeatureType.USERBASEDREFRESHRATE, FeatureProperty.ZeroRefreshRateEnabled, false, this.globalMode);
                HandleSetOptionExceptions(setTask);
            }
        }

        private void hupTpsOnRadioButton_CheckedChanged(object sender, EventArgs e)
        {
            if (sender is RadioButton button && button.Checked)
            {
                Task setTask = this._commClient.SetOption(FeatureType.TOUCHPAUSESCAN, FeatureProperty.TouchPauseScanEnabled, true, this.globalMode);
                HandleSetOptionExceptions(setTask);
            }
        }

        private void hupTpsOffRadioButton_CheckedChanged(object sender, EventArgs e)
        {
            if (sender is RadioButton button && button.Checked)
            {
                Task setTask = this._commClient.SetOption(FeatureType.TOUCHPAUSESCAN, FeatureProperty.TouchPauseScanEnabled, false, this.globalMode);
                HandleSetOptionExceptions(setTask);
            }
        }

        private void hupSetOsDimLevelOffRadioButton_CheckedChanged(object sender, EventArgs e)
        {
            if (sender is RadioButton button && button.Checked)
            {
                Task setTask = this._commClient.SetOption(FeatureType.USERBASEDREFRESHRATE, FeatureProperty.OverrideVideoDimLevel, false, this.globalMode);
                HandleSetOptionExceptions(setTask);
            }
        }

        private void hupSetOsDimLevelOnRadioButton_CheckedChanged(object sender, EventArgs e)
        {
            if (sender is RadioButton button && button.Checked)
            {
                Task setTask = this._commClient.SetOption(FeatureType.USERBASEDREFRESHRATE, FeatureProperty.OverrideVideoDimLevel, true, this.globalMode);
                HandleSetOptionExceptions(setTask);
            }
        }

        private void onlookerDetectionUseFaceAttentionOnRadioButton_CheckedChanged(object sender, EventArgs e)
        {
            if (sender is RadioButton button && button.Checked)
            {
                Task setTask = this._commClient.SetOption(FeatureType.ONLOOKERDETECTION, FeatureProperty.UseFaceAttention, true, this.globalMode);
                HandleSetOptionExceptions(setTask);
            }
        }

        private void onlookerDetectionUseFaceAttentionOffRadioButton_CheckedChanged(object sender, EventArgs e)
        {
            if (sender is RadioButton button && button.Checked)
            {
                Task setTask = this._commClient.SetOption(FeatureType.ONLOOKERDETECTION, FeatureProperty.UseFaceAttention, false, this.globalMode);
                HandleSetOptionExceptions(setTask);
            }
        }

        private void hupIgnoreUpYesRadioButton_CheckedChanged(object sender, EventArgs e)
        {
            if (sender is RadioButton button && button.Checked)
            {
                Task setTask = this._commClient.SetOption(FeatureType.USERBASEDREFRESHRATE, FeatureProperty.IgnoreUserPresenceStatus, true, this.globalMode);
                HandleSetOptionExceptions(setTask);
            }
        }

        private void hupIgnoreUpNoRadioButton_CheckedChanged(object sender, EventArgs e)
        {
            if (sender is RadioButton button && button.Checked)
            {
                Task setTask = this._commClient.SetOption(FeatureType.USERBASEDREFRESHRATE, FeatureProperty.IgnoreUserPresenceStatus, false, this.globalMode);
                HandleSetOptionExceptions(setTask);
            }
        }
    }
}
