# <copyright>
# INTEL CONFIDENTIAL
#
# Copyright 2021 Intel Corporation
#
# This software and the related documents are Intel copyrighted materials, and your use of
# them is governed by the express license under which they were provided to you ("License").
# Unless the License provides otherwise, you may not use, modify, copy, publish, distribute,
# disclose or transmit this software or the related documents without Intel's prior written
# permission.
#
# This software and the related documents are provided as is, with no express or implied
# warranties, other than those that are expressly stated in the License.
#
# <copyright>

# Suppress irrelevant PS Script Analyzer warnings (trailing Param() is needed to help PSSA parse the file)
[Diagnostics.CodeAnalysis.SuppressMessageAttribute("PSAvoidUsingPositionalParameters", "", Scope="function")] Param()

# Functions common to all cmdlets
Function ValidateGetAdapterNameParams($AdapterName, $Adapters, [ref]$ErrorMessages)
{
    $AdapterNames = @()

    if ((-Not $Adapters) -and $AdapterName)
    {
        foreach ($n in $AdapterName)
        {
            $TmpAdapterNames = @()
            try
            {
                $PhysicalAdapterArray = $script:PnpDevice.Where({ $_.Name -Like $n })
                $IntelAdapterArray = $PhysicalAdapterArray.Where({ $_.Manufacturer -eq "Intel" })

                if ($IntelAdapterArray)
                {
                    $TmpAdapterNames = $IntelAdapterArray.Name
                }
                elseif ($PhysicalAdapterArray)
                {
                    # Add non-Intel devices for warning display
                    $TmpAdapterNames = $PhysicalAdapterArray.Name
                }
            }
            catch
            {
                # Failed due to Adapters passed to Name parameter
                $AdapterNames = @()
                $ErrorMessages.Value += $Messages.InvalidParams
                break
            }

            if (-Not $TmpAdapterNames)
            {
                $ErrorMessages.Value += $Messages.AdapterNotFound -f $n
                continue
            }
            $AdapterNames += $TmpAdapterNames
        }
    }
    elseif ((-Not $AdapterName) -and $Adapters)
    {
        foreach ($a in $Adapters)
        {
            if (CheckPropertyExists $a "CreationClassName")
            {
                if ($a.CreationClassName -eq "MSFT_NetAdapter")
                {
                    $AdapterNames += $a.InterfaceDescription
                }
                elseif ($a.CreationClassName -eq "Win32_NetworkAdapter" -or
                    $a.CreationClassName -eq "IANet_PhysicalEthernetAdapter" -or
                    $a.CreationClassName -eq "Win32_PnpEntity")
                {
                    $AdapterNames += $a.Name
                }
                else
                {
                    $ErrorMessages.Value += $Messages.InvalidObject -f $a.GetType().Name
                }
            }
            elseif ($null -ne $a.PSTypeNames -and $a.PSTypeNames[0] -eq "IntelEthernetAdapter")
            {
                $AdapterNames += $a.Name
            }
            else
            {
                $ErrorMessages.Value += $Messages.InvalidObject -f $a.GetType().Name
            }
        }
    }
    elseif (-Not ($AdapterName -and $Adapters))
    {
        $AdapterNames = ($script:PnpDevice.Where({ $_.Manufacturer -eq "Intel" })).Name
    }
    elseif ($AdapterName -and $Adapters)
    {
        $ErrorMessages.Value += $Messages.InvalidParamsAdapterAndName
    }

    return $AdapterNames
}

Function ValidateSetAdapterNameParams($AdapterName, $Adapters, [ref]$ErrorMessages)
{
    $AdapterNames = @()

    do
    {
        if ($AdapterName -and $Adapters)
        {
            $ErrorMessages.Value += $Messages.InvalidParamsAdapterAndName
            break
        }
        elseif ($AdapterName)
        {
            foreach ($n in $AdapterName)
            {
                $TmpAdapterNames = @()
                try
                {
                    $PhysicalAdapterArray = $script:PnpDevice.Where({ $_.Name -Like $n })
                    $IntelAdapterArray = $PhysicalAdapterArray.Where({ $_.Manufacturer -eq "Intel" })

                    if ($IntelAdapterArray)
                    {
                        $TmpAdapterNames = $IntelAdapterArray.Name
                    }
                    elseif ($PhysicalAdapterArray)
                    {
                        # Add non-Intel devices for warning display
                        $TmpAdapterNames = $PhysicalAdapterArray.Name
                    }
                }
                catch
                {
                    # Failed due to Adapters passed to Name parameter
                    $AdapterNames = @()
                    $ErrorMessages.Value += $Messages.InvalidParams
                    break
                }

                if (-Not $TmpAdapterNames)
                {
                    $ErrorMessages.Value += $Messages.AdapterNotFound -f $n
                    continue
                }
                $AdapterNames += $TmpAdapterNames
            }
        }
        elseif ($Adapters)
        {
            foreach ($a in $Adapters)
            {
                if (CheckPropertyExists $a "CreationClassName")
                {
                    if ($a.CreationClassName -eq "MSFT_NetAdapter")
                    {
                        $AdapterNames += $a.InterfaceDescription
                    }
                    elseif ($a.CreationClassName -eq "Win32_NetworkAdapter" -or
                        $a.CreationClassName -eq "IANet_PhysicalEthernetAdapter" -or
                        $a.CreationClassName -eq "Win32_PnpEntity")
                    {
                        $AdapterNames += $a.Name
                    }
                    else
                    {
                        $ErrorMessages.Value += $Messages.InvalidObject -f $a.GetType().Name
                    }
                }
                elseif ($null -ne $a.PSTypeNames -and $a.PSTypeNames[0] -eq "IntelEthernetAdapter")
                {
                    $AdapterNames += $a.Name
                }
                else
                {
                    $ErrorMessages.Value += $Messages.InvalidObject -f $a.GetType().Name
                }
            }
        }
        else
        {
            $ErrorMessages.Value += $Messages.InvalidParamsAdapterOrName
            break
        }
    } while ($false)

    return $AdapterNames
}

Function ValidatePathParams([ref]$LogPath, $UseDefaultPath, $LogName, [ref]$ErrorMessages)
{
    $Result = $true

    try
    {
        if ($UseDefaultPath)
        {
            $DefaultPath = $ENV:LOCALAPPDATA + "\Intel\Wired Networking\" + $LogName

            if (-not (Test-Path -Path $DefaultPath -ErrorAction Stop))
            {
                New-Item -Path $ENV:LOCALAPPDATA -Name "\Intel\Wired Networking" -ItemType "directory" -ErrorAction SilentlyContinue
            }
            $LogPath.Value = $DefaultPath
        }
        else
        {
            $LogPath.Value = $Path
        }

        $PathRoot = GetPSDriveRoot $LogPath.Value
        if (-Not [string]::IsNullOrEmpty($PathRoot))
        {
            $PathRoot = $PathRoot.TrimEnd("\")
            $strPathNoQualifier = Split-Path $LogPath.Value -NoQualifier
            $LogPath.Value = $PathRoot + $strPathNoQualifier
        }

        $isPathFile = Test-Path -Path $LogPath.Value -PathType Leaf -ErrorAction Stop

        if (($isPathFile) -and (-not $Append) -and (-not $Force))
        {
            $ErrorMessages.Value += $Messages.LogmanFileExists -f $AdapterName
            $Result = $false
        }
        elseif (-not $isPathFile)
        {
            if (Test-Path -Path $LogPath.Value -ErrorAction Stop)
            {
                $ErrorMessages.Value += $Messages.FolderFileNameExits
                $Result = $false
            }
            else
            {
                $strAbsolutePath = [IO.Path]::GetFullPath($LogPath.Value)
                $strParentFolder = Split-Path -Path $strAbsolutePath

                if (-Not (Test-Path -Path $strParentFolder -ErrorAction Stop))
                {
                    $ErrorMessages.Value += $Messages.PathIncorrect
                    $Result = $false
                }
            }
        }
    }
    catch
    {
        $ErrorMessages.Value += $Messages.PathIncorrect
        $Result = $false
    }
    return $Result
}

function GetPSDriveRoot($Path)
{
    $strQualifier = Split-Path -Path $Path -Qualifier -ErrorAction SilentlyContinue
    if (-Not [string]::IsNullOrEmpty($strQualifier))
    {
        $strPSDriveName = $strQualifier.TrimEnd(":")
        $CurrentPSDrive = Get-PSDrive -Name $strPSDriveName -ErrorAction SilentlyContinue
        if ($null -ne $CurrentPSDrive)
        {
            return $CurrentPSDrive.Root
        }
    }

    return $null
}

function InvokeCimMethod($ClassName, $InstanceName = "", $MethodName, $params = @{}, $Namespace = "root\wmi")
{
    $query = "Select * from $ClassName"
    if ($InstanceName)
    {
        $query += " where instancename like '$InstanceName'"
    }

    Invoke-CimMethod -Query $query -MethodName $MethodName -Arguments $params -Namespace $Namespace -ErrorAction SilentlyContinue
}

function GetIntelEthernetDevices($AdditionalDriverArray = $null)
{
    $script:PnpDevice = @(Get-PnpDevice | Where-Object { $_.Class -eq "Net" } -ErrorAction SilentlyContinue)
    if ($script:PnpDevice)
    {
        $script:SupportedAdapters = @($script:PnpDevice.Where({ $_.Service -in @('icea', 'scea', 'i40ea', 'i40eb') }))

        foreach ($AdditionalDriver in $AdditionalDriverArray)
        {
            $script:SupportedAdapters += $script:PnpDevice.Where({ $_.Service -like $AdditionalDriver })
        }
    }
}

function GetSupportedAdaptersByDriver([array]$strSupportedDriverArray)
{
    return $script:SupportedAdapters.Where({ $_.Service -in $strSupportedDriverArray })
}

function GetSupportedAdapters($AdapterNames, [ref]$WarningMessages)
{
    $SupportedAdapterNames = @()
    $AdapterNames = $AdapterNames | Sort-Object
    foreach ($a in $AdapterNames)
    {
        if ($script:SupportedAdapters.Name -Contains $a)
        {
            $SupportedAdapterNames += $a
        }
        else
        {
            $WarningMessages.Value += $Messages.NoCmdletSupport -f $a
        }
    }

    return $SupportedAdapterNames
}

Function GetOSSpecificDriverName($DriverName)
{
    $WIN10_RS5_BUILD = 17682
    $VER_NT_SERVER = 3

    $OSVersion = [Environment]::OSVersion.Version
    $iMajorVersion = $OSVersion.Major
    $iMinorVersion = $OSVersion.Minor
    $iBuildNumber = $OSVersion.Build

    $strOSSpecificDriverName = ''

    if (10 -eq $iMajorVersion -and 0 -eq $iMinorVersion)
    {
        if ($VER_NT_SERVER -eq ((Get-CimInstance Win32_OperatingSystem -ErrorAction SilentlyContinue).ProductType))
        {
            $bIsWin2016ServerAndHigher = $true
        }
    }

    if ($bIsWin2016ServerAndHigher)
    {
        if (IsWin2022ServerOrLater)
        {
            $strOSSpecificDriverName = $DriverName
        }
        elseif ($iBuildNumber -ge $WIN10_RS5_BUILD)
        {
            $strOSSpecificDriverName = $DriverName + "68"
        }
        else
        {
            $strOSSpecificDriverName = $DriverName + "65"
        }
    }

    return $strOSSpecificDriverName
}

function GetDriverInfParentFolder($strDriverName, $Verb)
{
    $strDriverStorePathArray = Get-WindowsDriver -Online -ErrorAction SilentlyContinue | Select-Object OriginalFileName

    if ($null -eq $strDriverStorePathArray)
    {
        switch ($Verb)
        {
            Start {$script:ErrorMessagesStart += $Messages.NoCmdletSupport -f $AdapterName; break}
            Register
            {
                $script:ErrorMessagesRegister += $Messages.RegisterCountersFailure -f $a
                Write-Verbose "LASTEXITCODE: $LASTEXITCODE"
                break
            }
        }
    }

    # The system keeps a history of installed drivers, Get the latest installed driver inf path
    $strPathToInf = Get-ChildItem -Path $strDriverStorePathArray.OriginalFileName.Where({ $_ -like "*$strDriverName*" }) | Sort-Object -Descending -Property "CreationTime" | Select-Object -First 1

    [System.IO.Path]::GetDirectoryName($strPathToInf)
}

function CheckDeviceError($AdapterName)
{
    $SupportedAdapter = $script:SupportedAdapters.Where({ $_.FriendlyName -eq $AdapterName })

    if ($SupportedAdapter)
    {
        # if the device is not 'working properly'
        if ([Int32]$SupportedAdapter.ConfigManagerErrorCode -ne 0)
        {
            $PreErrorActionPreference = $global:ErrorActionPreference
            $global:ErrorActionPreference = 'SilentlyContinue'
            # Workaround due to ProblemDescription being empty by default - change current path
            Push-Location -Path (Get-Module -Name PnPDevice).ModuleBase
            $StatusMsg = $AdapterName + ": " + $SupportedAdapter.ProblemDescription
            # Reset path
            Pop-Location
            $global:ErrorActionPreference = $PreErrorActionPreference
            $StatusMsg
        }
    }
}

function ValidateSingleAdapter([array]$PipelineInput, [array]$AdapterName, [ref]$ErrorMessages)
{
    $Result = $false

    do
    {
        if ($PipelineInput.Count -gt 1)
        {
            $ErrorMessages.Value += $Messages.InvalidParams
            break
        }

        if ($AdapterName.Count -gt 1)
        {
            $ErrorMessages.Value += $Messages.InvalidParams
            break
        }

        $Result = $true
    } while ($false)

    return $Result
}

function IsWin2022ServerOrLater()
{
    $WIN2022_BUILD = 20298
    $OSVersion = [Environment]::OSVersion.Version
    $iMajorVersion = $OSVersion.Major
    $iBuildNumber = $OSVersion.Build

    if (($iMajorVersion -eq 10) -and ($iBuildNumber -ge $WIN2022_BUILD) -and -not (IsOperatingSystemClientBased))
    {
        return $true
    }

    return $false
}

function IsOperatingSystemClientBased()
{
    $VER_NT_WORKSTATION = 1
    return ($VER_NT_WORKSTATION -eq ((Get-CimInstance Win32_OperatingSystem -ErrorAction SilentlyContinue).ProductType))
}

function CheckPropertyExists($TestObject, $strPropertyName)
{
    $bReturnValue = $true
    try
    {
        return (($TestObject | Get-Member -MemberType "Property" -ErrorAction Stop).Name -contains $strPropertyName)
    }
    catch
    {
        $bReturnValue = $false
    }
    return $bReturnValue
}

function GetAdapterSettings($AdapterName, $DisplayName, $RegistryKeyword)
{
    $AdapterSettings = @()
    $SettingsRet = @()
    if ([string]::IsNullOrEmpty($DisplayName) -and [string]::IsNullOrEmpty($RegistryKeyword))
    {
        $AdapterSettings = GetAllAdapterSettings $AdapterName
        $SettingsRet += InitializeProfileSetting $AdapterName
    }
    else
    {
        # Need to handle Profile differently, this is an EthernetCmdlets created setting
        if ($DisplayName -eq 'Profile' -or $RegistryKeyword -eq 'PerformanceProfile')
        {
            $SettingsRet += InitializeProfileSetting $AdapterName

            # Remove Profile from passed in array to avoid GatherSettingEnumOutput (InitializeProfileSetting handles it uniquely)
            if ($null -ne $DisplayName)
            {
                $DisplayName = $DisplayName -ne 'Profile'
            }
            if ($null -ne $RegistryKeyword)
            {
                $RegistryKeyword = $RegistryKeyword -ne 'PerformanceProfile'
            }
        }

        $AdapterSettings = GetSettings $AdapterName $DisplayName $RegistryKeyword
    }

    foreach ($Setting in $AdapterSettings)
    {
        switch ([int]($Setting.DisplayParameterType))
        {
            {$_ -in [int][DisplayParameterType]::int,
            [int][DisplayParameterType]::long,
            [int][DisplayParameterType]::word,
            [int][DisplayParameterType]::dword}
            { $SettingsRet += GatherSettingIntOutput $a $Setting; break }
            {$_ -in [int][DisplayParameterType]::enum,
            [int][DisplayParameterType]::edit}
            {$SettingsRet += GatherSettingEnumOutput $a $Setting; break}
            default {$SettingsRet += GatherSettingEnumOutput $a $Setting; break}
        }
    }

    return $SettingsRet
}

function GetAllAdapterSettings($AdapterName)
{
    return $script:MSNetAdvProperty.Where({$_.InterfaceDescription -eq $AdapterName})
}

function GetSettings($AdapterName, $DisplayName, $RegistryKeyword)
{
    $SettingArray = @()

    if (-not [string]::IsNullOrEmpty($DisplayName))
    {
        foreach ($TmpDisplayName in $DisplayName)
        {
            if ($DisplayName -eq 'Profile')
            {
                continue
            }
            $TmpSetting = $script:MSNetAdvProperty.Where({$_.InterfaceDescription -eq $AdapterName -and $_.DisplayName -like $TmpDisplayName})
            if (-not $TmpSetting)
            {
                $script:WarningMessagesGet += $Messages.InvalidSetting -f $AdapterName, $TmpDisplayName
            }
            else
            {
                $SettingArray += $TmpSetting
            }
        }
    }

    if (-not [string]::IsNullOrEmpty($RegistryKeyword))
    {
        foreach ($TmpRegistryKeyword in $RegistryKeyword)
        {
            if ($RegistryKeyword -eq 'PerformanceProfile')
            {
                continue
            }
            $TmpSetting = $script:MSNetAdvProperty.Where({$_.InterfaceDescription -eq $AdapterName -and $_.RegistryKeyword -like $TmpRegistryKeyword})
            if (-not $TmpSetting)
            {
                $script:WarningMessagesGet += $Messages.InvalidSetting -f $AdapterName, $TmpRegistryKeyword
            }
            else
            {
                $SettingArray += $TmpSetting
            }
        }
    }

    return $SettingArray
}

function GatherSettingEnumOutput($AdapterName, $Setting)
{
    $SettingMiniHelp = GetSettingMinihelp($Setting.RegistryKeyword)

    # Assemble it all together in PSCustomObject
    return [PsCustomObject] @{
        PSTypeName      = 'IntelEthernetSettingEnum'
        Caption         = $Setting.RegistryKeyword
        CurrentValue    = $Setting.RegistryValue
        DefaultValue    = $Setting.DefaultRegistryValue
        Description     = $Setting.DisplayName
        DescriptionMap  = $Setting.ValidDisplayValues
        DisplayName     = $Setting.DisplayName
        DisplayValue    = $Setting.DisplayValue
        Minihelp        = $SettingMiniHelp
        Name            = $AdapterName
        ParentId        = $Setting.InstanceID.split(':')[0]
        PossibleValues  = $Setting.ValidRegistryValues
        RegistryKeyword = $Setting.RegistryKeyword
        RegistryValue   = $Setting.RegistryValue
    }
}

function GatherSettingIntOutput($AdapterName, $Setting)
{
    $SettingMiniHelp = GetSettingMinihelp($Setting.RegistryKeyword)

    # Assemble it all together in PSCustomObject
    return [PsCustomObject] @{
        PSTypeName      = 'IntelEthernetSettingInt'
        Base            = $Setting.NumericParameterBaseValue
        Caption         = $Setting.RegistryKeyword
        CurrentValue    = $Setting.RegistryValue
        DefaultValue    = $Setting.DefaultRegistryValue
        Description     = $Setting.DisplayName
        DisplayName     = $Setting.DisplayName
        DisplayValue    = $Setting.DisplayValue
        Minihelp        = $SettingMiniHelp
        Name            = $AdapterName
        ParentId        = $Setting.InstanceID.split(':')[0]
        RegistryKeyword = $Setting.RegistryKeyword
        RegistryValue   = $Setting.RegistryValue
        Min             = $Setting.NumericParameterMinValue
        Max             = $Setting.NumericParameterMaxValue
        Step            = $Setting.NumericParameterStepValue
    }
}

function GetSettingMinihelp($RegistryKeyword)
{
    $SettingMiniHelp = $MiniHelp.$($RegistryKeyword)
    if ([string]::IsNullOrEmpty($SettingMiniHelp))
    {
        $SettingMiniHelp = $Messages.MiniHelpNotFound
    }

    return $SettingMiniHelp
}

function InitializeProfileSetting($AdapterName)
{
    # for our supported adapters, by default its safe to add: Standard Server, Web Server, and Low Latency
    # PossibleValues and DescriptionMap for Profile
    $ProfileValuesMap = @{ '2' = $Messages.StandardServer; '3' = $Messages.WebServer; '6' = $Messages.LowLatency }
    $DefaultValue = '2'

    $IntelDCBInstalled = $false
    $HyperVInstalled = $false

    $IntelDCB = Get-Service | Where-Object {$_.Name -eq "IntelDCB"}
    if ($null -ne $IntelDCB)
    {
        $IntelDCBInstalled = $true
    }

    # ID 20 represents Hyper-V
    $HyperV = Get-CimInstance -Namespace "root\cimv2" -ClassName "Win32_ServerFeature" | Where-Object {$_.ID -eq '20'}
    if ($null -ne $HyperV)
    {
        $HyperVInstalled = $true
    }

    if ($IntelDCBInstalled)
    {
        $ProfileValuesMap.Add('4', $Messages.StorageServer)
        $DefaultValue = '4'
    }
    if ($HyperVInstalled)
    {
        $ProfileValuesMap.Add('5', $Messages.VirtServer)
        $DefaultValue = '5'
    }
    if ($IntelDCBInstalled -and $HyperVInstalled)
    {
        $ProfileValuesMap.Add('7', $Messages.StorageVirt)
        $DefaultValue = '7'
    }

    $CurrentProfile = GetCurrentProfileValue $AdapterName
    $CurrentProfileRegistryValue = $null
    $CurrentProfileDisplayValue = $null
    if ($null -ne $CurrentProfile)
    {
        $CurrentProfileRegistryValue = $CurrentProfile.RegistryValue
        $CurrentProfileDisplayValue = $CurrentProfile.DisplayValue
        if ($CurrentProfileRegistryValue -eq 1)
        {
            $ProfileValuesMap.Add($CurrentProfileRegistryValue, $CurrentProfileDisplayValue)
        }
    }
    $ProfileValuesMap = $ProfileValuesMap.GetEnumerator() | Sort-Object -Property Name
    $SettingMiniHelp = GetSettingMinihelp("PerformanceProfile")
    $ParentId = $script:MSNetAdvProperty.Where({ $_.InterfaceDescription -eq $AdapterName}).InstanceID.split(':')[0]

    # Assemble it all together in PSCustomObject
    return [PsCustomObject] @{
        PSTypeName      = 'IntelEthernetSettingEnum'
        Caption         = 'PerformanceProfile'
        CurrentValue    = $CurrentProfile.RegistryValue
        DefaultValue    = $DefaultValue
        Description     = $Messages.Profile
        DescriptionMap  = $ProfileValuesMap.Value
        DisplayName     = $Messages.Profile
        DisplayValue    = $CurrentProfileDisplayValue
        Minihelp        = $SettingMiniHelp
        Name            = $AdapterName
        ParentId        = $ParentId
        PossibleValues  = $ProfileValuesMap.GetEnumerator().Name
        RegistryKeyword = 'PerformanceProfile'
        RegistryValue   = $CurrentProfileRegistryValue
    }
}

function GetCurrentProfileValue($AdapterName)
{
    $AdapterRegKey = GetAdapterPropertiesFromRegistry $AdapterName

    if ($null -ne $AdapterRegKey.PerformanceProfile)
    {
        $ProfileRegValue = $AdapterRegKey.PerformanceProfile
        # Profile is already custom settings or in case one of the associated Profile settings got changed
        if ($ProfileRegValue -eq '1' -or ($false -eq (IsProfileStillApplied $ProfileRegValue $AdapterName)))
        {
            $ProfileRegValue = '1'
            $ProfileDisplayVal = $Messages.CustomSettings
        }
        else
        {
            $ProfileDisplayVal = $PROFILE_VALUE_TO_NAME[$ProfileRegValue]
        }

        return [PsCustomObject] @{
            # return the key from the hashtable
            DisplayValue = $ProfileDisplayVal
            RegistryValue = $ProfileRegValue }
    }
}

function IsProfileStillApplied($RegistryValue, $AdapterName)
{
    $TmpProfileName = $PROFILE_VALUE_TO_NAME[$RegistryValue]
    if ($null -ne $TmpProfileName)
    {
        $DriverFamily = $script:SupportedAdapters.Where({ $_.Name -eq $AdapterName }).Service
        $tmp = GetProfileSettingsFromXml $TmpProfileName $DriverFamily
        foreach ($ProfileSetting in $tmp.GetEnumerator())
        {
            $TmpSetting = $script:MSNetAdvProperty.Where({$_.InterfaceDescription -eq $AdapterName -and $_.RegistryKeyword -eq $ProfileSetting.Key})
            if ($TmpSetting.RegistryValue -ne $ProfileSetting.Value)
            {
                return $false
            }
        }
    }

    return $true
}

function GetProfileSettingsFromXml()
{
    Param
    (
        [parameter(Mandatory=$true)]
        [ValidateNotNullOrEmpty()]
        [string]
        $ProfileToSet,

        [parameter(Mandatory=$true)]
        [ValidateNotNullOrEmpty()]
        [ValidateSet("i40ea", "i40eb", "icea", "scea")]
        [string]
        $AdapterFamily
    )

    $Profile_to_XmlProfile = @{
        "Standard Server" = "standard"
        "Web Server" = "web"
        "Storage Server" = "storage"
        "Virtualization Server (Hyper-V*)" = "hyperv"
        "Low Latency" = "lowlatency"
        "Storage + Virtualization" = "storagevirtualization"
    }

    $OSBuild = [Environment]::OSVersion.Version.Build

    # Each Key treated as a range of Build Numbers (14393 to 16299 is NDIS v6.60)
    $OSBuild_to_NDIS = [ordered]@{
        14393 = "ndis660"
        16299 = "ndis680"
    }

    $Adapter_to_ProfileString = @{
        i40ea = "40gig"
        i40eb = "40gig"
        icea  = "E8XX"
        scea  = "E8XX"
    }

    $Adapter_to_ProfileDeviceString = @{
        i40ea = "X710"
        i40eb = "X722"
    }

    [xml]$Xml = Get-Content -Path $PSScriptRoot\PerformanceProfiles.xml
    # validate XML against XSD before using it:
    try
    {
        $Xml.Schemas.Add('', "$PSScriptRoot\PerformanceProfiles.xsd") | Out-Null
        $Xml.Validate($null)
    }
    catch [System.Xml.Schema.XmlSchemaValidationException]
    {
        $script:ErrorMessagesProfile += $Messages.ProfileXmlError
        return
    }

    $PerfProfiles = $Xml.SelectNodes("//ProfileList/Profile")

    # Convert cmdlet parameters to strings from xml
    $AdapterFamilyProfile = $Adapter_to_ProfileString[$AdapterFamily]

    foreach ($CurrentBuildNumber in $OSBuild_to_NDIS.Keys)
    {
        if ($OSBuild -gt $CurrentBuildNumber)
        {
            $NDISVersion = $OSBuild_to_NDIS.$CurrentBuildNumber
        }
    }

    $ProfileName = $Profile_to_XmlProfile[$ProfileToSet]

    # Construct profile names
    # Top-level, eg 40gig_standard
    $TopLevelProfile = $AdapterFamilyProfile + "_" + $ProfileName
    Write-Verbose $TopLevelProfile

    # + NDIS version, eg 40gig_standard.ndis660
    $NdisProfile = $AdapterFamilyProfile + "_" + $ProfileName + "." + $NDISVersion
    Write-Verbose $NdisProfile

    # + device name, eg 40gig_standard.ndis660.X722
    # adding '*' at the end so it matches .no_npar too
    $DeviceProfile = $AdapterFamilyProfile + "_" + $ProfileName + "." + $NDISVersion + "." + $Adapter_to_ProfileDeviceString[$AdapterFamily] + "*"
    Write-Verbose $DeviceProfile

    $TopLevelProfile = $PerfProfiles | Where-Object { $_.Id -like  $TopLevelProfile }
    $NdisProfile     = $PerfProfiles | Where-Object { $_.Id -like  $NdisProfile }
    $DeviceProfile   = $PerfProfiles | Where-Object { $_.Id -like  $DeviceProfile }

    # assemble it all together - from generic to detailed profile
    # ie DeviceProfile settings should take precedence over those from TopLevelProfile
    $Profiles = @($TopLevelProfile, $NdisProfile, $DeviceProfile)
    $RetVal = @{}

    foreach ($Profile in $Profiles)
    {
        foreach($Setting in $Profile.Setting)
        {
            $RetVal[$Setting.id] = $Setting.'#text'
        }
    }

    return $RetVal
}

function GetAdapterPropertiesFromRegistry($AdapterName)
{
    # Individual Adapter GUID  - (Get-NetAdapter -InterfaceDescription $AdapterName | Where-Object {$_.InterfaceDescription -eq $AdapterName}).InterfaceGuid
    $AdapterInstanceID = ($script:MSNetAdapters.Where({$_.InterfaceDescription -eq $AdapterName})).InterfaceGuid
    $972Key = Get-ItemProperty -Path "HKLM:\SYSTEM\CurrentControlSet\Control\Class\{4d36e972-e325-11ce-bfc1-08002be10318}\*" -ErrorAction SilentlyContinue
    $AdapterRegKey = $972Key.Where({$_.NetCfgInstanceId -eq $AdapterInstanceID})
    return $AdapterRegKey
}

function SetAdapterPropertyInRegistry($AdapterName, $Property, $Value)
{
    $AdapterRegKey = GetAdapterPropertiesFromRegistry $AdapterName
    Set-ItemProperty -Path $AdapterRegKey.PSPath -Name $Property $Value -ErrorAction SilentlyContinue
}

# SIG # Begin signature block
# MIIocQYJKoZIhvcNAQcCoIIoYjCCKF4CAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCDHTpMmFSI/lkdn
# F+7QxBBCxdWOoc/RHC+hwO4bd7j5dKCCEfIwggVvMIIEV6ADAgECAhBI/JO0YFWU
# jTanyYqJ1pQWMA0GCSqGSIb3DQEBDAUAMHsxCzAJBgNVBAYTAkdCMRswGQYDVQQI
# DBJHcmVhdGVyIE1hbmNoZXN0ZXIxEDAOBgNVBAcMB1NhbGZvcmQxGjAYBgNVBAoM
# EUNvbW9kbyBDQSBMaW1pdGVkMSEwHwYDVQQDDBhBQUEgQ2VydGlmaWNhdGUgU2Vy
# dmljZXMwHhcNMjEwNTI1MDAwMDAwWhcNMjgxMjMxMjM1OTU5WjBWMQswCQYDVQQG
# EwJHQjEYMBYGA1UEChMPU2VjdGlnbyBMaW1pdGVkMS0wKwYDVQQDEyRTZWN0aWdv
# IFB1YmxpYyBDb2RlIFNpZ25pbmcgUm9vdCBSNDYwggIiMA0GCSqGSIb3DQEBAQUA
# A4ICDwAwggIKAoICAQCN55QSIgQkdC7/FiMCkoq2rjaFrEfUI5ErPtx94jGgUW+s
# hJHjUoq14pbe0IdjJImK/+8Skzt9u7aKvb0Ffyeba2XTpQxpsbxJOZrxbW6q5KCD
# J9qaDStQ6Utbs7hkNqR+Sj2pcaths3OzPAsM79szV+W+NDfjlxtd/R8SPYIDdub7
# P2bSlDFp+m2zNKzBenjcklDyZMeqLQSrw2rq4C+np9xu1+j/2iGrQL+57g2extme
# me/G3h+pDHazJyCh1rr9gOcB0u/rgimVcI3/uxXP/tEPNqIuTzKQdEZrRzUTdwUz
# T2MuuC3hv2WnBGsY2HH6zAjybYmZELGt2z4s5KoYsMYHAXVn3m3pY2MeNn9pib6q
# RT5uWl+PoVvLnTCGMOgDs0DGDQ84zWeoU4j6uDBl+m/H5x2xg3RpPqzEaDux5mcz
# mrYI4IAFSEDu9oJkRqj1c7AGlfJsZZ+/VVscnFcax3hGfHCqlBuCF6yH6bbJDoEc
# QNYWFyn8XJwYK+pF9e+91WdPKF4F7pBMeufG9ND8+s0+MkYTIDaKBOq3qgdGnA2T
# OglmmVhcKaO5DKYwODzQRjY1fJy67sPV+Qp2+n4FG0DKkjXp1XrRtX8ArqmQqsV/
# AZwQsRb8zG4Y3G9i/qZQp7h7uJ0VP/4gDHXIIloTlRmQAOka1cKG8eOO7F/05QID
# AQABo4IBEjCCAQ4wHwYDVR0jBBgwFoAUoBEKIz6W8Qfs4q8p74Klf9AwpLQwHQYD
# VR0OBBYEFDLrkpr/NZZILyhAQnAgNpFcF4XmMA4GA1UdDwEB/wQEAwIBhjAPBgNV
# HRMBAf8EBTADAQH/MBMGA1UdJQQMMAoGCCsGAQUFBwMDMBsGA1UdIAQUMBIwBgYE
# VR0gADAIBgZngQwBBAEwQwYDVR0fBDwwOjA4oDagNIYyaHR0cDovL2NybC5jb21v
# ZG9jYS5jb20vQUFBQ2VydGlmaWNhdGVTZXJ2aWNlcy5jcmwwNAYIKwYBBQUHAQEE
# KDAmMCQGCCsGAQUFBzABhhhodHRwOi8vb2NzcC5jb21vZG9jYS5jb20wDQYJKoZI
# hvcNAQEMBQADggEBABK/oe+LdJqYRLhpRrWrJAoMpIpnuDqBv0WKfVIHqI0fTiGF
# OaNrXi0ghr8QuK55O1PNtPvYRL4G2VxjZ9RAFodEhnIq1jIV9RKDwvnhXRFAZ/ZC
# J3LFI+ICOBpMIOLbAffNRk8monxmwFE2tokCVMf8WPtsAO7+mKYulaEMUykfb9gZ
# pk+e96wJ6l2CxouvgKe9gUhShDHaMuwV5KZMPWw5c9QLhTkg4IUaaOGnSDip0TYl
# d8GNGRbFiExmfS9jzpjoad+sPKhdnckcW67Y8y90z7h+9teDnRGWYpquRRPaf9xH
# +9/DUp/mBlXpnYzyOmJRvOwkDynUWICE5EV7WtgwggYcMIIEBKADAgECAhAz1wio
# kUBTGeKlu9M5ua1uMA0GCSqGSIb3DQEBDAUAMFYxCzAJBgNVBAYTAkdCMRgwFgYD
# VQQKEw9TZWN0aWdvIExpbWl0ZWQxLTArBgNVBAMTJFNlY3RpZ28gUHVibGljIENv
# ZGUgU2lnbmluZyBSb290IFI0NjAeFw0yMTAzMjIwMDAwMDBaFw0zNjAzMjEyMzU5
# NTlaMFcxCzAJBgNVBAYTAkdCMRgwFgYDVQQKEw9TZWN0aWdvIExpbWl0ZWQxLjAs
# BgNVBAMTJVNlY3RpZ28gUHVibGljIENvZGUgU2lnbmluZyBDQSBFViBSMzYwggGi
# MA0GCSqGSIb3DQEBAQUAA4IBjwAwggGKAoIBgQC70f4et0JbePWQp64sg/GNIdMw
# hoV739PN2RZLrIXFuwHP4owoEXIEdiyBxasSekBKxRDogRQ5G19PB/YwMDB/NSXl
# wHM9QAmU6Kj46zkLVdW2DIseJ/jePiLBv+9l7nPuZd0o3bsffZsyf7eZVReqskmo
# PBBqOsMhspmoQ9c7gqgZYbU+alpduLyeE9AKnvVbj2k4aOqlH1vKI+4L7bzQHkND
# brBTjMJzKkQxbr6PuMYC9ruCBBV5DFIg6JgncWHvL+T4AvszWbX0w1Xn3/YIIq62
# 0QlZ7AGfc4m3Q0/V8tm9VlkJ3bcX9sR0gLqHRqwG29sEDdVOuu6MCTQZlRvmcBME
# Jd+PuNeEM4xspgzraLqVT3xE6NRpjSV5wyHxNXf4T7YSVZXQVugYAtXueciGoWnx
# G06UE2oHYvDQa5mll1CeHDOhHu5hiwVoHI717iaQg9b+cYWnmvINFD42tRKtd3V6
# zOdGNmqQU8vGlHHeBzoh+dYyZ+CcblSGoGSgg8sCAwEAAaOCAWMwggFfMB8GA1Ud
# IwQYMBaAFDLrkpr/NZZILyhAQnAgNpFcF4XmMB0GA1UdDgQWBBSBMpJBKyjNRsjE
# osYqORLsSKk/FDAOBgNVHQ8BAf8EBAMCAYYwEgYDVR0TAQH/BAgwBgEB/wIBADAT
# BgNVHSUEDDAKBggrBgEFBQcDAzAaBgNVHSAEEzARMAYGBFUdIAAwBwYFZ4EMAQMw
# SwYDVR0fBEQwQjBAoD6gPIY6aHR0cDovL2NybC5zZWN0aWdvLmNvbS9TZWN0aWdv
# UHVibGljQ29kZVNpZ25pbmdSb290UjQ2LmNybDB7BggrBgEFBQcBAQRvMG0wRgYI
# KwYBBQUHMAKGOmh0dHA6Ly9jcnQuc2VjdGlnby5jb20vU2VjdGlnb1B1YmxpY0Nv
# ZGVTaWduaW5nUm9vdFI0Ni5wN2MwIwYIKwYBBQUHMAGGF2h0dHA6Ly9vY3NwLnNl
# Y3RpZ28uY29tMA0GCSqGSIb3DQEBDAUAA4ICAQBfNqz7+fZyWhS38Asd3tj9lwHS
# /QHumS2G6Pa38Dn/1oFKWqdCSgotFZ3mlP3FaUqy10vxFhJM9r6QZmWLLXTUqwj3
# ahEDCHd8vmnhsNufJIkD1t5cpOCy1rTP4zjVuW3MJ9bOZBHoEHJ20/ng6SyJ6UnT
# s5eWBgrh9grIQZqRXYHYNneYyoBBl6j4kT9jn6rNVFRLgOr1F2bTlHH9nv1HMePp
# GoYd074g0j+xUl+yk72MlQmYco+VAfSYQ6VK+xQmqp02v3Kw/Ny9hA3s7TSoXpUr
# OBZjBXXZ9jEuFWvilLIq0nQ1tZiao/74Ky+2F0snbFrmuXZe2obdq2TWauqDGIgb
# MYL1iLOUJcAhLwhpAuNMu0wqETDrgXkG4UGVKtQg9guT5Hx2DJ0dJmtfhAH2KpnN
# r97H8OQYok6bLyoMZqaSdSa+2UA1E2+upjcaeuitHFFjBypWBmztfhj24+xkc6Zt
# CDaLrw+ZrnVrFyvCTWrDUUZBVumPwo3/E3Gb2u2e05+r5UWmEsUUWlJBl6MGAAjF
# 5hzqJ4I8O9vmRsTvLQA1E802fZ3lqicIBczOwDYOSxlP0GOabb/FKVMxItt1UHeG
# 0PL4au5rBhs+hSMrl8h+eplBDN1Yfw6owxI9OjWb4J0sjBeBVESoeh2YnZZ/WVim
# VGX/UUIL+Efrz/jlvzCCBlswggTDoAMCAQICEDB3Np9sRenync55S1/V2zEwDQYJ
# KoZIhvcNAQELBQAwVzELMAkGA1UEBhMCR0IxGDAWBgNVBAoTD1NlY3RpZ28gTGlt
# aXRlZDEuMCwGA1UEAxMlU2VjdGlnbyBQdWJsaWMgQ29kZSBTaWduaW5nIENBIEVW
# IFIzNjAeFw0yNDAxMTYwMDAwMDBaFw0yNTAxMTUyMzU5NTlaMIG7MRAwDgYDVQQF
# EwcyMTg5MDc0MRMwEQYLKwYBBAGCNzwCAQMTAlVTMRkwFwYLKwYBBAGCNzwCAQIT
# CERlbGF3YXJlMR0wGwYDVQQPExRQcml2YXRlIE9yZ2FuaXphdGlvbjELMAkGA1UE
# BhMCVVMxEzARBgNVBAgMCkNhbGlmb3JuaWExGjAYBgNVBAoMEUludGVsIENvcnBv
# cmF0aW9uMRowGAYDVQQDDBFJbnRlbCBDb3Jwb3JhdGlvbjCCAaIwDQYJKoZIhvcN
# AQEBBQADggGPADCCAYoCggGBALUx6h43Sbt+ZfuTa/V0GFWgAAfUIWg9ruenPoc8
# FTqJxPdbqYRlGKEK2vCSQyBTn7tGV54Y/hMGvRTBQce4dMFe7R5wL9p92J3R4jkR
# MadZxJ4HuWalG13CpjQZP1Jg61Wx0KjsCPd2VdbBTpTiLRI+PKjokE6I/TLuA2bA
# tqy1phe2/82SRlx/fVGuIzI1BOQCaC9Olao3kJ9JwKt6VZidIgmzXAYLpzZ2VhW9
# HTRLaqTLTKxgO2mtpMTbfqc7RR/oYHG2cbGCoW3KB5Tlgjhm85ZsMEMV48O7JWeB
# wQoNy9admwqsdUzr5HdDbM7/EuiNHvP+sIauOmHEkLtqxM37sSgDBQX3V+vqiR5b
# H6xTcfIceaa4ukwKuVmfWrB35TGSEkWFzZu3ZUYGKRCI9WZ8GixV/5PHm571xSrD
# jbrigtkHtPahVVC8EUojnIGiDXovnm7ZG0FEXKuVybNOSpD3BzE2Pw3j/evKHuIr
# hWCzdPBOkbbEoYFBxts6UtJNFwIDAQABo4IBvDCCAbgwHwYDVR0jBBgwFoAUgTKS
# QSsozUbIxKLGKjkS7EipPxQwHQYDVR0OBBYEFKRLkSq7xH/Ru5kWUgmDt9ZZsJHj
# MA4GA1UdDwEB/wQEAwIHgDAMBgNVHRMBAf8EAjAAMBMGA1UdJQQMMAoGCCsGAQUF
# BwMDMEkGA1UdIARCMEAwNQYMKwYBBAGyMQECAQYBMCUwIwYIKwYBBQUHAgEWF2h0
# dHBzOi8vc2VjdGlnby5jb20vQ1BTMAcGBWeBDAEDMEsGA1UdHwREMEIwQKA+oDyG
# Omh0dHA6Ly9jcmwuc2VjdGlnby5jb20vU2VjdGlnb1B1YmxpY0NvZGVTaWduaW5n
# Q0FFVlIzNi5jcmwwewYIKwYBBQUHAQEEbzBtMEYGCCsGAQUFBzAChjpodHRwOi8v
# Y3J0LnNlY3RpZ28uY29tL1NlY3RpZ29QdWJsaWNDb2RlU2lnbmluZ0NBRVZSMzYu
# Y3J0MCMGCCsGAQUFBzABhhdodHRwOi8vb2NzcC5zZWN0aWdvLmNvbTAuBgNVHREE
# JzAloCMGCCsGAQUFBwgDoBcwFQwTVVMtREVMQVdBUkUtMjE4OTA3NDANBgkqhkiG
# 9w0BAQsFAAOCAYEAI6NvLiKQmXol9qHiDAxrF41RjhawR5g4ZQuWIxU2uMkjL9Eh
# NV16cKJjZ+KSPebFErInVlf/NLvKCfSUpf+2olf7Phu1pbb2p2R1mFtGYIA1Bcwa
# UmIsA/XsQ+ZHYlVdqBcpJCdc/jTzfAQen0gv1hFOIfk0vOKjjAAuTIgQNkG3c5Sw
# FMZjhu+wrSXx9Qvv85BhIX/xF1xYkKN4iBwqqebxAhmaE2tZUI7X/kKDx9QexONP
# Wo5Vw9b3P9I9pP8pWS/Txa4AC8x5AcZ8f6FUkl5Y750Ma6XEeHJZlUGkf1d/Ph9M
# c6bg5/P4wuXceTd4iqu3sK4ZxbiXWiFD+PtZ9UbzDarUNCqskSwudIzEcgwt7glZ
# YUNC+gD64uddohltoXXIasQNs/LK+qzOuqZKhFkRf+v4+Q28EQYuVIcJDHspW1CV
# k3Y4zhEb1dhCAPZ9jyXyz827Uji6HD5dnGO2lPCcEvM/aWEjYYM285g0CZix3LeB
# XQydSyjK95klPc99MYIV1TCCFdECAQEwazBXMQswCQYDVQQGEwJHQjEYMBYGA1UE
# ChMPU2VjdGlnbyBMaW1pdGVkMS4wLAYDVQQDEyVTZWN0aWdvIFB1YmxpYyBDb2Rl
# IFNpZ25pbmcgQ0EgRVYgUjM2AhAwdzafbEXp8p3OeUtf1dsxMA0GCWCGSAFlAwQC
# AQUAoGowGQYJKoZIhvcNAQkDMQwGCisGAQQBgjcCAQQwHAYKKwYBBAGCNwIBCzEO
# MAwGCisGAQQBgjcCARYwLwYJKoZIhvcNAQkEMSIEIIYx4OJmKyxsQNhE+SWixA3T
# LRfWCdCGk9pwrrWytT2fMA0GCSqGSIb3DQEBAQUABIIBgJu1q85C7NufIGOFouF5
# G4G7kR8jbBwvo4x64f51SkC076frjeI8yTAL/BICq6WANG9A8UHo/aqhzmNt6VV3
# r8An7J/LnFy5LmYumIn4OpSGMfGmWoJfzW0xvOzx1D64zGpDycfqkdF5H2hgnunN
# gEBpTA7QCOESn0NMeTkJT0/dprSLgPpCc5e/EU2vDTG2V0Jzl2DsAdxz32oXPPf1
# ks8Mk1ahgm7tla9safDZgmfoSxS0ZEFXr7wx5uVgyc2xXtGG26x7zAQj6eswoAma
# zHC1znlSKq9EKw7/T6/7pgQjtnn2pmROomnZI55S9X2nxTqgDeK75FGWkoATa2tS
# /dCg9q/BIxigw/JGdVhUZyFXdFDNsFpGuVodSbCNx2w1w8NtJLXwNDJYSnBNrTSz
# zjSTTfaFq2us/sMutbXZv00TaxTIV9DGV+HTiGXWwre5i4dWz6POIijZGSRUoE4v
# KdtupSAWzdCiLoM7nNOvBkMRJXoGW7uRh568pMSfbJcUsKGCE08wghNLBgorBgEE
# AYI3AwMBMYITOzCCEzcGCSqGSIb3DQEHAqCCEygwghMkAgEDMQ8wDQYJYIZIAWUD
# BAICBQAwgfAGCyqGSIb3DQEJEAEEoIHgBIHdMIHaAgEBBgorBgEEAbIxAgEBMDEw
# DQYJYIZIAWUDBAIBBQAEIFcGg3xTeZiYaLnGAa13++rrGl+ZQkA7XnZ0SsjDx3CA
# AhUAibrNXexA23SDH/8eUiWsDMlRKVIYDzIwMjQwMzE4MTgyMzQxWqBupGwwajEL
# MAkGA1UEBhMCR0IxEzARBgNVBAgTCk1hbmNoZXN0ZXIxGDAWBgNVBAoTD1NlY3Rp
# Z28gTGltaXRlZDEsMCoGA1UEAwwjU2VjdGlnbyBSU0EgVGltZSBTdGFtcGluZyBT
# aWduZXIgIzSggg3pMIIG9TCCBN2gAwIBAgIQOUwl4XygbSeoZeI72R0i1DANBgkq
# hkiG9w0BAQwFADB9MQswCQYDVQQGEwJHQjEbMBkGA1UECBMSR3JlYXRlciBNYW5j
# aGVzdGVyMRAwDgYDVQQHEwdTYWxmb3JkMRgwFgYDVQQKEw9TZWN0aWdvIExpbWl0
# ZWQxJTAjBgNVBAMTHFNlY3RpZ28gUlNBIFRpbWUgU3RhbXBpbmcgQ0EwHhcNMjMw
# NTAzMDAwMDAwWhcNMzQwODAyMjM1OTU5WjBqMQswCQYDVQQGEwJHQjETMBEGA1UE
# CBMKTWFuY2hlc3RlcjEYMBYGA1UEChMPU2VjdGlnbyBMaW1pdGVkMSwwKgYDVQQD
# DCNTZWN0aWdvIFJTQSBUaW1lIFN0YW1waW5nIFNpZ25lciAjNDCCAiIwDQYJKoZI
# hvcNAQEBBQADggIPADCCAgoCggIBAKSTKFJLzyeHdqQpHJk4wOcO1NEc7GjLAWTk
# is13sHFlgryf/Iu7u5WY+yURjlqICWYRFFiyuiJb5vYy8V0twHqiDuDgVmTtoeWB
# IHIgZEFsx8MI+vN9Xe8hmsJ+1yzDuhGYHvzTIAhCs1+/f4hYMqsws9iMepZKGRNc
# rPznq+kcFi6wsDiVSs+FUKtnAyWhuzjpD2+pWpqRKBM1uR/zPeEkyGuxmegN77tN
# 5T2MVAOR0Pwtz1UzOHoJHAfRIuBjhqe+/dKDcxIUm5pMCUa9NLzhS1B7cuBb/Rm7
# HzxqGXtuuy1EKr48TMysigSTxleGoHM2K4GX+hubfoiH2FJ5if5udzfXu1Cf+hgl
# TxPyXnypsSBaKaujQod34PRMAkjdWKVTpqOg7RmWZRUpxe0zMCXmloOBmvZgZpBY
# B4DNQnWs+7SR0MXdAUBqtqgQ7vaNereeda/TpUsYoQyfV7BeJUeRdM11EtGcb+Re
# DZvsdSbu/tP1ki9ShejaRFEqoswAyodmQ6MbAO+itZadYq0nC/IbSsnDlEI3iCCE
# qIeuw7ojcnv4VO/4ayewhfWnQ4XYKzl021p3AtGk+vXNnD3MH65R0Hts2B0tEUJT
# cXTC5TWqLVIS2SXP8NPQkUMS1zJ9mGzjd0HI/x8kVO9urcY+VXvxXIc6ZPFgSwVP
# 77kv7AkTAgMBAAGjggGCMIIBfjAfBgNVHSMEGDAWgBQaofhhGSAPw0F3RSiO0TVf
# BhIEVTAdBgNVHQ4EFgQUAw8xyJEqk71j89FdTaQ0D9KVARgwDgYDVR0PAQH/BAQD
# AgbAMAwGA1UdEwEB/wQCMAAwFgYDVR0lAQH/BAwwCgYIKwYBBQUHAwgwSgYDVR0g
# BEMwQTA1BgwrBgEEAbIxAQIBAwgwJTAjBggrBgEFBQcCARYXaHR0cHM6Ly9zZWN0
# aWdvLmNvbS9DUFMwCAYGZ4EMAQQCMEQGA1UdHwQ9MDswOaA3oDWGM2h0dHA6Ly9j
# cmwuc2VjdGlnby5jb20vU2VjdGlnb1JTQVRpbWVTdGFtcGluZ0NBLmNybDB0Bggr
# BgEFBQcBAQRoMGYwPwYIKwYBBQUHMAKGM2h0dHA6Ly9jcnQuc2VjdGlnby5jb20v
# U2VjdGlnb1JTQVRpbWVTdGFtcGluZ0NBLmNydDAjBggrBgEFBQcwAYYXaHR0cDov
# L29jc3Auc2VjdGlnby5jb20wDQYJKoZIhvcNAQEMBQADggIBAEybZVj64HnP7xXD
# Mm3eM5Hrd1ji673LSjx13n6UbcMixwSV32VpYRMM9gye9YkgXsGHxwMkysel8Cbf
# +PgxZQ3g621RV6aMhFIIRhwqwt7y2opF87739i7Efu347Wi/elZI6WHlmjl3vL66
# kWSIdf9dhRY0J9Ipy//tLdr/vpMM7G2iDczD8W69IZEaIwBSrZfUYngqhHmo1z2s
# IY9wwyR5OpfxDaOjW1PYqwC6WPs1gE9fKHFsGV7Cg3KQruDG2PKZ++q0kmV8B3w1
# RB2tWBhrYvvebMQKqWzTIUZw3C+NdUwjwkHQepY7w0vdzZImdHZcN6CaJJ5OX07T
# jw/lE09ZRGVLQ2TPSPhnZ7lNv8wNsTow0KE9SK16ZeTs3+AB8LMqSjmswaT5qX01
# 0DJAoLEZKhghssh9BXEaSyc2quCYHIN158d+S4RDzUP7kJd2KhKsQMFwW5kKQPqA
# bZRhe8huuchnZyRcUI0BIN4H9wHU+C4RzZ2D5fjKJRxEPSflsIZHKgsbhHZ9e2hP
# jbf3E7TtoC3ucw/ZELqdmSx813UfjxDElOZ+JOWVSoiMJ9aFZh35rmR2kehI/shV
# Cu0pwx/eOKbAFPsyPfipg2I2yMO+AIccq/pKQhyJA9z1XHxw2V14Tu6fXiDmCWp8
# KwijSPUV/ARP380hHHrl9Y4a1LlAMIIG7DCCBNSgAwIBAgIQMA9vrN1mmHR8qUY2
# p3gtuTANBgkqhkiG9w0BAQwFADCBiDELMAkGA1UEBhMCVVMxEzARBgNVBAgTCk5l
# dyBKZXJzZXkxFDASBgNVBAcTC0plcnNleSBDaXR5MR4wHAYDVQQKExVUaGUgVVNF
# UlRSVVNUIE5ldHdvcmsxLjAsBgNVBAMTJVVTRVJUcnVzdCBSU0EgQ2VydGlmaWNh
# dGlvbiBBdXRob3JpdHkwHhcNMTkwNTAyMDAwMDAwWhcNMzgwMTE4MjM1OTU5WjB9
# MQswCQYDVQQGEwJHQjEbMBkGA1UECBMSR3JlYXRlciBNYW5jaGVzdGVyMRAwDgYD
# VQQHEwdTYWxmb3JkMRgwFgYDVQQKEw9TZWN0aWdvIExpbWl0ZWQxJTAjBgNVBAMT
# HFNlY3RpZ28gUlNBIFRpbWUgU3RhbXBpbmcgQ0EwggIiMA0GCSqGSIb3DQEBAQUA
# A4ICDwAwggIKAoICAQDIGwGv2Sx+iJl9AZg/IJC9nIAhVJO5z6A+U++zWsB21hoE
# pc5Hg7XrxMxJNMvzRWW5+adkFiYJ+9UyUnkuyWPCE5u2hj8BBZJmbyGr1XEQeYf0
# RirNxFrJ29ddSU1yVg/cyeNTmDoqHvzOWEnTv/M5u7mkI0Ks0BXDf56iXNc48Ray
# cNOjxN+zxXKsLgp3/A2UUrf8H5VzJD0BKLwPDU+zkQGObp0ndVXRFzs0IXuXAZSv
# f4DP0REKV4TJf1bgvUacgr6Unb+0ILBgfrhN9Q0/29DqhYyKVnHRLZRMyIw80xSi
# nL0m/9NTIMdgaZtYClT0Bef9Maz5yIUXx7gpGaQpL0bj3duRX58/Nj4OMGcrRrc1
# r5a+2kxgzKi7nw0U1BjEMJh0giHPYla1IXMSHv2qyghYh3ekFesZVf/QOVQtJu5F
# GjpvzdeE8NfwKMVPZIMC1Pvi3vG8Aij0bdonigbSlofe6GsO8Ft96XZpkyAcSpcs
# dxkrk5WYnJee647BeFbGRCXfBhKaBi2fA179g6JTZ8qx+o2hZMmIklnLqEbAyfKm
# /31X2xJ2+opBJNQb/HKlFKLUrUMcpEmLQTkUAx4p+hulIq6lw02C0I3aa7fb9xhA
# V3PwcaP7Sn1FNsH3jYL6uckNU4B9+rY5WDLvbxhQiddPnTO9GrWdod6VQXqngwID
# AQABo4IBWjCCAVYwHwYDVR0jBBgwFoAUU3m/WqorSs9UgOHYm8Cd8rIDZsswHQYD
# VR0OBBYEFBqh+GEZIA/DQXdFKI7RNV8GEgRVMA4GA1UdDwEB/wQEAwIBhjASBgNV
# HRMBAf8ECDAGAQH/AgEAMBMGA1UdJQQMMAoGCCsGAQUFBwMIMBEGA1UdIAQKMAgw
# BgYEVR0gADBQBgNVHR8ESTBHMEWgQ6BBhj9odHRwOi8vY3JsLnVzZXJ0cnVzdC5j
# b20vVVNFUlRydXN0UlNBQ2VydGlmaWNhdGlvbkF1dGhvcml0eS5jcmwwdgYIKwYB
# BQUHAQEEajBoMD8GCCsGAQUFBzAChjNodHRwOi8vY3J0LnVzZXJ0cnVzdC5jb20v
# VVNFUlRydXN0UlNBQWRkVHJ1c3RDQS5jcnQwJQYIKwYBBQUHMAGGGWh0dHA6Ly9v
# Y3NwLnVzZXJ0cnVzdC5jb20wDQYJKoZIhvcNAQEMBQADggIBAG1UgaUzXRbhtVOB
# kXXfA3oyCy0lhBGysNsqfSoF9bw7J/RaoLlJWZApbGHLtVDb4n35nwDvQMOt0+Lk
# VvlYQc/xQuUQff+wdB+PxlwJ+TNe6qAcJlhc87QRD9XVw+K81Vh4v0h24URnbY+w
# QxAPjeT5OGK/EwHFhaNMxcyyUzCVpNb0llYIuM1cfwGWvnJSajtCN3wWeDmTk5Sb
# sdyybUFtZ83Jb5A9f0VywRsj1sJVhGbks8VmBvbz1kteraMrQoohkv6ob1olcGKB
# c2NeoLvY3NdK0z2vgwY4Eh0khy3k/ALWPncEvAQ2ted3y5wujSMYuaPCRx3wXdah
# c1cFaJqnyTdlHb7qvNhCg0MFpYumCf/RoZSmTqo9CfUFbLfSZFrYKiLCS53xOV5M
# 3kg9mzSWmglfjv33sVKRzj+J9hyhtal1H3G/W0NdZT1QgW6r8NDT/LKzH7aZlib0
# PHmLXGTMze4nmuWgwAxyh8FuTVrTHurwROYybxzrF06Uw3hlIDsPQaof6aFBnf6x
# uKBlKjTg3qj5PObBMLvAoGMs/FwWAKjQxH/qEZ0eBsambTJdtDgJK0kHqv3sMNrx
# py/Pt/360KOE2See+wFmd7lWEOEgbsausfm2usg1XTN2jvF8IAwqd661ogKGuinu
# tFoAsYyr4/kKyVRd1LlqdJ69SK6YMYIELDCCBCgCAQEwgZEwfTELMAkGA1UEBhMC
# R0IxGzAZBgNVBAgTEkdyZWF0ZXIgTWFuY2hlc3RlcjEQMA4GA1UEBxMHU2FsZm9y
# ZDEYMBYGA1UEChMPU2VjdGlnbyBMaW1pdGVkMSUwIwYDVQQDExxTZWN0aWdvIFJT
# QSBUaW1lIFN0YW1waW5nIENBAhA5TCXhfKBtJ6hl4jvZHSLUMA0GCWCGSAFlAwQC
# AgUAoIIBazAaBgkqhkiG9w0BCQMxDQYLKoZIhvcNAQkQAQQwHAYJKoZIhvcNAQkF
# MQ8XDTI0MDMxODE4MjM0MVowPwYJKoZIhvcNAQkEMTIEMGmyELnkA2OsrA+Ff7AR
# MdpJjbj1xrhuyGy0CMTJGJLOlIuhv9f7jTY9LTP3LFR9LjCB7QYLKoZIhvcNAQkQ
# Agwxgd0wgdowgdcwFgQUrmKvdQoMvUfWRh91aOK8jOfKT5QwgbwEFALWW5Xig3DB
# VwCV+oj5I92Tf62PMIGjMIGOpIGLMIGIMQswCQYDVQQGEwJVUzETMBEGA1UECBMK
# TmV3IEplcnNleTEUMBIGA1UEBxMLSmVyc2V5IENpdHkxHjAcBgNVBAoTFVRoZSBV
# U0VSVFJVU1QgTmV0d29yazEuMCwGA1UEAxMlVVNFUlRydXN0IFJTQSBDZXJ0aWZp
# Y2F0aW9uIEF1dGhvcml0eQIQMA9vrN1mmHR8qUY2p3gtuTANBgkqhkiG9w0BAQEF
# AASCAgBJct3ANObYMad0DJ6hX2xnBpuAycyyzhkqvwDhJXQGKqQByMcLqX5pg5Ul
# WhLlyoKNs/wuyIUNMwTuAFyRGh5YYYvtljpHgNgh7gXhHoGGpbvo5Ji8HiWrb3Tr
# 8EQPS4IWmLjuetb0suwFjqUoR7TDFBVR+Rxpnm5Q7lktjszuWG0ghG6WghkWu7it
# uJfzW23OoUOL+TWAVNMYXi4g91K3aA7jtQUcVVoDbwM8rYKFhCQ/MIYJMQJdIaj4
# KPQeB2jx3iUPhnF8QlAdbh79282ptT335lP6V1c/LjM3Y4jnrVEZqmgy1YGZMGmP
# JIPvN8KUzavaTHuykmz6xv2+Jgx9c9YfefV1+70k26fpQXIPcQSd3A0MzJSyiL1X
# ptw9eFie0c4cwi1BqYfDX40NwxN1XF7B/X1IXE4yTr/UZxmjE71EbO/lFyFCGMql
# SO6Kua3Yjx2p0F2XPKeP5l2VvzVuApnpc+3VEbzup8b/FdTI7/trQEm3mIJ4vWlL
# ZvXDeOK0xWxFgGv6uT3ImJOTdXYnndx+kibIqxhmuINW7rM5MpmQS4k2OTlqzV6g
# W1a9ZBldqDfKWC7TlGAiTbk3ELNVR0MKb5UpiizB6rtB4h4yIgOc7edAJB3iyBAs
# UqZKpqlYLZCc+9LFPnEpBPB5x+hEAYdNUH+1qcNHVq7eT0nmKg==
# SIG # End signature block
