﻿using System;
using System.Threading;
using System.Diagnostics;
using Intel.Cst.Client;
using System.Threading.Tasks;

namespace HupConsoleApp
{
    class HupListener : IFeatureCallback
    {
        private bool error;
        private readonly ManualResetEvent responseEvent;
        private readonly Stopwatch stopWatch;

        public HupListener()
        {
            this.responseEvent = new ManualResetEvent(false);
            Reset();
            this.stopWatch = new Stopwatch();
            this.stopWatch.Start();
        }

        public void Reset()
        {
            this.error = false;
            this.responseEvent.Reset();
        }

        public void OnSuccess(FeatureType featureType, ResponseType responseType)
        {
            Console.WriteLine("{0} {1}", featureType, responseType);
            this.responseEvent.Set();
        }

        public void OnEvent(FeatureType featureType, EventType eventType, object eventPayload)
        {
            Console.Write("{0}: {1} OnEvent: {2}", this.stopWatch.ElapsedMilliseconds, featureType, eventType.ToString());
            if (eventType == EventType.PROXIMITY)
            {
                float distance = (float)eventPayload;
                Console.Write(" distance = {0}mm", (int)distance);
            }
            Console.WriteLine();
        }

        public void OnError(FeatureType featureType, Error state)
        {
            this.error = true;
            Console.WriteLine("{0} OnError: {1}", featureType, state.Description);
            this.responseEvent.Set();
        }

        public int Wait(int timeoutMs)
        {
            var stopwatch = new Stopwatch();
            stopwatch.Start();
            if (this.responseEvent.WaitOne(timeoutMs))
            {
                return (int)stopwatch.ElapsedMilliseconds;
            }
            else
            {
                return -1;
            }
        }

        public void WaitLog(int timeoutMs)
        {
            int responseTime = Wait(timeoutMs);
            if (responseTime < 0)
            {
                Console.WriteLine("Error - no response");
                return;
            }
            if (!this.error)
            {
                Console.WriteLine("Succeeded in {0}ms", responseTime);
            }
            else
            {
                Console.WriteLine("Failed");
            }
        }
    }

    class Program
    {
        static void Main()
        {
            bool finished = false;
            var hupListener = new HupListener();
            var client = new ClientBackendLink(hupListener);

            Console.WriteLine("Select command");
            Console.WriteLine(" 1 - EnableWalkAwayLock");
            Console.WriteLine(" 2 - DisableWalkAwayLock");
            Console.WriteLine(" 3 - EnableWakeOnApproach");
            Console.WriteLine(" 4 - DisableWakeOnApproach");
            Console.WriteLine(" 5 - GetOptions(WAL)");
            Console.WriteLine(" 6 - GetOptions(WOA)");
            Console.WriteLine(" 0 - Exit");

            do
            {
                ConsoleKeyInfo option = Console.ReadKey(true);
                try
                {
                    switch (option.KeyChar)
                    {
                        case '0':
                            finished = true;
                            break;
                        case '1':
                            Console.WriteLine("Calling EnableWalkAwayLock");
                            hupListener.Reset();
                            client.Enable(FeatureType.LOCK);
                            hupListener.WaitLog(2000);
                            break;
                        case '2':
                            Console.WriteLine("Calling DisableWalkAwayLock");
                            hupListener.Reset();
                            client.Disable(FeatureType.LOCK);
                            hupListener.WaitLog(2000);
                            break;
                        case '3':
                            Console.WriteLine("Calling EnableWakeOnApproach");
                            hupListener.Reset();
                            client.Enable(FeatureType.WAKE);
                            hupListener.WaitLog(2000);
                            break;
                        case '4':
                            Console.WriteLine("Calling DisableWakeOnApproach");
                            hupListener.Reset();
                            client.Disable(FeatureType.WAKE);
                            hupListener.WaitLog(2000);
                            break;
                        case '5':
                            Console.WriteLine("Calling GetOptions for WAL");
                            hupListener.Reset();
                            Task<FeatureSetting> lockTask = client.GetOptions(FeatureType.LOCK);
                            lockTask.Wait();
                            Console.WriteLine(lockTask.Result);
                            break;
                        case '6':
                            Console.WriteLine("Calling GetOptions for WOA");
                            hupListener.Reset();
                            Task<FeatureSetting> wakeTask = client.GetOptions(FeatureType.WAKE);
                            wakeTask.Wait();
                            Console.WriteLine(wakeTask.Result);
                            break;
                    }
                }
                catch (Exception ex)
                {
                    Console.WriteLine("Error: {0}", ex.InnerException.Message);
                }
            } while (!finished);
        }
    }
}
