/* hwInterfaceAbstraction.c */

//#define ABSTRACTION_DEBUG

/***********************************************************************
*                    Copyright Aquantia Corporation
*                                Freeware
*
* $File: //depot/icm/proj/Atlantic/rev1.0/software/Software/Test/MBU-dependencies/hwInterface/include/hwInterfaceAbstraction.c $
*
* $Revision: #4 $
*
* $DateTime: 2016/09/21 08:49:55 $
*
* $Author: epomozov $
*
* $Label:  $
*
***********************************************************************/

/*! \file
    This file contains the abstractions for the following functions:

    AQ_API_MDIO_Write                                  AQ_API_MDIO_Write_Abstracted
    AQ_API_MDIO_Read                                   AQ_API_MDIO_Read_Abstracted
    AQ_API_MDIO_BlockRead                              AQ_API_MDIO_BlockRead_Abstracted
    AQ_API_MDIO_BlockWrite               ------>       AQ_API_MDIO_BlockWrite_Abstracted
    HW_Write                                           HW_Write_Abstracted
    HW_Read                                            HW_Read_Abstracted
    HW_BlockRead                                       HW_BlockRead_Abstracted
    HW_BlockWrite                                      HW_BlockWrite_Abstracted
    HW_BlockReadAdapter                                HW_BlockReadAdapter_Abstracted  
    HW_BlockWriteAdapter                               HW_BlockWriteAdapter_Abstracted
    
  The point of this is to allow virtual MMDs that indirectly address various regions of the the chip.  
  The indirection is implemented in these abstractions.
  
  This function is directly included into the body of any hwInterface.c or hwInterface.cc file.
*/

/*******************************************  MMD 0A  *************************************************
  Description: This MMD is located in DRAM at 0x3FFE0480 and is indirectly addressed via the
  uP Mailbox I/F.  Since this is a 32-bit word, and the MMD is 16 bits, reads can be done directly
  but writes must be done as read-modify-write.
  
  Read Address:
  Write ((0x3FFE0480 + Address) >> 16) & 0xFFFF -> 1E.202          // Address MSW
  Write (0x3FFE0480 + Address) & 0xFFFF -> 1E.203                  // Address LSW
  Write 0x8000 -> 1E.200                                           // Execute Read
  Poll on 1E.200.8 being low                                       // Operation Done
  if ((Address & 0x1)) Read 1E.204                                 // Odd, so MSW
  else Read 1E.205                                                 // Even, so LSW
  
  Write Data to Address:
  Write ((0x3FFE0480 + Address) >> 16) & 0xFFFF -> 1E.202          // Address MSW
  Write (0x3FFE0480 + Address) & 0xFFFF -> 1E.203                  // Address LSW
  Write 0x8000 -> 1E.200                                           // Execute Read (of read-modify-write)
  Poll on 1E.200.8 being low                                       // Operation Done
  Write ((0x3FFE0480 + Address) >> 16) & 0xFFFF -> 1E.202          // Re-write Address MSW as self-incrementing
  Write (0x3FFE0480 + Address) & 0xFFFF -> 1E.203                  // Re-write Address LSW as self-incrementing
  if ((Address & 0x1)) Write Data -> 1E.204                        // Odd, so MSW
  else Write Data -> 1E.205                                        // Even, so LSW
  Write 0xC000 -> 1E.200                                           // Execute Write
  Poll on 1E.200.8 being low                                       // Operation Done
  
******************************************************************************************************/

/*******************************************  MMD 1C  *************************************************

  *****
  APPIA
  *****
AHB control registers:

There are 6 devices (CMU, Lane0, Lane1, Lane2, Lane3, Common), each has a set of 256 8-bit registers - 
address 0x00-0xff.  These registers are accessed by selecting the device in 4.c180[2:0] (0x1 selects 
CMU, 0x2 selects Lane0). The AHB registers can then be accessed at 4.c200-4.c2ff. Most registers 
values are bit reversed when compared to SB register spec, but I know of at least one exception 
(below). I would just bit reverse them all to be consistent.

TBUS status registers:

All devices except Common has a set of 26 12-bit status registers - address 0x00-0x19. They are 
accessed through the AHB CMU register space. The device and the TBUS register address is first 
written to AHB CMU register 0xb. Device selection is 0xb[7:5], address is 0xb[4:0]. The value written 
to 0xb has to be bit-reversed. The TBUS register value can then be read through AHB CMU register 0xef 
and 0xff. 0xef contains bits [7:0], 0xff bits [11:8]. The values are NOT bit reversed just to screw 
with your mind.

Here's the python function we use:

def readTbus(self, device, address):
        """
        Read snowbush testbus data
        On entry:
            device = 0 -> CMU
                     1 -> Lane0
                     2 -> Lane1
                     3 -> Lane0
                     4 -> Lane3
        On exit:
            data = 12-bit tbus data
        """
        #Check device
        if device < 0 or device > 4:
            raise ExceptionBadSerdesParameter
        
        tbusAddress = self.bitReverse((device << 5) + address)

        #Write tbus address
        self.writeAhbCMU(0xb, tbusAddress)

        #Read back tbus value, 12 bits split across two registers
        return (self.readAhbCMU(0xff) << 8) + self.readAhbCMU(0xef)


  *****
  Aspen
  *****
VSEMI registers:

There are 6 devices (Lane0, Lane1, Lane2, Lane3, Common).  The address space for each device is 
technically 13 bits.  VESMI's register map documentation shows bitfields in address ranges 0-0x169
for the lane-specific devices, and in 0-0x323 for the common device.  There is a test bus that
can be accessed indirectly via registers in the common device, but these are NOT mapped through
our 0x1C virtual MMD.

1C.4000 thru 1C.41FF Lane0 registers (8 bits), VSEMI-lane0 addresses 0 thru 0x1FF
1C.4300 thru 1C.44FF Lane1 registers (8 bits), VSEMI-lane1 addresses 0 thru 0x1FF
1C.4600 thru 1C.47FF Lane2 registers (8 bits), VSEMI-lane2 addresses 0 thru 0x1FF
1C.4900 thru 1C.4AFF Lane3 registers (8 bits), VSEMI-lane3 addresses 0 thru 0x1FF
1C.4C00 thru 1C.4FFF Common registers (8 bits), VSEMI-common addresses 0 thru 0x3FF

******************************************************************************************************/
#ifdef WIN32
#include <windows.h>
#endif
#define MMD_0x0A_BASE_ADDRESS 0x3FFE0480
#define MAX_POLL_COUNT 40

#define MAX_NUM_COMMAND_STATUS_POLLS 50
#define MAX_NUM_DELEGATION_RETRIES 10

/*#define DELEGATE_SCRATCHPADS*/
/*#define DELEGATE_SIMONLIST*/
#define DELEGATE_NEC_READS
#define DELEGATE_DEC_READS
#define DO_VSEMI_SERDES_REDIRECTION
#define DELEGATE_VSEMI_SERDES_REDIRECTION


/*! Returns 1 if the specified register is a NEC register, else 0. */
uint32_t isNECRegister(uint32_t MMD, uint32_t address)
{
  if ((MMD == 0x1F) && 
       /* Channel 0 */
      (address == 0x8037 || address == 0x8038 || address == 0x8039 || address == 0x8045 || 
       address == 0x8046 || address == 0x8047 || address == 0x8048 || address == 0x8049 || 
       address == 0x804A || address == 0x806F || address == 0x8093 || address == 0x8094 || 
       address == 0x8095 || address == 0x80F3 || address == 0x80F4 || address == 0x80F5 || 
       address == 0x81C0 || address == 0x81C1 || address == 0x8280 || address == 0x8281 || 
       address == 0x8282 || address == 0x8283 || address == 0x8284 || address == 0x8288 || 
       address == 0x8289 || address == 0x828A || address == 0x828B || address == 0x828C || 
       address == 0x8290 || address == 0x8291 || address == 0x8292 || address == 0x8293 || 
       address == 0x8294 || address == 0x8298 || address == 0x8299 || address == 0x829A || 
       address == 0x829B || address == 0x829C || address == 0x82A0 || address == 0x82A1 || 
       address == 0x82A2 || address == 0x82A3 || address == 0x82A4 || address == 0x82A8 || 
       address == 0x82A9 || address == 0x82AA || address == 0x82AB || address == 0x82AC || 
       address == 0x82B0 || address == 0x82B1 || address == 0x82B2 || address == 0x82B3 || 
       address == 0x82B4 || address == 0x82B8 || address == 0x82B9 || address == 0x82BA || 
       address == 0x82BB || address == 0x82BC || address == 0x82C0 || address == 0x82C1 || 
       address == 0x82C2 || address == 0x82C3 || address == 0x82C4 || address == 0x82C8 || 
       address == 0x82C9 || address == 0x82CA || address == 0x82CB || address == 0x82CC || 
       address == 0x82D0 || address == 0x82D1 || address == 0x82D2 || address == 0x82D3 || 
       address == 0x82D4 || address == 0x82D8 || address == 0x82D9 || address == 0x82DA || 
       address == 0x82DB || address == 0x82DC || address == 0x8803 || address == 0x8807 || 
       address == 0x8812 || address == 0x881C || address == 0x882E || address == 0x882F || 
       address == 0x8830 || address == 0x8831 || address == 0x8832 || address == 0x8833 || 
       address == 0x8834 || address == 0x8835 || address == 0x8836 || address == 0x8837 || 
       address == 0x8838 || address == 0x8839 || address == 0x8980 || address == 0x8988 || 
       address == 0x8990 || address == 0x8998 || address == 0x89A0 || address == 0x89A8 || 
       address == 0x89B0 || address == 0x89B8 || address == 0x89C0 || address == 0x89C8 || 
       address == 0x89D0 || address == 0x89D8 || 

       /* Channel 1 */
       address == 0x9037 || address == 0x9038 || address == 0x9039 || address == 0x9045 || 
       address == 0x9046 || address == 0x9047 || address == 0x9048 || address == 0x9049 || 
       address == 0x904A || address == 0x906F || address == 0x9093 || address == 0x9094 || 
       address == 0x9095 || address == 0x90F3 || address == 0x90F4 || address == 0x90F5 || 
       address == 0x91C0 || address == 0x91C1 || address == 0x9280 || address == 0x9281 || 
       address == 0x9282 || address == 0x9283 || address == 0x9284 || address == 0x9288 || 
       address == 0x9289 || address == 0x928A || address == 0x928B || address == 0x928C || 
       address == 0x9290 || address == 0x9291 || address == 0x9292 || address == 0x9293 || 
       address == 0x9294 || address == 0x9298 || address == 0x9299 || address == 0x929A || 
       address == 0x929B || address == 0x929C || address == 0x92A0 || address == 0x92A1 || 
       address == 0x92A2 || address == 0x92A3 || address == 0x92A4 || address == 0x92A8 || 
       address == 0x92A9 || address == 0x92AA || address == 0x92AB || address == 0x92AC || 
       address == 0x92B0 || address == 0x92B1 || address == 0x92B2 || address == 0x92B3 || 
       address == 0x92B4 || address == 0x92B8 || address == 0x92B9 || address == 0x92BA || 
       address == 0x92BB || address == 0x92BC || address == 0x92C0 || address == 0x92C1 || 
       address == 0x92C2 || address == 0x92C3 || address == 0x92C4 || address == 0x92C8 || 
       address == 0x92C9 || address == 0x92CA || address == 0x92CB || address == 0x92CC || 
       address == 0x92D0 || address == 0x92D1 || address == 0x92D2 || address == 0x92D3 || 
       address == 0x92D4 || address == 0x92D8 || address == 0x92D9 || address == 0x92DA || 
       address == 0x92DB || address == 0x92DC || address == 0x9803 || address == 0x9807 || 
       address == 0x9812 || address == 0x981C || address == 0x982E || address == 0x982F || 
       address == 0x9830 || address == 0x9831 || address == 0x9832 || address == 0x9833 || 
       address == 0x9834 || address == 0x9835 || address == 0x9836 || address == 0x9837 || 
       address == 0x9838 || address == 0x9839 || address == 0x9980 || address == 0x9988 || 
       address == 0x9990 || address == 0x9998 || address == 0x99A0 || address == 0x99A8 || 
       address == 0x99B0 || address == 0x99B8 || address == 0x99C0 || address == 0x99C8 || 
       address == 0x99D0 || address == 0x99D8 || 

       /* Channel 2 */
       address == 0xA037 || address == 0xA038 || address == 0xA039 || address == 0xA045 || 
       address == 0xA046 || address == 0xA047 || address == 0xA048 || address == 0xA049 || 
       address == 0xA04A || address == 0xA06F || address == 0xA093 || address == 0xA094 || 
       address == 0xA095 || address == 0xA0F3 || address == 0xA0F4 || address == 0xA0F5 || 
       address == 0xA1C0 || address == 0xA1C1 || address == 0xA280 || address == 0xA281 || 
       address == 0xA282 || address == 0xA283 || address == 0xA284 || address == 0xA288 || 
       address == 0xA289 || address == 0xA28A || address == 0xA28B || address == 0xA28C || 
       address == 0xA290 || address == 0xA291 || address == 0xA292 || address == 0xA293 || 
       address == 0xA294 || address == 0xA298 || address == 0xA299 || address == 0xA29A || 
       address == 0xA29B || address == 0xA29C || address == 0xA2A0 || address == 0xA2A1 || 
       address == 0xA2A2 || address == 0xA2A3 || address == 0xA2A4 || address == 0xA2A8 || 
       address == 0xA2A9 || address == 0xA2AA || address == 0xA2AB || address == 0xA2AC || 
       address == 0xA2B0 || address == 0xA2B1 || address == 0xA2B2 || address == 0xA2B3 || 
       address == 0xA2B4 || address == 0xA2B8 || address == 0xA2B9 || address == 0xA2BA || 
       address == 0xA2BB || address == 0xA2BC || address == 0xA2C0 || address == 0xA2C1 || 
       address == 0xA2C2 || address == 0xA2C3 || address == 0xA2C4 || address == 0xA2C8 || 
       address == 0xA2C9 || address == 0xA2CA || address == 0xA2CB || address == 0xA2CC || 
       address == 0xA2D0 || address == 0xA2D1 || address == 0xA2D2 || address == 0xA2D3 || 
       address == 0xA2D4 || address == 0xA2D8 || address == 0xA2D9 || address == 0xA2DA || 
       address == 0xA2DB || address == 0xA2DC || address == 0xA803 || address == 0xA807 || 
       address == 0xA812 || address == 0xA81C || address == 0xA82E || address == 0xA82F || 
       address == 0xA830 || address == 0xA831 || address == 0xA832 || address == 0xA833 || 
       address == 0xA834 || address == 0xA835 || address == 0xA836 || address == 0xA837 || 
       address == 0xA838 || address == 0xA839 || address == 0xA980 || address == 0xA988 || 
       address == 0xA990 || address == 0xA998 || address == 0xA9A0 || address == 0xA9A8 || 
       address == 0xA9B0 || address == 0xA9B8 || address == 0xA9C0 || address == 0xA9C8 || 
       address == 0xA9D0 || address == 0xA9D8 || 

       /* Channel 3 */
       address == 0xB037 || address == 0xB038 || address == 0xB039 || address == 0xB045 || 
       address == 0xB046 || address == 0xB047 || address == 0xB048 || address == 0xB049 || 
       address == 0xB04A || address == 0xB06F || address == 0xB093 || address == 0xB094 || 
       address == 0xB095 || address == 0xB0F3 || address == 0xB0F4 || address == 0xB0F5 || 
       address == 0xB1C0 || address == 0xB1C1 || address == 0xB280 || address == 0xB281 || 
       address == 0xB282 || address == 0xB283 || address == 0xB284 || address == 0xB288 || 
       address == 0xB289 || address == 0xB28A || address == 0xB28B || address == 0xB28C || 
       address == 0xB290 || address == 0xB291 || address == 0xB292 || address == 0xB293 || 
       address == 0xB294 || address == 0xB298 || address == 0xB299 || address == 0xB29A || 
       address == 0xB29B || address == 0xB29C || address == 0xB2A0 || address == 0xB2A1 || 
       address == 0xB2A2 || address == 0xB2A3 || address == 0xB2A4 || address == 0xB2A8 || 
       address == 0xB2A9 || address == 0xB2AA || address == 0xB2AB || address == 0xB2AC || 
       address == 0xB2B0 || address == 0xB2B1 || address == 0xB2B2 || address == 0xB2B3 || 
       address == 0xB2B4 || address == 0xB2B8 || address == 0xB2B9 || address == 0xB2BA || 
       address == 0xB2BB || address == 0xB2BC || address == 0xB2C0 || address == 0xB2C1 || 
       address == 0xB2C2 || address == 0xB2C3 || address == 0xB2C4 || address == 0xB2C8 || 
       address == 0xB2C9 || address == 0xB2CA || address == 0xB2CB || address == 0xB2CC || 
       address == 0xB2D0 || address == 0xB2D1 || address == 0xB2D2 || address == 0xB2D3 || 
       address == 0xB2D4 || address == 0xB2D8 || address == 0xB2D9 || address == 0xB2DA || 
       address == 0xB2DB || address == 0xB2DC || address == 0xB803 || address == 0xB807 || 
       address == 0xB812 || address == 0xB81C || address == 0xB82E || address == 0xB82F || 
       address == 0xB830 || address == 0xB831 || address == 0xB832 || address == 0xB833 || 
       address == 0xB834 || address == 0xB835 || address == 0xB836 || address == 0xB837 || 
       address == 0xB838 || address == 0xB839 || address == 0xB980 || address == 0xB988 || 
       address == 0xB990 || address == 0xB998 || address == 0xB9A0 || address == 0xB9A8 || 
       address == 0xB9B0 || address == 0xB9B8 || address == 0xB9C0 || address == 0xB9C8 || 
       address == 0xB9D0 || address == 0xB9D8))
  {
    return 1;
  }
  else
  {
    return 0;
  }
}


/*! Returns 1 if the specified register is a DEC register, else 0. */
uint32_t isDECRegister(uint32_t MMD, uint32_t address)
{
  if ((MMD == 0x1F) && 
       /* Channel 0 */
      (address == 0x8035 || address == 0x8041 || address == 0x8042 || address == 0x806D || 
       address == 0x8090 || address == 0x80F0 || address == 0x80F1 || address == 0x80F2 || 
       address == 0x81A0 || address == 0x81A1 || address == 0x8200 || address == 0x8201 || 
       address == 0x8202 || address == 0x8203 || address == 0x8204 || address == 0x8208 || 
       address == 0x8209 || address == 0x820A || address == 0x820B || address == 0x820C || 
       address == 0x8210 || address == 0x8211 || address == 0x8212 || address == 0x8213 || 
       address == 0x8214 || address == 0x8218 || address == 0x8219 || address == 0x821A || 
       address == 0x821B || address == 0x821C || address == 0x8220 || address == 0x8221 || 
       address == 0x8222 || address == 0x8223 || address == 0x8224 || address == 0x8228 || 
       address == 0x8229 || address == 0x822A || address == 0x822B || address == 0x822C || 
       address == 0x8230 || address == 0x8231 || address == 0x8232 || address == 0x8233 || 
       address == 0x8234 || address == 0x8238 || address == 0x8239 || address == 0x823A || 
       address == 0x823B || address == 0x823C || address == 0x8240 || address == 0x8241 || 
       address == 0x8242 || address == 0x8243 || address == 0x8244 || address == 0x8248 || 
       address == 0x8249 || address == 0x824A || address == 0x824B || address == 0x824C || 
       address == 0x8250 || address == 0x8251 || address == 0x8252 || address == 0x8253 || 
       address == 0x8254 || address == 0x8258 || address == 0x8259 || address == 0x825A || 
       address == 0x825B || address == 0x825C || address == 0x8260 || address == 0x8261 || 
       address == 0x8262 || address == 0x8263 || address == 0x8264 || address == 0x8268 || 
       address == 0x8269 || address == 0x826A || address == 0x826B || address == 0x826C || 
       address == 0x8270 || address == 0x8271 || address == 0x8272 || address == 0x8273 || 
       address == 0x8274 || address == 0x8278 || address == 0x8279 || address == 0x827A || 
       address == 0x827B || address == 0x827C || address == 0x8800 || address == 0x8805 || 
       address == 0x8811 || address == 0x881A || address == 0x882A || address == 0x882B || 
       address == 0x882C || address == 0x882D || address == 0x8900 || address == 0x8908 || 
       address == 0x8910 || address == 0x8918 || address == 0x8920 || address == 0x8928 || 
       address == 0x8930 || address == 0x8938 || address == 0x8940 || address == 0x8948 || 
       address == 0x8950 || address == 0x8958 || address == 0x8960 || address == 0x8968 || 
       address == 0x8970 || address == 0x8978 ||       

       /* Channel 1 */
       address == 0x9035 || address == 0x9041 || address == 0x9042 || address == 0x906D || 
       address == 0x9090 || address == 0x90F0 || address == 0x90F1 || address == 0x90F2 || 
       address == 0x91A0 || address == 0x91A1 || address == 0x9200 || address == 0x9201 || 
       address == 0x9202 || address == 0x9203 || address == 0x9204 || address == 0x9208 || 
       address == 0x9209 || address == 0x920A || address == 0x920B || address == 0x920C || 
       address == 0x9210 || address == 0x9211 || address == 0x9212 || address == 0x9213 || 
       address == 0x9214 || address == 0x9218 || address == 0x9219 || address == 0x921A || 
       address == 0x921B || address == 0x921C || address == 0x9220 || address == 0x9221 || 
       address == 0x9222 || address == 0x9223 || address == 0x9224 || address == 0x9228 || 
       address == 0x9229 || address == 0x922A || address == 0x922B || address == 0x922C || 
       address == 0x9230 || address == 0x9231 || address == 0x9232 || address == 0x9233 || 
       address == 0x9234 || address == 0x9238 || address == 0x9239 || address == 0x923A || 
       address == 0x923B || address == 0x923C || address == 0x9240 || address == 0x9241 || 
       address == 0x9242 || address == 0x9243 || address == 0x9244 || address == 0x9248 || 
       address == 0x9249 || address == 0x924A || address == 0x924B || address == 0x924C || 
       address == 0x9250 || address == 0x9251 || address == 0x9252 || address == 0x9253 || 
       address == 0x9254 || address == 0x9258 || address == 0x9259 || address == 0x925A || 
       address == 0x925B || address == 0x925C || address == 0x9260 || address == 0x9261 || 
       address == 0x9262 || address == 0x9263 || address == 0x9264 || address == 0x9268 || 
       address == 0x9269 || address == 0x926A || address == 0x926B || address == 0x926C || 
       address == 0x9270 || address == 0x9271 || address == 0x9272 || address == 0x9273 || 
       address == 0x9274 || address == 0x9278 || address == 0x9279 || address == 0x927A || 
       address == 0x927B || address == 0x927C || address == 0x9800 || address == 0x9805 || 
       address == 0x9811 || address == 0x981A || address == 0x982A || address == 0x982B || 
       address == 0x982C || address == 0x982D || address == 0x9900 || address == 0x9908 || 
       address == 0x9910 || address == 0x9918 || address == 0x9920 || address == 0x9928 || 
       address == 0x9930 || address == 0x9938 || address == 0x9940 || address == 0x9948 || 
       address == 0x9950 || address == 0x9958 || address == 0x9960 || address == 0x9968 || 
       address == 0x9970 || address == 0x9978 ||       

       /* Channel 2 */
       address == 0xA035 || address == 0xA041 || address == 0xA042 || address == 0xA06D || 
       address == 0xA090 || address == 0xA0F0 || address == 0xA0F1 || address == 0xA0F2 || 
       address == 0xA1A0 || address == 0xA1A1 || address == 0xA200 || address == 0xA201 || 
       address == 0xA202 || address == 0xA203 || address == 0xA204 || address == 0xA208 || 
       address == 0xA209 || address == 0xA20A || address == 0xA20B || address == 0xA20C || 
       address == 0xA210 || address == 0xA211 || address == 0xA212 || address == 0xA213 || 
       address == 0xA214 || address == 0xA218 || address == 0xA219 || address == 0xA21A || 
       address == 0xA21B || address == 0xA21C || address == 0xA220 || address == 0xA221 || 
       address == 0xA222 || address == 0xA223 || address == 0xA224 || address == 0xA228 || 
       address == 0xA229 || address == 0xA22A || address == 0xA22B || address == 0xA22C || 
       address == 0xA230 || address == 0xA231 || address == 0xA232 || address == 0xA233 || 
       address == 0xA234 || address == 0xA238 || address == 0xA239 || address == 0xA23A || 
       address == 0xA23B || address == 0xA23C || address == 0xA240 || address == 0xA241 || 
       address == 0xA242 || address == 0xA243 || address == 0xA244 || address == 0xA248 || 
       address == 0xA249 || address == 0xA24A || address == 0xA24B || address == 0xA24C || 
       address == 0xA250 || address == 0xA251 || address == 0xA252 || address == 0xA253 || 
       address == 0xA254 || address == 0xA258 || address == 0xA259 || address == 0xA25A || 
       address == 0xA25B || address == 0xA25C || address == 0xA260 || address == 0xA261 || 
       address == 0xA262 || address == 0xA263 || address == 0xA264 || address == 0xA268 || 
       address == 0xA269 || address == 0xA26A || address == 0xA26B || address == 0xA26C || 
       address == 0xA270 || address == 0xA271 || address == 0xA272 || address == 0xA273 || 
       address == 0xA274 || address == 0xA278 || address == 0xA279 || address == 0xA27A || 
       address == 0xA27B || address == 0xA27C || address == 0xA800 || address == 0xA805 || 
       address == 0xA811 || address == 0xA81A || address == 0xA82A || address == 0xA82B || 
       address == 0xA82C || address == 0xA82D || address == 0xA900 || address == 0xA908 || 
       address == 0xA910 || address == 0xA918 || address == 0xA920 || address == 0xA928 || 
       address == 0xA930 || address == 0xA938 || address == 0xA940 || address == 0xA948 || 
       address == 0xA950 || address == 0xA958 || address == 0xA960 || address == 0xA968 || 
       address == 0xA970 || address == 0xA978 ||       

       /* Channel 3 */
       address == 0xB035 || address == 0xB041 || address == 0xB042 || address == 0xB06D || 
       address == 0xB090 || address == 0xB0F0 || address == 0xB0F1 || address == 0xB0F2 || 
       address == 0xB1A0 || address == 0xB1A1 || address == 0xB200 || address == 0xB201 || 
       address == 0xB202 || address == 0xB203 || address == 0xB204 || address == 0xB208 || 
       address == 0xB209 || address == 0xB20A || address == 0xB20B || address == 0xB20C || 
       address == 0xB210 || address == 0xB211 || address == 0xB212 || address == 0xB213 || 
       address == 0xB214 || address == 0xB218 || address == 0xB219 || address == 0xB21A || 
       address == 0xB21B || address == 0xB21C || address == 0xB220 || address == 0xB221 || 
       address == 0xB222 || address == 0xB223 || address == 0xB224 || address == 0xB228 || 
       address == 0xB229 || address == 0xB22A || address == 0xB22B || address == 0xB22C || 
       address == 0xB230 || address == 0xB231 || address == 0xB232 || address == 0xB233 || 
       address == 0xB234 || address == 0xB238 || address == 0xB239 || address == 0xB23A || 
       address == 0xB23B || address == 0xB23C || address == 0xB240 || address == 0xB241 || 
       address == 0xB242 || address == 0xB243 || address == 0xB244 || address == 0xB248 || 
       address == 0xB249 || address == 0xB24A || address == 0xB24B || address == 0xB24C || 
       address == 0xB250 || address == 0xB251 || address == 0xB252 || address == 0xB253 || 
       address == 0xB254 || address == 0xB258 || address == 0xB259 || address == 0xB25A || 
       address == 0xB25B || address == 0xB25C || address == 0xB260 || address == 0xB261 || 
       address == 0xB262 || address == 0xB263 || address == 0xB264 || address == 0xB268 || 
       address == 0xB269 || address == 0xB26A || address == 0xB26B || address == 0xB26C || 
       address == 0xB270 || address == 0xB271 || address == 0xB272 || address == 0xB273 || 
       address == 0xB274 || address == 0xB278 || address == 0xB279 || address == 0xB27A || 
       address == 0xB27B || address == 0xB27C || address == 0xB800 || address == 0xB805 || 
       address == 0xB811 || address == 0xB81A || address == 0xB82A || address == 0xB82B || 
       address == 0xB82C || address == 0xB82D || address == 0xB900 || address == 0xB908 || 
       address == 0xB910 || address == 0xB918 || address == 0xB920 || address == 0xB928 || 
       address == 0xB930 || address == 0xB938 || address == 0xB940 || address == 0xB948 || 
       address == 0xB950 || address == 0xB958 || address == 0xB960 || address == 0xB968 || 
       address == 0xB970 || address == 0xB978))
  {
    return 1;
  }
  else
  {
    return 0;
  }
}


/*! Returns 1 if the specified register is on Simon's list, else 0. */
uint32_t isSimonListRegister(uint32_t MMD, uint32_t address)
{
#ifdef WIN32
  address;
#endif
  if (MMD == 0x3 /*&& */)
       /* Channel 0 */
      /*((address >= 0xE63A && address <= 0xE63C) || 
       (address >= 0xE643 && address <= 0xE644) || 
       (address >= 0xE660 && address <= 0xE661) || 
       (address >= 0xE910 && address <= 0xE912) || 
       (address == 0xC650) || 
       (address == 0xC65D) || 
       (address == 0xC652) || 
       (address == 0xC651) || 
       (address == 0xCC07) || 
       (address == 0xFA00))) */
  {
    return 1;
  }
  else
  {
    return 0;
  }
}


/*! Returns 1 if the specified register should be written via 
 * FW delegation, else 0. */
uint32_t shouldDelegateWrite(uint32_t MMD, uint32_t address)
{
#ifdef WIN32
  address;
#endif
#ifdef DELEGATE_VSEMI_SERDES_REDIRECTION
  if (MMD == 0x1C/* && address >= 0x4000 && address <= 0x4FFF*/)
  {
    return 1;
  }
#endif
#ifdef DELEGATE_SIMONLIST
  else if (isSimonListRegister(MMD, address))
  {
    return 1;
  }
#endif

  return 0;
}


/*! Returns 1 if the specified register should be read via
 * FW delegation., else 0 */
uint32_t shouldDelegateRead(uint32_t MMD, uint32_t address)
{
#ifdef DELEGATE_NEC_READS
  if (isNECRegister(MMD, address))
  {
    return 1;
  }
#endif
#ifdef DELEGATE_DEC_READS
  if (isDECRegister(MMD, address))
  {
    return 1;
  }
#endif
#ifdef DELEGATE_VSEMI_SERDES_REDIRECTION
  if (MMD == 0x1C/* && address >= 0x4000 && address <= 0x4FFF*/)
  {
    return 1;
  }
#endif
#ifdef DELEGATE_SIMONLIST
  if (isSimonListRegister(MMD, address))
  {
    return 1;
  }
#endif
#ifdef DELEGATE_SCRATCHPADS
  if (MMD == 0x1E && (address >= 0x300 && address <= 0x323))
  {
    return 1;
  }
#endif

  return 0;
}

uint32_t shouldDelegateWrite32( uint32_t address )
{
#ifdef WIN32
  address;
#endif
    return 0;
}


/*! Returns 1 if the specified register should be read via
* FW delegation., else 0 */
uint32_t shouldDelegateRead32( uint32_t address )
{
#ifdef WIN32
  address;
#endif
    return 0;
}

uint32_t shouldDelegateWrite64( uint32_t address )
{
#ifdef WIN32
  address;
#endif
    return 0;
}


/*! Returns 1 if the specified register should be read via
* FW delegation., else 0 */
uint32_t shouldDelegateRead64( uint32_t address )
{
#ifdef WIN32
  address;
#endif
    return 0;
}

void AQ_API_MDIO_Write
(
    /*! The address of the PHY being addressed. */
  uint32_t PHY_ID,
    /*! The address of the MMD within the PHY being addressed.*/
  uint32_t MMD,
    /*! The 32-bit address being written to.*/
  uint32_t address,
    /*! The 32-bits of data being sent out with the write command. This may be either address or data.*/
  uint32_t data
)
{
  /* declare local variables */
  uint32_t pollCount = 0;
  uint32_t x, commandStatus, numCommandStatusPolls, numDelegationRetries, delegationSuccess;
  unsigned long targetAddress;
#ifdef EXTRA_READ_BEFORE_WRITE
  uint32_t prevValue;
#endif
  
  if (shouldDelegateWrite(MMD, address))
  {
    numDelegationRetries = 0;
    delegationSuccess = 0;
    while ((!delegationSuccess) && (numDelegationRetries < MAX_NUM_DELEGATION_RETRIES))
    {
      /* Clear the command status register */
      AQ_API_MDIO_Write(PHY_ID, 0x1E, 0xC47F, 0);
      /* Write the target address */
      AQ_API_MDIO_Write(PHY_ID, 0x1E, 0xC47C, address);
      /* Write the data */
      AQ_API_MDIO_Write(PHY_ID, 0x1E, 0xC47D, data);
      /* Write the target MMD and "write" command-type */
      AQ_API_MDIO_Write(PHY_ID, 0x1E, 0xC47E, MMD | (2 << 8));

      /* Poll the command-status register until we see "done" or "error" */
      commandStatus = 0;
      numCommandStatusPolls = 0;
      while (commandStatus == 0 && numCommandStatusPolls < MAX_NUM_COMMAND_STATUS_POLLS)
      {
        commandStatus = AQ_API_MDIO_Read(PHY_ID, 0x1E, 0xC47F);
        numCommandStatusPolls++;
      }

      if (numCommandStatusPolls >= MAX_NUM_COMMAND_STATUS_POLLS)
      {
        fprintf(stderr, "Warning: PIF Mailbox commandStatus polling limit exceeded, retrying.\n");
      }

      if (commandStatus == 2)
      {
        fprintf(stderr, "Warning: PIF Mailbox reported an error, retrying.\n");
      }

      if (commandStatus == 1) delegationSuccess = 1;
    }

    if (!delegationSuccess)
    {
      fprintf(stderr, "Error: PIF Mailbox retry limit succeeded, so giving up.\n");
    }

    return; 
  }


  switch (MMD)
  {
    case 0x0A:
      /* Read the dWord for the Read-Modify-Write */
      targetAddress = MMD_0x0A_BASE_ADDRESS + ((unsigned long) (address) << 1);  /* address is byte based, so need to multiply by 2 */
    #ifdef EXTRA_READ_BEFORE_WRITE
      prevValue = AQ_API_MDIO_Read_Abstracted (PHY_ID, 0x1E, 0x202);
      AQ_API_MDIO_Write_Abstracted (PHY_ID, 0x1E, 0x202, (targetAddress >> 16) & 0x0000FFFF);
      HW_Logging_MDIOWrite (HW_GetDefaultAdapter(), PHY_ID, 0x1E, 0x202, (targetAddress >> 16) & 0x0000FFFF, prevValue);
    #else
      AQ_API_MDIO_Write_Abstracted (PHY_ID, 0x1E, 0x202, (targetAddress >> 16) & 0x0000FFFF);
      HW_Logging_MDIOWrite (HW_GetDefaultAdapter(), PHY_ID, 0x1E, 0x202, (targetAddress >> 16) & 0x0000FFFF);
    #endif

    #ifdef EXTRA_READ_BEFORE_WRITE
      prevValue = AQ_API_MDIO_Read_Abstracted (PHY_ID, 0x1E, 0x203);
      AQ_API_MDIO_Write_Abstracted (PHY_ID, 0x1E, 0x203, targetAddress & 0x0000FFFF);
      HW_Logging_MDIOWrite (HW_GetDefaultAdapter(), PHY_ID, 0x1E, 0x203, targetAddress & 0x0000FFFF, prevValue);
    #else
      AQ_API_MDIO_Write_Abstracted (PHY_ID, 0x1E, 0x203, targetAddress & 0x0000FFFF);
      HW_Logging_MDIOWrite (HW_GetDefaultAdapter(), PHY_ID, 0x1E, 0x203, targetAddress & 0x0000FFFF);
    #endif

    #ifdef EXTRA_READ_BEFORE_WRITE
      prevValue = AQ_API_MDIO_Read_Abstracted (PHY_ID, 0x1E, 0x200);
      AQ_API_MDIO_Write_Abstracted (PHY_ID, 0x1E, 0x200, 0x8000);
      HW_Logging_MDIOWrite (HW_GetDefaultAdapter(), PHY_ID, 0x1E, 0x200, 0x8000, prevValue);
    #else
      AQ_API_MDIO_Write_Abstracted (PHY_ID, 0x1E, 0x200, 0x8000);
      HW_Logging_MDIOWrite (HW_GetDefaultAdapter(), PHY_ID, 0x1E, 0x200, 0x8000);
    #endif

    #ifdef ABSTRACTION_DEBUG
      printf ("Read from %.8lX\n", targetAddress);
    #endif
      while (pollCount++ <= MAX_POLL_COUNT)
      {
      #ifdef ABSTRACTION_DEBUG
        printf ("Poll count = %i\n", pollCount);
      #endif
        x = AQ_API_MDIO_Read_Abstracted (PHY_ID, 0x1E, 0x200);
        HW_Logging_MDIORead (HW_GetDefaultAdapter(), PHY_ID, 0x1E, 0x200, x);
        if ((x & 0x0100) == 0)
        {
          /* interface is ready: do the write */
        #ifdef EXTRA_READ_BEFORE_WRITE
          prevValue = AQ_API_MDIO_Read_Abstracted (PHY_ID, 0x1E, 0x202);
          AQ_API_MDIO_Write_Abstracted (PHY_ID, 0x1E, 0x202, (targetAddress >> 16) & 0x0000FFFF);
          HW_Logging_MDIOWrite (HW_GetDefaultAdapter(), PHY_ID, 0x1E, 0x202, (targetAddress >> 16) & 0x0000FFFF, prevValue);
        #else
          AQ_API_MDIO_Write_Abstracted (PHY_ID, 0x1E, 0x202, (targetAddress >> 16) & 0x0000FFFF);
          HW_Logging_MDIOWrite (HW_GetDefaultAdapter(), PHY_ID, 0x1E, 0x202, (targetAddress >> 16) & 0x0000FFFF);
        #endif

        #ifdef EXTRA_READ_BEFORE_WRITE
          prevValue = AQ_API_MDIO_Read_Abstracted (PHY_ID, 0x1E, 0x203);
          AQ_API_MDIO_Write_Abstracted (PHY_ID, 0x1E, 0x203, targetAddress & 0x0000FFFF);
          HW_Logging_MDIOWrite (HW_GetDefaultAdapter(), PHY_ID, 0x1E, 0x203, targetAddress & 0x0000FFFF, prevValue);
        #else
          AQ_API_MDIO_Write_Abstracted (PHY_ID, 0x1E, 0x203, targetAddress & 0x0000FFFF);
          HW_Logging_MDIOWrite (HW_GetDefaultAdapter(), PHY_ID, 0x1E, 0x203, targetAddress & 0x0000FFFF);
        #endif
          if (address & 0x1)
          {
          #ifdef ABSTRACTION_DEBUG
            printf ("Address is odd - writing 1E.0204\n");
          #endif

          #ifdef EXTRA_READ_BEFORE_WRITE
            prevValue = AQ_API_MDIO_Read_Abstracted (PHY_ID, 0x1E, 0x204);
            AQ_API_MDIO_Write_Abstracted (PHY_ID, 0x1E, 0x204, data);
            HW_Logging_MDIOWrite (HW_GetDefaultAdapter(), PHY_ID, 0x1E, 0x204, data, prevValue);
          #else
            AQ_API_MDIO_Write_Abstracted (PHY_ID, 0x1E, 0x204, data);
            HW_Logging_MDIOWrite (HW_GetDefaultAdapter(), PHY_ID, 0x1E, 0x204, data);
          #endif
          }
          else
          {
          #ifdef ABSTRACTION_DEBUG
            printf ("Address is even - writing 1E.0205\n");
          #endif

          #ifdef EXTRA_READ_BEFORE_WRITE
            prevValue = AQ_API_MDIO_Read_Abstracted (PHY_ID, 0x1E, 0x205);
            AQ_API_MDIO_Write_Abstracted (PHY_ID, 0x1E, 0x205, data);
            HW_Logging_MDIOWrite (HW_GetDefaultAdapter(), PHY_ID, 0x1E, 0x205, data, prevValue);
          #else
            AQ_API_MDIO_Write_Abstracted (PHY_ID, 0x1E, 0x205, data);
            HW_Logging_MDIOWrite (HW_GetDefaultAdapter(), PHY_ID, 0x1E, 0x205, data);
          #endif
          }

        #ifdef EXTRA_READ_BEFORE_WRITE
          prevValue = AQ_API_MDIO_Read_Abstracted (PHY_ID, 0x1E, 0x200);
          AQ_API_MDIO_Write_Abstracted (PHY_ID, 0x1E, 0x200, 0xC000);
          HW_Logging_MDIOWrite (HW_GetDefaultAdapter(), PHY_ID, 0x1E, 0x200, 0xC000, prevValue);
        #else
          AQ_API_MDIO_Write_Abstracted (PHY_ID, 0x1E, 0x200, 0xC000);
          HW_Logging_MDIOWrite (HW_GetDefaultAdapter(), PHY_ID, 0x1E, 0x200, 0xC000);
        #endif

          return;
        }
      }
      
      /* poll count exceeded */
      return;
      
    case 0x1C:
      #ifdef DO_VSEMI_SERDES_REDIRECTION
        /* Do indirect write of VESMI register (without delegating to FW). */

        if (/*address >= 0x0000 && */address <= 0x1FFF) /* Lane 0 */
        {
          x = (0 << 0xD) | (address - 0x0000);
        }
        else if (address >= 0x2000 && address <= 0x3FFF) /* Lane 1 */
        {
          x = (1 << 0xD) | (address - 0x2000);
        }
        else if (address >= 0x4000 && address <= 0x5FFF) /* Lane 2 */
        {
          x = (2 << 0xD) | (address - 0x4000);
        }
        else if (address >= 0x6000 && address <= 0x7FFF) /* Lane 3 */
        {
          x = (3 << 0xD) | (address - 0x6000);
        }
        else if (address >= 0x8000 && address <= 0x9FFF) /* Common */
        {
          x = (4 << 0xD) | (address - 0x8000);
        }
        else /* outside mapped ranges */
        {
          return;
        }

      #ifdef EXTRA_READ_BEFORE_WRITE
        prevValue = AQ_API_MDIO_Read_Abstracted (PHY_ID, 0x1E, 0x2D00);
        AQ_API_MDIO_Write_Abstracted (PHY_ID, 0x1E, 0x2D00, x);
        HW_Logging_MDIOWrite (HW_GetDefaultAdapter(), PHY_ID, 0x1E, 0x2D00, x, prevValue);
      #else
        AQ_API_MDIO_Write_Abstracted (PHY_ID, 0x1E, 0x2D00, x);
        HW_Logging_MDIOWrite (HW_GetDefaultAdapter(), PHY_ID, 0x1E, 0x2D00, x);
      #endif

        x = (1 << 0xF) | (data & 0xF);

      #ifdef EXTRA_READ_BEFORE_WRITE
        prevValue = AQ_API_MDIO_Read_Abstracted (PHY_ID, 0x1E, 0x2D01);
        AQ_API_MDIO_Write_Abstracted (PHY_ID, 0x1E, 0x2D01, x);
        HW_Logging_MDIOWrite (HW_GetDefaultAdapter(), PHY_ID, 0x1E, 0x2D01, x, prevValue);
      #else
        AQ_API_MDIO_Write_Abstracted (PHY_ID, 0x1E, 0x2D01, x);
        HW_Logging_MDIOWrite (HW_GetDefaultAdapter(), PHY_ID, 0x1E, 0x2D01, x);
      #endif

        return;
      #else /* Do SB redirection */
        /* perform a range check */
        if (address <= 0x05FF)
        {
          /* set the block pointer using a read-modify-write */
          x = AQ_API_MDIO_Read_Abstracted (PHY_ID, 0x4, 0xC180);
          HW_Logging_MDIORead (HW_GetDefaultAdapter(), PHY_ID, 0x4, 0xC180, x);
          x = x & 0xFFF8;
          x |= (address >> 8) & 0xF;

        #ifdef EXTRA_READ_BEFORE_WRITE
          prevValue = AQ_API_MDIO_Read_Abstracted (PHY_ID, 0x4, 0xC180);
          AQ_API_MDIO_Write_Abstracted (PHY_ID, 0x4, 0xC180, x);
          HW_Logging_MDIOWrite (HW_GetDefaultAdapter(), PHY_ID, 0x4, 0xC180, x, prevValue);
        #else
          AQ_API_MDIO_Write_Abstracted (PHY_ID, 0x4, 0xC180, x);
          HW_Logging_MDIOWrite (HW_GetDefaultAdapter(), PHY_ID, 0x4, 0xC180, x);
        #endif
          
          /* bit reverse the data */
          /* x = (((data & 0xAA) >> 1) | ((data & 0x55) << 1));          */
          /* x = (((x & 0xCC) >> 2) | ((x & 0x33) << 2));                */
          /* x = (((x & 0xF0) >> 4) | ((x & 0x0F) << 4));                */
        #ifdef EXTRA_READ_BEFORE_WRITE
          prevValue = AQ_API_MDIO_Read_Abstracted (PHY_ID, 0x4, 0xC200 | (address & 0xFF));
          AQ_API_MDIO_Write_Abstracted (PHY_ID, 0x4, 0xC200 | (address & 0xFF), x);
          HW_Logging_MDIOWrite (HW_GetDefaultAdapter(), PHY_ID, 0x4, 0xC200 | (address & 0xFF), x, prevValue);
        #else
          AQ_API_MDIO_Write_Abstracted (PHY_ID, 0x4, 0xC200 | (address & 0xFF), x);
          HW_Logging_MDIOWrite (HW_GetDefaultAdapter(), PHY_ID, 0x4, 0xC200 | (address & 0xFF), x);
        #endif
        }
        return;
      #endif
    
    default:
    #ifdef EXTRA_READ_BEFORE_WRITE
      prevValue = AQ_API_MDIO_Read_Abstracted (PHY_ID, MMD, address);
      AQ_API_MDIO_Write_Abstracted (PHY_ID, MMD, address, data);
      HW_Logging_MDIOWrite (HW_GetDefaultAdapter(), PHY_ID, MMD, address, data, prevValue);
    #else
      AQ_API_MDIO_Write_Abstracted (PHY_ID, MMD, address, data);
      HW_Logging_MDIOWrite (HW_GetDefaultAdapter(), PHY_ID, MMD, address, data);
    #endif
      return;
  }
}


uint32_t AQ_API_MDIO_Read
(
    /*! The address of the PHY being addressed. */
  uint32_t PHY_ID,
    /*! The address of the MMD within the PHY being addressed.*/
  uint32_t MMD,
    /*! The 32-bit address being written to.*/
  uint32_t address
)
{
  /* declare local variables */
  uint32_t pollCount = 0;
  uint32_t x, commandStatus, numCommandStatusPolls, 
               numDelegationRetries, delegationSuccess, delegatedReadVal;
  unsigned long targetAddress;
#ifndef DO_VSEMI_SERDES_REDIRECTION
  uint32_t lsw, msw;
#endif
#ifdef EXTRA_READ_BEFORE_WRITE
  uint32_t prevValue;
#endif

  if (shouldDelegateRead(MMD, address))
  {
    numDelegationRetries = 0;
    delegationSuccess = 0;
    while ((!delegationSuccess) && (numDelegationRetries < MAX_NUM_DELEGATION_RETRIES))
    {
      /* Clear the command status register */
      AQ_API_MDIO_Write(PHY_ID, 0x1E, 0xC47F, 0);
      /* Write the target address */
      AQ_API_MDIO_Write(PHY_ID, 0x1E, 0xC47C, address);
      /* Write the target MMD and "read" command-type */
      AQ_API_MDIO_Write(PHY_ID, 0x1E, 0xC47E, MMD | (1 << 8));

      /* Poll the command-status register until we see "done" or "error" */
      commandStatus = 0;
      numCommandStatusPolls = 0;
      while (commandStatus == 0 && numCommandStatusPolls < MAX_NUM_COMMAND_STATUS_POLLS)
      {
        commandStatus = AQ_API_MDIO_Read(PHY_ID, 0x1E, 0xC47F);
        numCommandStatusPolls++;
      }

      if (numCommandStatusPolls >= MAX_NUM_COMMAND_STATUS_POLLS)
      {
        fprintf(stderr, "Warning: PIF Mailbox commandStatus polling limit exceeded, retrying.\n");
        /* XXX: Debugging
         * Read back the target MMD and command-type register to see if the register write didn't go through.*/
        fprintf(stderr, "1E.C47E readback value: 0x%X\n",
            AQ_API_MDIO_Read(PHY_ID, 0x1E, 0xC47E));
      }

      if (commandStatus == 2)
      {
        fprintf(stderr, "Warning: PIF Mailbox reported an error, retrying.\n");
      }

      if (commandStatus == 1) delegationSuccess = 1;
    }

    if (!delegationSuccess)
    {
      fprintf(stderr, "Error: PIF Mailbox retry limit succeeded, so giving up.\n");
    }

    delegatedReadVal = AQ_API_MDIO_Read(PHY_ID, 0x1E, 0xC47D);
    return delegatedReadVal;
  }
 

  switch (MMD)
  {
    case 0x0A:
      /* Read the dWord  */
      targetAddress = MMD_0x0A_BASE_ADDRESS + ((unsigned long) (address) << 1);  /* address is byte based, so need to multiply by 2 */

    #ifdef EXTRA_READ_BEFORE_WRITE
      prevValue = AQ_API_MDIO_Read_Abstracted (PHY_ID, 0x1E, 0x202);
      AQ_API_MDIO_Write_Abstracted (PHY_ID, 0x1E, 0x202, (targetAddress >> 16) & 0x0000FFFF);
      HW_Logging_MDIOWrite (HW_GetDefaultAdapter(), PHY_ID, 0x1E, 0x202, (targetAddress >> 16) & 0x0000FFFF, prevValue);
    #else
      AQ_API_MDIO_Write_Abstracted (PHY_ID, 0x1E, 0x202, (targetAddress >> 16) & 0x0000FFFF);
      HW_Logging_MDIOWrite (HW_GetDefaultAdapter(), PHY_ID, 0x1E, 0x202, (targetAddress >> 16) & 0x0000FFFF);
    #endif

    #ifdef EXTRA_READ_BEFORE_WRITE
      prevValue = AQ_API_MDIO_Read_Abstracted (PHY_ID, 0x1E, 0x203);
      AQ_API_MDIO_Write_Abstracted (PHY_ID, 0x1E, 0x203, targetAddress & 0x0000FFFF);
      HW_Logging_MDIOWrite (HW_GetDefaultAdapter(), PHY_ID, 0x1E, 0x203, targetAddress & 0x0000FFFF, prevValue);
    #else
      AQ_API_MDIO_Write_Abstracted (PHY_ID, 0x1E, 0x203, targetAddress & 0x0000FFFF);
      HW_Logging_MDIOWrite (HW_GetDefaultAdapter(), PHY_ID, 0x1E, 0x203, targetAddress & 0x0000FFFF);
    #endif

    #ifdef EXTRA_READ_BEFORE_WRITE
      prevValue = AQ_API_MDIO_Read_Abstracted (PHY_ID, 0x1E, 0x200);
      AQ_API_MDIO_Write_Abstracted (PHY_ID, 0x1E, 0x200, 0x8000);
      HW_Logging_MDIOWrite (HW_GetDefaultAdapter(), PHY_ID, 0x1E, 0x200, 0x8000, prevValue);
    #else
      AQ_API_MDIO_Write_Abstracted (PHY_ID, 0x1E, 0x200, 0x8000);
      HW_Logging_MDIOWrite (HW_GetDefaultAdapter(), PHY_ID, 0x1E, 0x200, 0x8000);
    #endif

    #ifdef ABSTRACTION_DEBUG
      printf ("Read from %.8lX\n", targetAddress);
    #endif
      while (pollCount++ <= MAX_POLL_COUNT)
      {
      #ifdef ABSTRACTION_DEBUG
        printf ("Poll count = %i\n", pollCount);
      #endif
        x = AQ_API_MDIO_Read_Abstracted (PHY_ID, 0x1E, 0x200);
        HW_Logging_MDIORead (HW_GetDefaultAdapter(), PHY_ID, 0x1E, 0x200, x);
        if ((x & 0x0100) == 0)
        {
          /* interface is ready: read the answer */
          if (address & 0x1)
          {
          #ifdef ABSTRACTION_DEBUG
            printf ("Address is odd - reading from 1E.0204\n");
          #endif
            x = AQ_API_MDIO_Read_Abstracted (PHY_ID, 0x1E, 0x204);
            HW_Logging_MDIORead (HW_GetDefaultAdapter(), PHY_ID, 0x1E, 0x204, x);

            return x;
          }
          else
          {
          #ifdef ABSTRACTION_DEBUG
            printf ("Address is even - reading from 1E.0205\n");
          #endif
            x = AQ_API_MDIO_Read_Abstracted (PHY_ID, 0x1E, 0x205);
            HW_Logging_MDIORead (HW_GetDefaultAdapter(), PHY_ID, 0x1E, 0x205, x);

            return x;
          }
        }
      }
      
      /* poll count exceeded */
      return 0x0000;
      
    case 0x1C:
      #ifdef DO_VSEMI_SERDES_REDIRECTION
        /* Do indirect read of VSEMI register (without delegating to FW). */

        if( /*address >= 0x0000 && */address <= 0x1FFF ) /* Lane 0 */
        {
          x = (0 << 0xD) | (address - 0x0000);
        }
        else if (address >= 0x2000 && address <= 0x3FFF) /* Lane 1 */
        {
          x = (1 << 0xD) | (address - 0x2000);
        }
        else if (address >= 0x4000 && address <= 0x5FFF) /* Lane 2 */
        {
          x = (2 << 0xD) | (address - 0x4000);
        }
        else if (address >= 0x6000 && address <= 0x7FFF) /* Lane 3 */
        {
          x = (3 << 0xD) | (address - 0x6000);
        }
        else if (address >= 0x8000 && address <= 0x9FFF) /* Common */
        {
          x = (4 << 0xD) | (address - 0x8000);
        }
        else /* outside mapped ranges */
        {
          return 0;
        }

      #ifdef EXTRA_READ_BEFORE_WRITE
        prevValue = AQ_API_MDIO_Read_Abstracted (PHY_ID, 0x1E, 0x2D00);
        AQ_API_MDIO_Write_Abstracted (PHY_ID, 0x1E, 0x2D00, x);
        HW_Logging_MDIOWrite (HW_GetDefaultAdapter(), PHY_ID, 0x1E, 0x2D00, x, prevValue);
      #else
        AQ_API_MDIO_Write_Abstracted (PHY_ID, 0x1E, 0x2D00, x);
        HW_Logging_MDIOWrite (HW_GetDefaultAdapter(), PHY_ID, 0x1E, 0x2D00, x);
      #endif

        x = 1 << 0xF;

      #ifdef EXTRA_READ_BEFORE_WRITE
        prevValue = AQ_API_MDIO_Read_Abstracted (PHY_ID, 0x1E, 0x2D02);
        AQ_API_MDIO_Write_Abstracted (PHY_ID, 0x1E, 0x2D02, x);
        HW_Logging_MDIOWrite (HW_GetDefaultAdapter(), PHY_ID, 0x1E, 0x2D02, x, prevValue);
      #else
        AQ_API_MDIO_Write_Abstracted (PHY_ID, 0x1E, 0x2D02, x);
        HW_Logging_MDIOWrite (HW_GetDefaultAdapter(), PHY_ID, 0x1E, 0x2D02, x);
      #endif

        x = AQ_API_MDIO_Read_Abstracted (PHY_ID, 0x1E, 0x2D02);
        HW_Logging_MDIORead (HW_GetDefaultAdapter(), PHY_ID, 0x1E, 0x2D02, x);

        if (!(x & 0x4000)) fprintf(stderr, "SERDES Read ACK bit not set when retrieving value (0x%04X).\n", x);

        return x & 0xFF;
      #else /* Do SB redirection */
        /* check if we are accessing the test-bus or not */
        if (address & 0x8000)
        {
          /* test bus access */
          /* perform a range check */
          if ((((address >> 8) & 0xF) < 0x5) && ((address & 0xFF) < 26))
          {
            /* write the pointer for the CMU = 0x0 */
            x = AQ_API_MDIO_Read_Abstracted (PHY_ID, 0x4, 0xC180);
            HW_Logging_MDIORead (HW_GetDefaultAdapter(), PHY_ID, 0x4, 0xC180, x);
            x = x & 0xFFF8;
          #ifdef EXTRA_READ_BEFORE_WRITE
            prevValue = AQ_API_MDIO_Read_Abstracted (PHY_ID, 0x4, 0xC180);
            AQ_API_MDIO_Write_Abstracted (PHY_ID, 0x4, 0xC180, x);
            HW_Logging_MDIOWrite (HW_GetDefaultAdapter(), PHY_ID, 0x4, 0xC180, x, prevValue);
          #else
            AQ_API_MDIO_Write_Abstracted (PHY_ID, 0x4, 0xC180, x);
            HW_Logging_MDIOWrite (HW_GetDefaultAdapter(), PHY_ID, 0x4, 0xC180, x);
          #endif
            
            /* create the indirection pointer and bit-reverse it */
            x = (((address >> 8) & 0xF) << 5) | (address & 0xFF);
            x = (((x & 0xAA) >> 1) | ((x & 0x55) << 1));
            x = (((x & 0xCC) >> 2) | ((x & 0x33) << 2));
            x = (((x & 0xF0) >> 4) | ((x & 0x0F) << 4));
          #ifdef EXTRA_READ_BEFORE_WRITE
            prevValue = AQ_API_MDIO_Read_Abstracted (PHY_ID, 0x4, 0xC20B);
            AQ_API_MDIO_Write_Abstracted (PHY_ID, 0x4, 0xC20B, x);
            HW_Logging_MDIOWrite (HW_GetDefaultAdapter(), PHY_ID, 0x4, 0xC20B, x, prevValue);
          #else
            AQ_API_MDIO_Write_Abstracted (PHY_ID, 0x4, 0xC20B, x);
            HW_Logging_MDIOWrite (HW_GetDefaultAdapter(), PHY_ID, 0x4, 0xC20B, x);
          #endif
            
            //read the answer
            lsw = AQ_API_MDIO_Read_Abstracted (PHY_ID, 0x4, 0xC2EF);
            HW_Logging_MDIORead (HW_GetDefaultAdapter(), PHY_ID, 0x4, 0xC2EF, lsw);
            msw = AQ_API_MDIO_Read_Abstracted (PHY_ID, 0x4, 0xC2FF);
            HW_Logging_MDIORead (HW_GetDefaultAdapter(), PHY_ID, 0x4, 0xC2FF, msw);
            return ((msw & 0xF) << 8) | lsw;
          }
        }
        else
        {
          /* regular access */
          /* perform a range check */
          if (address <= 0x05FF)
          {
            /* set the block pointer using a read-modify-write */
            x = AQ_API_MDIO_Read_Abstracted (PHY_ID, 0x4, 0xC180);
            HW_Logging_MDIORead (HW_GetDefaultAdapter(), PHY_ID, 0x4, 0xC180, x);
            x = x & 0xFFF8;
            x |= (address >> 8) & 0xF;
          #ifdef EXTRA_READ_BEFORE_WRITE
            prevValue = AQ_API_MDIO_Read_Abstracted (PHY_ID, 0x4, 0xC180);
            AQ_API_MDIO_Write_Abstracted (PHY_ID, 0x4, 0xC180, x);
            HW_Logging_MDIOWrite (HW_GetDefaultAdapter(), PHY_ID, 0x4, 0xC180, x, prevValue);
          #else
            AQ_API_MDIO_Write_Abstracted (PHY_ID, 0x4, 0xC180, x);
            HW_Logging_MDIOWrite (HW_GetDefaultAdapter(), PHY_ID, 0x4, 0xC180, x);
          #endif
            
            /* read the data and bit-reverse it */
            /* x = AQ_API_MDIO_Read_Abstracted (PHY_ID, 0x4, 0xC200 | (address & 0xFF));   */
            /* x = (((x & 0xAA) >> 1) | ((x & 0x55) << 1));                                */
            /* x = (((x & 0xCC) >> 2) | ((x & 0x33) << 2));                                */
            return (((x & 0xF0) >> 4) | ((x & 0x0F) << 4));
          }
        }
        return 0x0000;
      #endif
          
    default:
      x = AQ_API_MDIO_Read_Abstracted (PHY_ID, MMD, address);
      HW_Logging_MDIORead (HW_GetDefaultAdapter(), PHY_ID, MMD, address, x);
      return x;
  }
}


void AQ_API_MDIO_BlockWrite
(
    /*! The address of the PHY being addressed. This is ultimately some combination of the <BR>
        system address and the target PHY's MDIO Address, set by the pins Addr [4:0].*/
  uint32_t PHY_ID,
    /*! The address of the MMD within the Aquantia PHY being addressed.*/
  uint32_t MMD,
    /*! The 16-bit address being written to.*/
  uint32_t address,
    /*! The 16-bits of data being sent out with the write command. This may be either address or data.*/
  uint32_t data
)
{
  switch (MMD)
  {
    case 0x0A:
    case 0x1C:
      /* unsupported */
      return;
      
    default:
      AQ_API_MDIO_BlockWrite_Abstracted (PHY_ID, MMD, address, data);
      HW_Logging_BlockMDIOWrite (HW_GetDefaultAdapter(), PHY_ID, MMD, address, data);
      return;
  }
}


void AQ_API_MDIO_BlockRead
(
    /*! The address of the PHY being addressed. This is ultimately some combination of the <BR>
        system address and the target PHY's MDIO Address, set by the pins Addr [4:0].*/
  uint32_t PHY_ID,
    /*! The address of the MMD within the Aquantia PHY being addressed.*/
  uint32_t MMD,
    /*! The 16-bit address being written to.*/
  uint32_t address
)
{
  switch (MMD)
  {
    case 0x0A:
    case 0x1C:
      /* unsupported */
      return;
      
    default:
      AQ_API_MDIO_BlockRead_Abstracted (PHY_ID, MMD, address);
      HW_Logging_BlockMDIORead (HW_GetDefaultAdapter(), PHY_ID, MMD, address);
      return;
  }
}


void HW_Write
(
    /*! The number of the adapter in the HW_FilteredAdapterList. */
  uint32_t adapterNumber,
    /*! The address of the PHY being addressed. */
  uint32_t PHY_ID,
    /*! The address of the MMD within the PHY being addressed.*/
  uint32_t MMD,
    /*! The 16-bit address being written to.*/
  uint32_t address,
    /*! The 16-bits of data being sent out with the write command. This may be either address or data.*/
  uint32_t data
)
{
#ifdef EXTRA_READ_BEFORE_WRITE
  uint32_t prevValue;
#endif

  switch (MMD)
  {
    case 0x0A:
    case 0x1C:
      /* unsupported */
      return;
      
    default:
    #ifdef EXTRA_READ_BEFORE_WRITE
      /* Do an extra read to get the previous value of the register, before
       * doing the write.  Note that we're intentionally sneaking this read in 
       * "under the logging" by calling HW_Read_Abstracted instead of HW_Read */
      prevValue = HW_Read_Abstracted (adapterNumber, PHY_ID, MMD, address);
      HW_Write_Abstracted (adapterNumber, PHY_ID, MMD, address, data);
      HW_Logging_MDIOWrite (adapterNumber, PHY_ID, MMD, address, data, prevValue);
    #else
      HW_Write_Abstracted (adapterNumber, PHY_ID, MMD, address, data);
      HW_Logging_MDIOWrite (adapterNumber, PHY_ID, MMD, address, data);
    #endif
      return;
  }
}


uint32_t HW_Read
(
    /*! The number of the adapter in the HW_FilteredAdapterList. */
  uint32_t adapterNumber,
    /*! The address of the PHY being addressed. */
  uint32_t PHY_ID,
    /*! The address of the MMD within the PHY being addressed.*/
  uint32_t MMD,
    /*! The 16-bit address being written to.*/
  uint32_t address
)
{
  uint32_t x;
  switch (MMD)
  {
    case 0x0A:
    case 0x1C:
      /* unsupported */
      return 0x0000;
      
    default:
      x = HW_Read_Abstracted (adapterNumber, PHY_ID, MMD, address);
      HW_Logging_MDIORead (adapterNumber, PHY_ID, MMD, address, x);
      return x;
  }
}


void HW_BlockRead (uint32_t PHY_ID, uint32_t MMD, uint32_t address)
{
  switch (MMD)
  {
    case 0x0A:
    case 0x1C:
      /* unsupported */
      return;
      
    default:
      HW_BlockRead_Abstracted (PHY_ID, MMD, address);
      HW_Logging_BlockMDIORead (HW_GetDefaultAdapter(), PHY_ID, MMD, address);
      return;
  }
}


void HW_BlockWrite (uint32_t PHY_ID, uint32_t MMD, uint32_t address, uint32_t data)
{
  switch (MMD)
  {
    case 0x0A:
    case 0x1C:
      /* unsupported */
      return;
      
    default:
      HW_BlockWrite_Abstracted (PHY_ID, MMD, address, data);
      HW_Logging_BlockMDIOWrite (HW_GetDefaultAdapter(), PHY_ID, MMD, address, data);
      return;
  }
}


void HW_BlockReadAdapter (uint32_t adapterNumber, uint32_t PHY_ID, uint32_t MMD, uint32_t address)
{
  switch (MMD)
  {
    case 0x0A:
    case 0x1C:
      /* unsupported */
      return;
      
    default:
      HW_BlockReadAdapter_Abstracted (adapterNumber, PHY_ID, MMD, address);
      HW_Logging_BlockMDIORead (adapterNumber, PHY_ID, MMD, address);
      return;
  }
}


void HW_BlockWriteAdapter (uint32_t adapterNumber, uint32_t PHY_ID, uint32_t MMD,
                                      uint32_t address, uint32_t data)
{
  switch (MMD)
  {
    case 0x0A:
    case 0x1C:
      /* unsupported */
      return;
      
    default:
      HW_BlockWriteAdapter_Abstracted (adapterNumber, PHY_ID, MMD, address, data);
      HW_Logging_BlockMDIOWrite (adapterNumber, PHY_ID, MMD, address, data);
      return;
  }
}


/*! The current default PHY_ID that will be used during calls to 
 * ..._Default_PHY_ID functions. */
uint32_t Default_PHY_ID;


uint32_t HW_GetDefault_PHY_ID()
{
	return Default_PHY_ID;
}


void HW_SetDefault_PHY_ID 
(
  const uint32_t PHY_ID   /*!< The default PHY_ID */
)
{
  Default_PHY_ID = PHY_ID;
}


void AQ_API_MDIO_Write_Default_PHY_ID
(
    /*! The address of the MMD within the PHY being addressed.*/
  uint32_t MMD,
    /*! The 16-bit address being written to.*/
  uint32_t address,
    /*! The 16-bits of data being sent out with the write command. This may be either address or data.*/
  uint32_t data
)
{
  AQ_API_MDIO_Write (HW_GetDefault_PHY_ID(), MMD, address, data);
}


uint32_t AQ_API_MDIO_Read_Default_PHY_ID
(
    /*! The address of the MMD within the PHY being addressed.*/
  uint32_t MMD,
    /*! The 16-bit address being written to.*/
  uint32_t address
)
{
  return AQ_API_MDIO_Read (HW_GetDefault_PHY_ID(), MMD, address);
}


void AQ_API_MDIO_BlockWrite_Default_PHY_ID
(
    /*! The address of the MMD within the Aquantia PHY being addressed.*/
  uint32_t MMD,
    /*! The 16-bit address being written to.*/
  uint32_t address,
    /*! The 16-bits of data being sent out with the write command. This may be either address or data.*/
  uint32_t data
)
{
  AQ_API_MDIO_BlockWrite (HW_GetDefault_PHY_ID(), MMD, address, data);
}


void AQ_API_MDIO_BlockRead_Default_PHY_ID
(
    /*! The address of the MMD within the Aquantia PHY being addressed.*/
  uint32_t MMD,
    /*! The 16-bit address being written to.*/
  uint32_t address
)
{
  AQ_API_MDIO_BlockRead (HW_GetDefault_PHY_ID(), MMD, address);
}


void HW_BlockRead_Default_PHY_ID (uint32_t MMD, uint32_t address)
{
  HW_BlockRead (HW_GetDefault_PHY_ID(), MMD, address);
}


void HW_BlockWrite_Default_PHY_ID (uint32_t MMD, uint32_t address, uint32_t data)
{
  HW_BlockWrite (HW_GetDefault_PHY_ID(), MMD, address, data);
}


uint32_t HW_Read_Default_PHY_ID (uint32_t MMD, uint32_t address)
{
  return AQ_API_MDIO_Read (HW_GetDefault_PHY_ID(), MMD, address);
}


void HW_Write_Default_PHY_ID (uint32_t MMD, uint32_t address, uint32_t data)
{
  AQ_API_MDIO_Write (HW_GetDefault_PHY_ID(), MMD, address, data);
}


void HW_API_Write8
(
	/*! The address of the MAC being addressed. */
	uint32_t DEV_ID,
	/*! The 32-bit address being written to.*/
	uint32_t address,
	/*! The 8-bits of data being sent out with the write command. This may be either address or data.*/
	uint8_t data
)
{
	/* declare local variables */
#ifdef EXTRA_READ_BEFORE_WRITE
	uint8_t prevValue;
	prevValue = AQ_API_Read8_Abstracted(DEV_ID, address);
	AQ_API_Write8_Abstracted(DEV_ID, address, data);
	HW_Logging_Write32(HW_GetDefaultAdapter(), DEV_ID, address, data, prevValue);
#else
	AQ_API_Write8_Abstracted(DEV_ID, address, data);
	HW_Logging_Write32(HW_GetDefaultAdapter(), DEV_ID, address, data);
#endif
	return;
}

void HW_API_Write16
(
	/*! The address of the MAC being addressed. */
	uint32_t DEV_ID,
	/*! The 32-bit address being written to.*/
	uint32_t address,
	/*! The 16-bits of data being sent out with the write command. This may be either address or data.*/
	uint16_t data
)
{
	/* declare local variables */
#ifdef EXTRA_READ_BEFORE_WRITE
	uint16_t prevValue;
	prevValue = AQ_API_Read16_Abstracted(DEV_ID, address);
	AQ_API_Write16_Abstracted(DEV_ID, address, data);
	HW_Logging_Write32(HW_GetDefaultAdapter(), DEV_ID, address, data, prevValue);
#else
	AQ_API_Write16_Abstracted(DEV_ID, address, data);
	HW_Logging_Write32(HW_GetDefaultAdapter(), DEV_ID, address, data);
#endif
	return;
}

void HW_API_Write32
(
/*! The address of the MAC being addressed. */
uint32_t DEV_ID,
/*! The 32-bit address being written to.*/
uint32_t address,
/*! The 32-bits of data being sent out with the write command. This may be either address or data.*/
uint32_t data
)
{
    /* declare local variables */
#ifdef EXTRA_READ_BEFORE_WRITE
    uint32_t prevValue;
    prevValue = AQ_API_Read32_Abstracted( DEV_ID, address );
    AQ_API_Write32_Abstracted( DEV_ID, address, data );
    HW_Logging_Write32( HW_GetDefaultAdapter(), DEV_ID, address, data, prevValue );
#else
    AQ_API_Write32_Abstracted( DEV_ID, address, data );
    HW_Logging_Write32( HW_GetDefaultAdapter(), DEV_ID, address, data );
#endif
    return;
}


uint8_t HW_API_Read8
(
	/*! The address of the MAC being addressed. */
	uint32_t DEV_ID,
	/*! The 32-bit address being written to.*/
	uint32_t address
)
{
	uint8_t x;
	/* declare local variables */

	x = AQ_API_Read8_Abstracted(DEV_ID, address);
	HW_Logging_Read32(HW_GetDefaultAdapter(), DEV_ID, address, x);
	return x;
}

uint16_t HW_API_Read16
(
	/*! The address of the MAC being addressed. */
	uint32_t DEV_ID,
	/*! The 32-bit address being written to.*/
	uint32_t address
)
{
	uint16_t x;
	/* declare local variables */

	x = AQ_API_Read16_Abstracted(DEV_ID, address);
	HW_Logging_Read32(HW_GetDefaultAdapter(), DEV_ID, address, x);
	return x;
}


uint32_t HW_API_Read32
(
/*! The address of the MAC being addressed. */
uint32_t DEV_ID,
/*! The 32-bit address being written to.*/
uint32_t address
)
{
	uint32_t x;
    /* declare local variables */

	x = AQ_API_Read32_Abstracted( DEV_ID, address );
    HW_Logging_Read32( HW_GetDefaultAdapter(), DEV_ID, address, x );
    return x;
}

void AQ_API_BlockWrite32
(
/*! The address of the PHY being addressed. This is ultimately some combination of the <BR>
system address and the target DEV's MDIO Address, set by the pins Addr [4:0].*/
uint32_t DEV_ID,
/*! The 32-bit address being written to.*/
uint32_t address,
/*! The 32-bits of data being sent out with the write command. This may be either address or data.*/
uint32_t data
)
{
    AQ_API_BlockWrite32_Abstracted( DEV_ID, address, data );
    HW_Logging_BlockWrite32( HW_GetDefaultAdapter(), DEV_ID, address, data );
}


void AQ_API_BlockRead32
(
/*! The address of the PHY being addressed. This is ultimately some combination of the <BR>
system address and the target DEV's MDIO Address, set by the pins Addr [4:0].*/
uint32_t DEV_ID,
/*! The 32-bit address being written to.*/
uint32_t address
)
{
    AQ_API_BlockRead32_Abstracted( DEV_ID, address );
    HW_Logging_BlockRead32( HW_GetDefaultAdapter(), DEV_ID, address );
    return;
}

void HW_Write8
(
	/*! The number of the adapter in the HW_FilteredAdapterList. */
	uint32_t adapterNumber,
	/*! The address of the PHY being addressed. This is ultimately some combination of the <BR>
	system address and the target DEV's MDIO Address, set by the pins Addr [4:0].*/
	uint32_t DEV_ID,
	/*! The 32-bit address being written to.*/
	uint32_t address,
	/*! The 8-bits of data being sent out with the write command. This may be either address or data.*/
	uint8_t data
)
{
#ifdef EXTRA_READ_BEFORE_WRITE
	uint8_t prevValue;
#endif

#ifdef EXTRA_READ_BEFORE_WRITE
	/* Do an extra read to get the previous value of the register, before
	* doing the write.  Note that we're intentionally sneaking this read in
	* "under the logging" by calling HW_Read_Abstracted instead of HW_Read */
	prevValue = HW_Read8_Abstracted(adapterNumber, DEV_ID, address);
	HW_Write8_Abstracted(adapterNumber, DEV_ID, address, data);
	HW_Logging_Write32(adapterNumber, DEV_ID, address, data, prevValue);
#else
	HW_Write8_Abstracted(adapterNumber, DEV_ID, address, data);
	HW_Logging_Write32(adapterNumber, DEV_ID, address, data);
#endif
	return;
}

void HW_Write16
(
	/*! The number of the adapter in the HW_FilteredAdapterList. */
	uint32_t adapterNumber,
	/*! The address of the PHY being addressed. This is ultimately some combination of the <BR>
	system address and the target DEV's MDIO Address, set by the pins Addr [4:0].*/
	uint32_t DEV_ID,
	/*! The 32-bit address being written to.*/
	uint32_t address,
	/*! The 16-bits of data being sent out with the write command. This may be either address or data.*/
	uint16_t data
)
{
#ifdef EXTRA_READ_BEFORE_WRITE
	uint16_t prevValue;
#endif

#ifdef EXTRA_READ_BEFORE_WRITE
	/* Do an extra read to get the previous value of the register, before
	* doing the write.  Note that we're intentionally sneaking this read in
	* "under the logging" by calling HW_Read_Abstracted instead of HW_Read */
	prevValue = HW_Read16_Abstracted(adapterNumber, DEV_ID, address);
	HW_Write16_Abstracted(adapterNumber, DEV_ID, address, data);
	HW_Logging_Write32(adapterNumber, DEV_ID, address, data, prevValue);
#else
	HW_Write16_Abstracted(adapterNumber, DEV_ID, address, data);
	HW_Logging_Write32(adapterNumber, DEV_ID, address, data);
#endif
	return;
}

void HW_Write32
(
/*! The number of the adapter in the HW_FilteredAdapterList. */
uint32_t adapterNumber,
/*! The address of the PHY being addressed. This is ultimately some combination of the <BR>
system address and the target DEV's MDIO Address, set by the pins Addr [4:0].*/
uint32_t DEV_ID,
/*! The 32-bit address being written to.*/
uint32_t address,
/*! The 32-bits of data being sent out with the write command. This may be either address or data.*/
uint32_t data
)
{
#ifdef EXTRA_READ_BEFORE_WRITE
    uint32_t prevValue;
#endif

#ifdef EXTRA_READ_BEFORE_WRITE
    /* Do an extra read to get the previous value of the register, before
    * doing the write.  Note that we're intentionally sneaking this read in
    * "under the logging" by calling HW_Read_Abstracted instead of HW_Read */
    prevValue = HW_Read32_Abstracted( adapterNumber, DEV_ID, address );
    HW_Write32_Abstracted( adapterNumber, DEV_ID, address, data );
    HW_Logging_Write32( adapterNumber, DEV_ID, address, data, prevValue );
#else
    HW_Write32_Abstracted( adapterNumber, DEV_ID, address, data );
    HW_Logging_Write32( adapterNumber, DEV_ID, address, data );
#endif
    return;
}

uint8_t HW_Read8
(
	/*! The number of the adapter in the HW_FilteredAdapterList. */
	uint32_t adapterNumber,
	/*! The address of the PHY being addressed. This is ultimately some combination of the <BR>
	system address and the target DEV's MDIO Address, set by the pins Addr [4:0].*/
	uint32_t DEV_ID,
	/*! The 32-bit address being written to.*/
	uint32_t address
)
{
	uint8_t x;
	x = HW_Read8_Abstracted(adapterNumber, DEV_ID, address);
	HW_Logging_Read32(adapterNumber, DEV_ID, address, x);
	return x;
}
uint16_t HW_Read16
(
	/*! The number of the adapter in the HW_FilteredAdapterList. */
	uint32_t adapterNumber,
	/*! The address of the PHY being addressed. This is ultimately some combination of the <BR>
	system address and the target DEV's MDIO Address, set by the pins Addr [4:0].*/
	uint32_t DEV_ID,
	/*! The 32-bit address being written to.*/
	uint32_t address
)
{
	uint16_t x;
	x = HW_Read16_Abstracted(adapterNumber, DEV_ID, address);
	HW_Logging_Read32(adapterNumber, DEV_ID, address, x);
	return x;
}

uint32_t HW_Read32
(
/*! The number of the adapter in the HW_FilteredAdapterList. */
uint32_t adapterNumber,
/*! The address of the PHY being addressed. This is ultimately some combination of the <BR>
system address and the target DEV's MDIO Address, set by the pins Addr [4:0].*/
uint32_t DEV_ID,
/*! The 32-bit address being written to.*/
uint32_t address
)
{
    uint32_t x;
    x = HW_Read32_Abstracted( adapterNumber, DEV_ID, address );
    HW_Logging_Read32( adapterNumber, DEV_ID, address, x );
    return x;
}

void HW_BlockRead32( uint32_t DEV_ID, uint32_t address )
{
    HW_BlockRead32_Abstracted( DEV_ID, address );
    HW_Logging_BlockRead32( HW_GetDefaultAdapter(), DEV_ID, address );
    return;
}


void HW_BlockWrite32( uint32_t DEV_ID, uint32_t address, uint32_t data )
{
    HW_BlockWrite32_Abstracted( DEV_ID, address, data );
    HW_Logging_BlockWrite32( HW_GetDefaultAdapter(), DEV_ID, address, data );
    return;
}


void HW_BlockRead32Adapter( uint32_t adapterNumber, uint32_t DEV_ID, uint32_t address )
{
    HW_BlockRead32Adapter_Abstracted( adapterNumber, DEV_ID, address );
    HW_Logging_BlockRead32( adapterNumber, DEV_ID, address );
    return;
}


void HW_BlockWrite32Adapter( uint32_t adapterNumber, uint32_t DEV_ID, uint32_t address, uint32_t data )
{
    HW_BlockWrite32Adapter_Abstracted( adapterNumber, DEV_ID, address, data );
    HW_Logging_BlockWrite32( adapterNumber, DEV_ID, address, data );
    return;
}

void HW_API_Write64
(
/*! The address of the MAC being addressed. */
uint32_t DEV_ID,
/*! The 32-bit address being written to.*/
uint32_t address,
/*! The 64-bits of data being sent out with the write command. This may be either address or data.*/
uint64_t data
)
{
#ifdef EXTRA_READ_BEFORE_WRITE
    uint64_t prevValue;
    prevValue = AQ_API_Read64_Abstracted( DEV_ID, address );
    AQ_API_Write64_Abstracted( DEV_ID, address, data );
    HW_Logging_Write64( HW_GetDefaultAdapter(), DEV_ID, address, data, prevValue );
#else
    AQ_API_Write64_Abstracted( DEV_ID, address, data );
    HW_Logging_Write64( HW_GetDefaultAdapter(), DEV_ID, address, data );
#endif
    return;
}


uint64_t HW_API_Read64
(
/*! The address of the MAC being addressed. */
uint32_t DEV_ID,
/*! The 32-bit address being written to.*/
uint32_t address
)
{
    /* declare local variables */
    uint64_t x;
    x = AQ_API_Read64_Abstracted( DEV_ID, address );
    HW_Logging_Read64( HW_GetDefaultAdapter(), DEV_ID, address, x );
    return x;
}

void AQ_API_BlockWrite64
(
/*! The address of the PHY being addressed. This is ultimately some combination of the <BR>
system address and the target DEV's MDIO Address, set by the pins Addr [4:0].*/
uint32_t DEV_ID,
/*! The 32-bit address being written to.*/
uint32_t address,
/*! The 64-bits of data being sent out with the write command. This may be either address or data.*/
uint64_t data
)
{
    AQ_API_BlockWrite64_Abstracted( DEV_ID, address, data );
    HW_Logging_BlockWrite64( HW_GetDefaultAdapter(), DEV_ID, address, data );
}


void AQ_API_BlockRead64
(
/*! The address of the PHY being addressed. This is ultimately some combination of the <BR>
system address and the target DEV's MDIO Address, set by the pins Addr [4:0].*/
uint32_t DEV_ID,
/*! The 32-bit address being written to.*/
uint32_t address
)
{
    AQ_API_BlockRead64_Abstracted( DEV_ID, address );
    HW_Logging_BlockRead64( HW_GetDefaultAdapter(), DEV_ID, address );
    return;
}

void HW_Write64
(
/*! The number of the adapter in the HW_FilteredAdapterList. */
uint32_t adapterNumber,
/*! The address of the PHY being addressed. This is ultimately some combination of the <BR>
system address and the target DEV's MDIO Address, set by the pins Addr [4:0].*/
uint32_t DEV_ID,
/*! The 32-bit address being written to.*/
uint32_t address,
/*! The 64-bits of data being sent out with the write command. This may be either address or data.*/
uint64_t data
)
{
#ifdef EXTRA_READ_BEFORE_WRITE
    uint64_t prevValue;
#endif

#ifdef EXTRA_READ_BEFORE_WRITE
    /* Do an extra read to get the previous value of the register, before
    * doing the write.  Note that we're intentionally sneaking this read in
    * "under the logging" by calling HW_Read_Abstracted instead of HW_Read */
    prevValue = HW_Read64_Abstracted( adapterNumber, DEV_ID, address );
    HW_Write64_Abstracted( adapterNumber, DEV_ID, address, data );
    HW_Logging_Write64( adapterNumber, DEV_ID, address, data, prevValue );
#else
    HW_Write64_Abstracted( adapterNumber, DEV_ID, address, data );
    HW_Logging_Write64( adapterNumber, DEV_ID, address, data );
#endif
    return;
}

uint64_t HW_Read64
(
/*! The number of the adapter in the HW_FilteredAdapterList. */
uint32_t adapterNumber,
/*! The address of the PHY being addressed. This is ultimately some combination of the <BR>
system address and the target DEV's MDIO Address, set by the pins Addr [4:0].*/
uint32_t DEV_ID,
/*! The 64-bit address being written to.*/
uint32_t address
)
{
    uint64_t x;
    x = HW_Read64_Abstracted( adapterNumber, DEV_ID, address );
    HW_Logging_Read64( adapterNumber, DEV_ID, address, x );
    return x;
}

void HW_BlockRead64( uint32_t DEV_ID, uint32_t address )
{
    HW_BlockRead64_Abstracted( DEV_ID, address );
    HW_Logging_BlockRead64( HW_GetDefaultAdapter(), DEV_ID, address );
    return;
}


void HW_BlockWrite64( uint32_t DEV_ID, uint32_t address, uint64_t data )
{
    HW_BlockWrite64_Abstracted( DEV_ID, address, data );
    HW_Logging_BlockWrite64( HW_GetDefaultAdapter(), DEV_ID, address, data );
    return;
}


void HW_BlockRead64Adapter( uint32_t adapterNumber, uint32_t DEV_ID, uint32_t address )
{
    HW_BlockRead64Adapter_Abstracted( adapterNumber, DEV_ID, address );
    HW_Logging_BlockRead64( adapterNumber, DEV_ID, address );
    return;
}


void HW_BlockWrite64Adapter( uint32_t adapterNumber, uint32_t DEV_ID, uint32_t address, uint64_t data )
{
    HW_BlockWrite64Adapter_Abstracted( adapterNumber, DEV_ID, address, data );
    HW_Logging_BlockWrite64( adapterNumber, DEV_ID, address, data );
    return;
}
