//
//  mdioAccessThroughMac.c
//  MBU-Dependencies
//
//  Created by Egor Pomozov on 18/04/2018.
//  Copyright © 2018 Aquantia Corp. All rights reserved.
//

#ifdef _WINDLL
#include "hwInterfaceWin.h"
#elif defined(__linux__)
#include "hwInterfaceLinux.h"
#else //macOS
#include "hwInterfaceMacOs.h"
#endif

#include "hwInterface.h"
#include "hwInterfacePci.h"
#include "mdioAccessThroughMac.h"

#define MDIO_PAR_MASK_MASK 0x0E000000
#define MDIO_PAR_VAL_MASK 0x70000000
#define MDIO_PAR_OFF_MASK 0x01F00000
#define MDIO_ADDRESS_MASK 0x000FFFFF
#define MDIO_TIMEOUT_MASK 0x0000FFFF
#define MDIO_REPEAT_MASK 0xFFFF0000
#define MDIO_PAR_MASK_OFFSET 25
#define MDIO_PAR_VAL_OFFSET 28
#define MDIO_PAR_OFF_OFFSET 20
#define MDIO_ADDRESS_OFFSET 0
#define MDIO_TIMEOUT_OFFSET 0
#define MDIO_REPEAT_OFFSET 16

typedef struct bitField
{
    uint32_t Mask;
    uint32_t Value;
    uint32_t Offset;
    uint32_t Address;
}bitField, bf;

static struct mdioControl
{
    bf en;
    bf dis;
    bf sAq;
    bf sRs;
    bf cl45;
    
    bf opR;
    bf opW;
    bf opA;
    bf opE;
    
    bf b;
    bf phy;
    bf mmd;
    
    bf a;
    bf r;
    bf w;
    
    uint32_t timeout;
    uint32_t repeat;
    
    HRESULT state;
} mdioCtrl = { 0 };

static void setBF(bf *nbf, uint32_t value)
{
    nbf->Value = (value & MDIO_PAR_VAL_MASK) >> MDIO_PAR_VAL_OFFSET;
    nbf->Mask = (value & MDIO_PAR_MASK_MASK) >> MDIO_PAR_MASK_OFFSET;
    nbf->Offset = (value & MDIO_PAR_OFF_MASK) >> MDIO_PAR_OFF_OFFSET;
    nbf->Address = (value & MDIO_ADDRESS_MASK) >> MDIO_ADDRESS_OFFSET;
    //    printf("Set addr 0x%4x offset 0x%04x mask %x val %x\n", nbf->Address, nbf->Offset, nbf->Mask, nbf->Value);
}


static int32_t checkMdioCtrl(void *currentHandle)
{
    int32_t res = EXIT_SUCCESS;
    
    if (mdioCtrl.opR.Address == 0 ||
        mdioCtrl.opW.Address == 0 ||
        //mdioCtrl.phy.Address == 0 ||
        mdioCtrl.mmd.Address == 0 ||
        mdioCtrl.r.Address == 0 ||
        mdioCtrl.w.Address == 0)
    {
        res = EXIT_FAILURE;
    }
    return res;
}

uint32_t GetMdioParameter(void *vdev, uint32_t param)
{
    //void *currentHandle = (void *)vdev;
    uint32_t value = (uint32_t)-1;
    switch(param)
    {
        case mdioTimeoutMs:
            value = mdioCtrl.timeout << MDIO_TIMEOUT_OFFSET;
            break;
    }
    return value;
}
 
void SetMdioDeviceParameters(void *vdev, uint32_t param, uint32_t value)
{
    void *currentHandle = (void *)vdev;
    /* declare local variables */
    switch (param)
    {
        case mdioEnableReg:
            setBF(&mdioCtrl.en, value);
            break;
        case mdioDisableReg:
            setBF(&mdioCtrl.dis, value);
            break;
        case mdioAqSemReg:
            setBF(&mdioCtrl.sAq, value);
            break;
        case mdioRelSemReg:
            setBF(&mdioCtrl.sRs, value);
            break;
        case mdioClauseReg:
            setBF(&mdioCtrl.cl45, value);
            break;
        case mdioReadOpReg:
            setBF(&mdioCtrl.opR, value);
            break;
        case mdioWriteOpReg:
            setBF(&mdioCtrl.opW, value);
            break;
        case mdioAddrOpReg:
            setBF(&mdioCtrl.opA, value);
            break;
        case mdioExecOpReg:
            setBF(&mdioCtrl.opE, value);
            break;
        case mdioBusyReg:
            setBF(&mdioCtrl.b, value);
            break;
        case mdioPhyReg:
            setBF(&mdioCtrl.phy, value);
            mdioCtrl.phy.Mask = 0x1f;
            break;
        case mdioMmdReg:
            setBF(&mdioCtrl.mmd, value);
            mdioCtrl.mmd.Mask = 0x1F;
            break;
        case mdioAddrReg:
            setBF(&mdioCtrl.a, value);
            mdioCtrl.a.Mask = 0xFFFF;
            break;
        case mdioReadValueReg:
            setBF(&mdioCtrl.r, value);
            mdioCtrl.r.Mask = 0xFFFF;
            break;
        case mdioWriteValueReg:
            setBF(&mdioCtrl.w, value);
            mdioCtrl.w.Mask = 0xFFFF;
            break;
        case mdioTimeoutMs:
            mdioCtrl.timeout = (value & MDIO_TIMEOUT_MASK) >> MDIO_TIMEOUT_OFFSET;
            mdioCtrl.repeat = (value & MDIO_REPEAT_MASK) >> MDIO_REPEAT_OFFSET;
            break;
        default:
            break;
    }
    mdioCtrl.state = checkMdioCtrl(currentHandle);
}

static int32_t mdioWait(void *currentHandle, bf *wf)
{
    /*"""
     Force any commands that have been buffered up to be written to the
     hardware.
     """*/
    int32_t res = EXIT_SUCCESS;
    uint32_t i, val;
    for (i = 0; i < mdioCtrl.repeat; i++)
    {
        res = ReadMacRegister32(currentHandle, 0, wf->Address, &val);
        if(((val >> wf->Offset) & wf->Mask) == wf->Value)
        {
            break;
        }
        usleep(mdioCtrl.timeout * 1000);
        if (i > mdioCtrl.repeat / 2)
        {
            //            printf("Long wait. reg 0x%04x val 0x%04x\n", wf->Address, val);
        }
    }
    return i < mdioCtrl.repeat ? res : EXIT_FAILURE;
}

static void rmw(void *currentHandle, bf *nbf)
{
    uint32_t nval, val;
    ReadMacRegister32(currentHandle, 0, nbf->Address, &val);
    nval = (val & (~(nbf->Mask << nbf->Offset))) | ((nbf->Value << nbf->Offset));
    WriteMacRegister32(currentHandle, 0, nbf->Address, nval);
}

static int32_t enableMdio(void *currentHandle)
{
    if (mdioCtrl.en.Address != 0)
    {
        rmw(currentHandle, &mdioCtrl.en);
    }
    else
    {
        printf("No MDIO en control");
    }
    return EXIT_SUCCESS;
}

static int32_t disableMdio(void *currentHandle)
{
    if (mdioCtrl.dis.Address != 0)
    {
        rmw(currentHandle, &mdioCtrl.dis);
    }
    else
    {
        printf("No MDIO en control");
    }
    return EXIT_SUCCESS;
}

static int32_t aquireSemaphore(void *currentHandle)
{
    return mdioCtrl.sAq.Address != 0 ? mdioWait(currentHandle, &mdioCtrl.sAq) : EXIT_SUCCESS;
}

static int32_t releaseSemaphore(void *currentHandle)
{
    if (mdioCtrl.sRs.Address != 0)
    {
        rmw(currentHandle, &mdioCtrl.sRs);
    }
    return EXIT_SUCCESS;
}

static int32_t mdioFlush(void *currentHandle)
{
    return mdioWait(currentHandle, &mdioCtrl.b);
}


int32_t MdioReadData(void *ch, uint32_t phyId, uint32_t mmd, uint32_t address, uint32_t *data)
{
    /*""" Read MDIO registers from PHY_ID.
     The address of the MMD within the Aquantia PHY being addressed.
     """*/
    int32_t hr = mdioCtrl.state;
    bf a = mdioCtrl.a, phy = mdioCtrl.phy, m = mdioCtrl.mmd;
    if( hr == EXIT_SUCCESS )
    {
        hr = aquireSemaphore(ch);
    }
    
    if( hr == EXIT_SUCCESS )
    {
        hr = enableMdio(ch);
    }
    
    if( hr == EXIT_SUCCESS )
    {
        phy.Value = phyId;
        m.Value = mmd;
        hr = mdioFlush(ch);
        if( hr == EXIT_SUCCESS )
        {
            rmw(ch, &mdioCtrl.cl45);
            rmw(ch, &mdioCtrl.opA);
            a.Value = mdioCtrl.cl45.Value ? address : 0;
            rmw(ch, &a);
            rmw(ch, &phy);
            rmw(ch, &m);
            rmw(ch, &mdioCtrl.opE);
            hr = mdioFlush(ch);
        }
        
        if( hr == EXIT_SUCCESS )
        {
            rmw(ch, &mdioCtrl.opR);
            rmw(ch, &phy);
            rmw(ch, &m);
            rmw(ch, &mdioCtrl.opE);
            
            hr = mdioFlush(ch);
        }
        
        if( hr == EXIT_SUCCESS )
        {
            hr = ReadMacRegister32(ch, 0, mdioCtrl.r.Address, data);
        }
        
        if( hr == EXIT_SUCCESS )
        {
            hr = disableMdio(ch);
        }
    }
    
    releaseSemaphore(ch);
    return hr;
}

int32_t MdioWriteData(void *ch, uint32_t phyId, uint32_t mmd, uint32_t address, uint32_t value)
{
    /*""" Write value to MDIO registers on PHY_ID.
     The address of the MMD within the Aquantia PHY being addressed.
     """*/
    int32_t hr = mdioCtrl.state;
    bf a = mdioCtrl.a, phy = mdioCtrl.phy, m = mdioCtrl.mmd, w = mdioCtrl.w;
    if( hr == EXIT_SUCCESS )
    {
        hr = aquireSemaphore(ch);
    }
    
    if( hr == EXIT_SUCCESS )
    {
        hr = enableMdio(ch);
    }
    
    if( hr == EXIT_SUCCESS )
    {
        phy.Value = phyId;
        m.Value = mmd;
        w.Value = value;
        hr = mdioFlush(ch);
        if( hr == EXIT_SUCCESS )
        {
            rmw(ch, &mdioCtrl.cl45);
            rmw(ch, &mdioCtrl.opA);
            a.Value = mdioCtrl.cl45.Value ? address : 0;
            rmw(ch, &a);
            rmw(ch, &phy);
            rmw(ch, &m);
            rmw(ch, &mdioCtrl.opE);
            hr = mdioFlush(ch);
        }
        
        if( hr == EXIT_SUCCESS )
        {
            rmw(ch, &mdioCtrl.opW);
            rmw(ch, &phy);
            rmw(ch, &m);
            rmw(ch, &w);
            rmw(ch, &mdioCtrl.opE);
            
            hr = mdioFlush(ch);
        }
        
        if( hr == EXIT_SUCCESS )
        {
            hr = disableMdio(ch);
        }
    }
    
    releaseSemaphore(ch);
    return hr;
}
