/**************************************************************************

Copyright (c) 2020 - 2025, Intel Corporation. All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

    * Redistributions of source code must retain the above copyright notice,
      this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    * Neither the name of Intel Corporation nor the names of its contributors
      may be used to endorse or promote products derived from this software
      without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE
FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

***************************************************************************/
#include <Library/HiiLib.h>
#include "Hii/Hii.h"
#include <Protocol/HiiPopup.h>

#include "Hii/CfgAccessProt/HiiConfigAccessInfo.h"
#include "Hii/HiiSetup.h"
#include "Hii/FormsetStd/HiiCommonDep.h"

#include "Hii/FormsetStd/PortOptions.h"


/* This is the generated IFR binary data for each formset defined in VFR.
  This data array is ready to be used as input of PreparePackageList() to
  create a packagelist (which contains Form packages, String packages, etc). */
extern UINT8                           InventoryBin[];

/* This is the generated String package data for all .UNI files.
  This data array is ready to be used as input of PreparePackageList() to
  create a packagelist (which contains Form packages, String packages, etc). */
extern UINT8                           *gUndiDxeHiiStringsPkgPtr;
extern EFI_HII_CONFIG_ACCESS_PROTOCOL  gHiiConfigAccessProt;

extern HII_VARSTORE_MAP_CFG            mHiiStandardMapCfg;

extern HII_STATIC_INV_STRING_ENTRY     mHiiHwStaticInvStringMap[];

extern UINTN                           mHiiHwStaticInvStringMapSize;

/** Gets the next language code from native RFC 4646 language code array.

  @param[in]      Langs      Pointer to the string containing language array
  @param[in,out]  LangsIdx   Index in the string from which to start search, on
                             output set to position after currently found language
  @param[in]      LangsSize  Size of Langs string in bytes
  @param[out]     SubLang    Pointer to string filled with currently found language

  @retval    TRUE            Operation successful
  @retval    FALSE           No language left in array
  @retval    FALSE           Invalid input parameter, LangsIdx out of range
**/
BOOLEAN
HiiGetNextLanguage (
  IN      CHAR8   *Langs,
  IN OUT  UINTN   *LangsIdx,
  IN      UINTN   LangsSize,
  OUT     CHAR8   *SubLang
  )
{
  UINTN  SubLangIdx = 0;

  IF_NULL3_RETURN (Langs, LangsIdx, SubLang, FALSE);
  IF_RETURN ((*LangsIdx + 1) > LangsSize, FALSE);

  while ((*LangsIdx + 1) < LangsSize) {
    if (Langs[*LangsIdx] == ';') {
      (*LangsIdx)++;
      break;
    }
    if (Langs[*LangsIdx] == '\0') {
      break;
    }
    SubLang[SubLangIdx++] = Langs[ (*LangsIdx)++];
  }

  SubLang[SubLangIdx] = '\0';

  return (SubLangIdx != 0); // any languages left ?
}

/** Processes static inventory strings map for specific language OR language agnostic entries.

  @param[in]   UndiPrivateData  Pointer to driver private data structure
  @param[in]   Language         Language for which map language specific entries should be processed,
                                or NULL when language agnostic entries should be processed

  @retval    EFI_SUCCESS            Operation successful
  @retval    EFI_INVALID_PARAMETER  Invalid input parameter
  @retval    EFI_INVALID_PARAMETER  Map entry has no getter
  @retval    EFI_UNSUPPORTED        Unsupported type of getter
  @retval    EFI_DEVICE_ERROR       Failed to set inventory string in package
  @retval    !EFI_SUCCESS           Failure of inventory string getter / support checker
**/
EFI_STATUS
HiiSetupStaticInventory (
  IN        UNDI_PRIVATE_DATA  *UndiPrivateData,
  IN CONST  CHAR8              *Language
  )
{
  EFI_STATUS                   Status;
  HII_STATIC_INV_STRING_ENTRY  *InvStringEntry;
  UINTN                        EntryIdx;
  BOOLEAN                      EntrySupported;
  CHAR16                       InvString[HII_MAX_STR_LEN];
  EFI_STRING_ID                InvStringId;

  DEBUGPRINT (HII, ("--> lang - %a\n", Language));

  ASSERT (UndiPrivateData != NULL);

  for (EntryIdx = 0; EntryIdx < mHiiHwStaticInvStringMapSize; EntryIdx++) {
    InvStringEntry = &mHiiHwStaticInvStringMap[EntryIdx];

    if (((Language == NULL) && (InvStringEntry->GetStringType == LANG)) ||
        ((Language == NULL) && (InvStringEntry->GetStringType == LANG_FOR_STRING_ID)) ||
        ((Language != NULL) && (InvStringEntry->GetStringType == ALL_LANG)) ||
        ((Language != NULL) && (InvStringEntry->GetStringType == ALL_LANG_FOR_STRING_ID)))
    {
      continue; // for language agnostic (NULL) omit language specific entry & vice versa
    }
    DEBUGPRINT (HII, ("Processing entry %d for lang %a\n", EntryIdx, Language));

    if (Language != NULL) {
      if (IS_UEFI_CONFIG_LANG (Language) &&
          !InvStringEntry->HasXUefi)
      {
        continue; // Don't modify x-UEFI language strings
      }
    }

    if (InvStringEntry->CheckSupport != NULL) { // evaluate support if needed
      Status = InvStringEntry->CheckSupport (UndiPrivateData, &EntrySupported);
      IF_RETURN (EFI_ERROR (Status), Status);
      if (!EntrySupported) {
        DEBUGPRINT (HII, (" - unsupported\n"));
        continue;
      }
    }

    if (InvStringEntry->GetString.Raw == NULL) {
      return EFI_INVALID_PARAMETER;
    }

    ZeroMem (InvString, sizeof (InvString));

    switch (InvStringEntry->GetStringType) {
    case ALL_LANG:
      Status = InvStringEntry->GetString.GetAllLang (UndiPrivateData, InvString);
      break;
    case LANG:
      Status = InvStringEntry->GetString.GetLang (UndiPrivateData, InvString, Language);
      break;
    case ALL_LANG_FOR_STRING_ID:
      Status = InvStringEntry->GetString.GetAllLangForStringId (UndiPrivateData, InvString, InvStringEntry->StringId);
      break;
    case LANG_FOR_STRING_ID:
      Status = InvStringEntry->GetString.GetLangForStringId (
                                           UndiPrivateData,
                                           InvString,
                                           Language,
                                           InvStringEntry->StringId
                                           );
      break;
    default:
      Status = EFI_UNSUPPORTED;
      break;
    }

    if (EFI_ERROR (Status)) {
      ZeroMem (InvString, sizeof (InvString));
      Status = GetInventoryStr (UndiPrivateData, InvString, STRING_TOKEN (STR_NA_TEXT), Language);
      IF_RETURN (EFI_ERROR (Status), Status);
    }

    DEBUGPRINT (HII, ("Setting Inv. string to:\n%s\n", InvString));

    InvStringId = HiiSetString (
                    UndiPrivateData->HiiInfo.HiiPkgListHandle,
                    InvStringEntry->StringId,
                    InvString,
                    Language
                    );
    IF_RETURN (InvStringId == 0, EFI_DEVICE_ERROR);
  }

  DEBUGPRINT (HII, ("Static Inv. processing success for lang %a\n", Language));
  return EFI_SUCCESS;
}

#define HII_VARSTORES_MAX 2

/** Helper function that adds varstore (config data structure) map config.

  @param[in,out]  HiiCfgAccessInfo  HII Config Access protocol info structure
  @param[in]      VarStoreMapCfg    Varstore map config structure
  @param[in]      DriverContext     Context of the driver that will be passed to varstore
                                    map getter/setters
  @param[in]      Default           Boolean value that tells if this varstore map config is default

  @retval    EFI_SUCCESS            Operation successful
  @retval    EFI_INVALID_PARAMETER  Invalid input parameter
  @retval    EFI_OUT_OF_RESOURCES   Maximum number of varstores reached
**/
EFI_STATUS
AddVarStoreConfig (
  IN OUT  HII_CFG_ACCESS_INFO   *HiiCfgAccessInfo,
  IN      HII_VARSTORE_MAP_CFG  *VarStoreMapCfg,
  IN      VOID                  *DriverContext,
  IN      BOOLEAN               Default
  )
{
  HII_VARSTORE_MAP_CFG  *VarStoreConfigDest;

  IF_NULL3_RETURN (HiiCfgAccessInfo, HiiCfgAccessInfo->VarStoreConfigs, DriverContext, EFI_INVALID_PARAMETER);
  IF_RETURN (HiiCfgAccessInfo->NumberOfVarStoreConfigs == HII_VARSTORES_MAX, EFI_OUT_OF_RESOURCES);

  VarStoreConfigDest = HiiCfgAccessInfo->VarStoreConfigs + HiiCfgAccessInfo->NumberOfVarStoreConfigs;

  CopyMem (
    VarStoreConfigDest,
    VarStoreMapCfg,
    sizeof (HII_VARSTORE_MAP_CFG)
    );
  VarStoreConfigDest->DriverContext = DriverContext;

  if (Default) {
    HiiCfgAccessInfo->DefaultVarStoreCfgIdx = HiiCfgAccessInfo->NumberOfVarStoreConfigs;
  }

  HiiCfgAccessInfo->NumberOfVarStoreConfigs++;

  return EFI_SUCCESS;
}

/**
  Helper function that creates a copy of the source string package provided and strips the requested language.

  Note: This function allocates memory for StringsPkgStripped.
  It is the responsibility of the caller to free the memory.

  @param[in]      Language               Language to be stripped from string package.
  @param[in]      SourcePkg              Pointer to a source string package.
  @param[out]     StringsPkgStripped     Pointer to an output buffer containing stripped string package
                                         (allocated internally).

  @retval         EFI_SUCCESS            Operation successful.
  @retval         EFI_OUT_OF_RESOURCES   Couldn't allocate buffer for stripped inventory string package.
**/
EFI_STATUS
StripStringsFromPkg (
  IN   CHAR   *Language,
  IN   UINT8  *SourcePkg,
  OUT  UINT8  **StringsPkgStripped
  )
{
  EFI_HII_STRING_PACKAGE_HDR  *CurrentPkg;
  UINT32                      CurrentPkgSize;
  UINT32                      CurrentPkgOffset;
  UINT8                       *NewStringPkg;
  UINT32                      NewPkgSize;
  UINT32                      StringsPkgSize;

  // The first 32 bits contain the binary length of the entire structure.
  StringsPkgSize = *((UINT32 *)SourcePkg);

  NewStringPkg = NULL;
  NewStringPkg = AllocateZeroPool (StringsPkgSize);
  IF_NULL_RETURN (NewStringPkg, EFI_OUT_OF_RESOURCES);

  // As the first 32 bits contain the binary length of the entire structure,
  // omit those and start the iteration after the first 32 bits.
  CurrentPkgOffset = sizeof (UINT32);
  NewPkgSize       = sizeof (UINT32);

  CurrentPkg = NULL;

  // Copy the content of the original structure - omitting the language that is to be stripped.
  while (CurrentPkgOffset < StringsPkgSize) {
    CurrentPkg     = (EFI_HII_STRING_PACKAGE_HDR *)(SourcePkg + CurrentPkgOffset);
    CurrentPkgSize = CurrentPkg->Header.Length;

    if (AsciiStrnCmp (CurrentPkg->Language, Language, AsciiStrnLenS (Language, HII_MAX_STR_LEN)) != 0) {
      CopyMem (NewStringPkg + NewPkgSize, SourcePkg + CurrentPkgOffset, CurrentPkgSize);
      NewPkgSize += CurrentPkgSize;
    }

    CurrentPkgOffset += CurrentPkgSize;
  }

  // Fill the first 32 bits with the new size of the structure.
  CopyMem (NewStringPkg, &NewPkgSize, sizeof (UINT32));

  NewStringPkg = ReallocatePool (StringsPkgSize, NewPkgSize, NewStringPkg);
  IF_NULL_RETURN (NewStringPkg, EFI_OUT_OF_RESOURCES);

  *StringsPkgStripped = NewStringPkg;

  return EFI_SUCCESS;
}

/**
  Helper function that changes language name in string package.

  The languages given MUST have the same length in order for the function to
  complete successfully.

  @param[in]      OldLanguage            Language to be renamed in string package.
  @param[in]      NewLanguage            New language to be assigned to a chosen package.
  @param[in]      StringsPkg             Pointer to a buffer containing string package.

  @retval         EFI_SUCCESS            Operation successful.
  @retval         EFI_INVALID_PARAMETER  Old language length and new language length mismatch.
  @retval         EFI_NOT_FOUND          Old language package not found in strings package.
**/
EFI_STATUS
RenameLanguageInStringsPkg (
  IN  CHAR   *OldLanguage,
  IN  CHAR   *NewLanguage,
  IN  UINT8  *StringsPkg
  )
{
  EFI_HII_STRING_PACKAGE_HDR  *CurrentPkg;
  UINT32                      CurrentPkgOffset;
  UINT32                      CurrentPkgSize;
  UINT32                      StringsPkgSize;
  UINTN                       OldLanguageLength;
  UINTN                       NewLanguageLength;

  OldLanguageLength = AsciiStrnLenS (OldLanguage, HII_MAX_STR_LEN);
  NewLanguageLength = AsciiStrnLenS (NewLanguage, HII_MAX_STR_LEN);
  IF_RETURN (OldLanguageLength != NewLanguageLength, EFI_INVALID_PARAMETER);

  // The first 32 bits contain the binary length of the entire structure.
  StringsPkgSize = *((UINT32 *)StringsPkg);

  // As the first 32 bits contain the binary length of the entire structure,
  // omit those and start the iteration after the first 32 bits.
  CurrentPkgOffset = sizeof (UINT32);

  CurrentPkg = NULL;

  while (CurrentPkgOffset < StringsPkgSize) {
    CurrentPkg     = (EFI_HII_STRING_PACKAGE_HDR *)(StringsPkg + CurrentPkgOffset);
    CurrentPkgSize = CurrentPkg->Header.Length;

    if (AsciiStrnCmp (CurrentPkg->Language, OldLanguage, OldLanguageLength) == 0) {
      CopyMem (CurrentPkg->Language, NewLanguage, NewLanguageLength);
      return EFI_SUCCESS;
    }

    CurrentPkgOffset += CurrentPkgSize;
  }

  return EFI_NOT_FOUND;
}



/** Main function that sets up inventory packages. Adds configuration for supported varstores &
  adds required packages, runs static inventory map processing.

  @param[in,out]   UndiPrivateData  Pointer to driver private data structure

  @retval    EFI_SUCCESS            Operation successful
  @retval    EFI_INVALID_PARAMETER  Invalid input parameter
  @retval    EFI_OUT_OF_RESOURCES   Failed to add packages
  @retval    EFI_OUT_OF_RESOURCES   Memory allocation failed
  @retval    EFI_OUT_OF_RESOURCES   Failed to get supported languages
  @retval    !EFI_SUCCESS           Failure of one of the other underlying functions
**/
EFI_STATUS
HiiSetupInventoryPackages (
  IN OUT  UNDI_PRIVATE_DATA  *UndiPrivateData
  )
{
  EFI_STATUS                   Status;
  CHAR8                        SubLang[HII_MAX_STR_LEN];
  UINTN                        LanguagesSize;
  UINTN                        LanguagesIdx               = 0;
  BOOLEAN                      InvSetupSuccess            = FALSE;
  CHAR8                        *Languages                 = NULL;
  EFI_GUID                     HiiFormGuid                = HII_FORM_GUID;
  HII_INFO                     *HiiInfo;
  UINT8                        *StringsPkg                = NULL;
  UINT8                        *StringsPkgTmp;
  BOOLEAN                      OemSupportsDellXUefi;

  StringsPkgTmp        = NULL;
  OemSupportsDellXUefi = FALSE;

  ASSERT (UndiPrivateData != NULL);

  HiiInfo = &UndiPrivateData->HiiInfo;

  Status = SetHwDependentAdapterSupportFlags (UndiPrivateData);
  IF_RETURN (EFI_ERROR (Status), Status);

  HiiInfo->HiiCfgAccessInfo.VarStoreConfigs = AllocateZeroPool (HII_VARSTORES_MAX * sizeof (HII_VARSTORE_MAP_CFG));
  IF_NULL_RETURN (HiiInfo->HiiCfgAccessInfo.VarStoreConfigs, EFI_OUT_OF_RESOURCES);

  HiiInfo->HiiCfgAccessInfo.OnActionChanging      = HiiOnActionChanging;
  HiiInfo->HiiCfgAccessInfo.OnActionChanged       = HiiOnActionChanged;
  HiiInfo->HiiCfgAccessInfo.OnActionFormOpen      = HiiOnActionFormOpen;
  HiiInfo->HiiCfgAccessInfo.CallBackDriverContext = UndiPrivateData;

  Status = AddVarStoreConfig (&HiiInfo->HiiCfgAccessInfo, &mHiiStandardMapCfg, UndiPrivateData, TRUE);
  IF_GOTO (EFI_ERROR (Status), ExitFreeRes);

  StringsPkg = gUndiDxeHiiStringsPkgPtr;
    HiiInfo->HiiPkgListHandle = HiiAddPackages (
                                  &HiiFormGuid,
                                  HiiInfo->HiiInstallHandle,
                                  StringsPkg,
                                  InventoryBin,
                                  NULL
                                  );

  IF_GOTO (HiiInfo->HiiPkgListHandle == NULL, ExitFreeRes);

  Languages = HiiGetSupportedLanguages (HiiInfo->HiiPkgListHandle);
  IF_GOTO (Languages == NULL, ExitFreeRes);

  DEBUGPRINT (HII, ("Package list languages - %a\n"));

  // Setup language agnostic inventory
  Status = HiiSetupStaticInventory (UndiPrivateData, NULL);
  IF_GOTO (EFI_ERROR (Status), ExitFreeRes);

  // assume size of returned languages will never be longer than HII_MAX_STR_LEN
  LanguagesSize = AsciiStrnLenS (Languages, HII_MAX_STR_LEN) + 1;

  // Setup language specific inventory
  while (HiiGetNextLanguage (Languages, &LanguagesIdx, LanguagesSize, SubLang)) {
    Status = HiiSetupStaticInventory (UndiPrivateData, SubLang);
    IF_GOTO (EFI_ERROR (Status), ExitFreeRes);
  }

  InvSetupSuccess = TRUE;
ExitFreeRes:
  if (Languages != NULL) {
    FreePool (Languages);
  }

  if (!InvSetupSuccess) {
    HiiUnload (UndiPrivateData);
    if (!EFI_ERROR (Status)) {
      Status = EFI_OUT_OF_RESOURCES;
    }
  }


  return Status;
}

/** Locates required HII protocols (Database, String, FormBrowser2 and ConfigRouting).

   @param[in,out]   HiiInfo      Hii info structure

   @retval   EFI_SUCCESS     All protocols located successfully
   @retval   EFI_NOT_FOUND   No instances of one of above protocols were found
**/
EFI_STATUS
HiiLocateProtocols (
  IN OUT  HII_INFO  *HiiInfo
  )
{
  EFI_STATUS                      Status;
  EFI_HII_DATABASE_PROTOCOL       *HiiDatabaseProt;
  EFI_HII_STRING_PROTOCOL         *HiiStringProt;
  EFI_FORM_BROWSER2_PROTOCOL      *FormBrowser2Prot;

  ASSERT (HiiInfo != NULL);

  Status = gBS->LocateProtocol (
                  &gEfiHiiDatabaseProtocolGuid,
                  NULL,
                  (VOID **) &HiiDatabaseProt
                  );
  IF_RETURN (EFI_ERROR (Status), Status);

  Status = gBS->LocateProtocol (
                  &gEfiHiiStringProtocolGuid,
                  NULL,
                  (VOID **) &HiiStringProt
                  );
  IF_RETURN (EFI_ERROR (Status), Status);

  Status = gBS->LocateProtocol (
                  &gEfiFormBrowser2ProtocolGuid,
                  NULL,
                  (VOID **) &FormBrowser2Prot
                  );
  IF_RETURN (EFI_ERROR (Status), Status);

  Status = gBS->LocateProtocol (
                  &gEfiHiiConfigRoutingProtocolGuid,
                  NULL,
                  (VOID **) &HiiInfo->HiiCfgAccessInfo.HiiConfigRoutingProt
                  );
  IF_RETURN (EFI_ERROR (Status), Status);
  Status = gBS->LocateProtocol (
                  &gEfiHiiPopupProtocolGuid,
                  NULL,
                  (VOID **)&HiiInfo->HiiCfgAccessInfo.HiiPopUpProt
                  );
  if (Status == EFI_NOT_FOUND) {
    DEBUGPRINT (HII, ("Hii Popup Protocol not available!\n"));
  } else {
    IF_RETURN (EFI_ERROR (Status), Status);
  }

  return EFI_SUCCESS;
}

/** Uninstalls HII protocol & package related resources, frees memory allocations.
   (Resources previously obtained by HiiInit ()).

   @param[in,out]   UndiPrivateData   Points to the driver instance private data.

   @retval   EFI_SUCCESS    HII resources uninstalled correctly
   @retval   !EFI_SUCCESS   Failed to uninstall HII resources
**/
EFI_STATUS
HiiUnload (
  IN OUT  UNDI_PRIVATE_DATA  *UndiPrivateData
  )
{
  EFI_STATUS           Status;
  HII_INFO             *HiiInfo;

  DEBUGPRINT (HII, ("--> ()\n"));
  ASSERT (UndiPrivateData != NULL);

  HiiInfo = &UndiPrivateData->HiiInfo;

  if (HiiInfo->HiiPkgListHandle != NULL) {
    HiiRemovePackages (HiiInfo->HiiPkgListHandle);
    HiiInfo->HiiPkgListHandle = NULL;
  }


  if (HiiInfo->HiiInstallHandle != NULL) {
    if (HiiInfo->HiiCfgAccessInfo.VarStoreConfigs != NULL) {
      FreePool (HiiInfo->HiiCfgAccessInfo.VarStoreConfigs);
      HiiInfo->HiiCfgAccessInfo.VarStoreConfigs = NULL;
    }
    // Free port options data buffer
    if (UndiPrivateData->NicInfo.PortOptionsDataBuffer != NULL) {
        FreePool (UndiPrivateData->NicInfo.PortOptionsDataBuffer);
        UndiPrivateData->NicInfo.PortOptionsDataBuffer = NULL;
    }

    Status = gBS->UninstallProtocolInterface (
                    HiiInfo->HiiInstallHandle,
                    &gEfiHiiConfigAccessProtocolGuid,
                    &HiiInfo->HiiCfgAccessInfo.HiiConfigAccessProt
                    );
    IF_RETURN (EFI_ERROR (Status), Status);

    HiiInfo->HiiInstallHandle = NULL;
  }

  ZeroMem (HiiInfo, sizeof (HII_INFO));

  return EFI_SUCCESS;
}

/** Installs HII Config Access interfaces required by driver instance -
   protocol & package related resources.

   @param[in,out]   UndiPrivateData   Points to the driver instance private data.

   @retval   EFI_SUCCESS          Successful operation
   @retval   EFI_ALREADY_STARTED  HII interfaces are already installed or weren't properly uninstalled
   @retval   !EFI_SUCCESS         Failed to setup Config Access protocol/package resources or
                                  failed to locate required protocols
**/
EFI_STATUS
HiiInit (
  IN OUT  UNDI_PRIVATE_DATA  *UndiPrivateData
  )
{
  EFI_STATUS               Status;
  HII_INFO                 *HiiInfo;
  HII_CFG_ACCESS_INFO      *HiiCfgAccessInfo;

  DEBUGPRINT (HII, ("--> ()\n"));
  ASSERT (UndiPrivateData != NULL);

  HiiInfo          = &UndiPrivateData->HiiInfo;
  HiiCfgAccessInfo = &HiiInfo->HiiCfgAccessInfo;

  IF_RETURN (HiiInfo->HiiInstallHandle != NULL, EFI_ALREADY_STARTED);
  IF_RETURN (HiiInfo->HiiPkgListHandle != NULL, EFI_ALREADY_STARTED);

  ZeroMem (HiiInfo, sizeof (HII_INFO));

  // 1. Locate required HII protocols in the platform BIOS
  Status = HiiLocateProtocols (HiiInfo);
  IF_RETURN (EFI_ERROR (Status), Status);

  // 2. Install HII Config Access protocol instance

  HiiInfo->HiiInstallHandle = UndiPrivateData->DeviceHandle;

  HiiCfgAccessInfo->Signature          = HII_CFG_ACCESS_INFO_SIG;
  HiiCfgAccessInfo->InstallationHandle = HiiInfo->HiiInstallHandle;

  CopyMem (
    &HiiCfgAccessInfo->HiiConfigAccessProt,
    &gHiiConfigAccessProt,
    sizeof (EFI_HII_CONFIG_ACCESS_PROTOCOL)
    );

  Status = gBS->InstallProtocolInterface (
                  &HiiInfo->HiiInstallHandle,
                  &gEfiHiiConfigAccessProtocolGuid,
                  EFI_NATIVE_INTERFACE,
                  &HiiCfgAccessInfo->HiiConfigAccessProt
                  );
  IF_GOTO (EFI_ERROR (Status), ConfigAccessInstallError);

  // 3. Setup & add packages related to HII Config Access protocol instance
  // Read port options and allocate memory for data buffer
  Status = GetPortOptionsForAllPorts (&UndiPrivateData->NicInfo);
  IF_RETURN (EFI_ERROR (Status), Status);
  Status = HiiSetupInventoryPackages (UndiPrivateData);
  IF_RETURN (EFI_ERROR (Status), Status);

  Status = CreateDynamicVfrContent (UndiPrivateData);
  IF_GOTO (EFI_ERROR (Status), CreateDynamicVfrError);

  DEBUGPRINT (HII, ("HiiInit() - EFI_SUCCESS\n"));
  return EFI_SUCCESS;

ConfigAccessInstallError:
  HiiInfo->HiiInstallHandle = NULL;
CreateDynamicVfrError:
  HiiUnload (UndiPrivateData);
  return Status;
}

/** Uninstalls HII packages installed in partial init flow.
   Reverts HiiAddStringPkgOnly () operations.

   @param[in,out]   UndiPrivateData   Points to the driver instance private data

   @retval   EFI_SUCCESS          Always returned
**/
EFI_STATUS
HiiRemoveStringPkgOnly (
  IN OUT  UNDI_PRIVATE_DATA  *UndiPrivateData
  )
{
  HII_INFO  *HiiInfo;

  ASSERT (UndiPrivateData != NULL);

  HiiInfo = &UndiPrivateData->HiiInfo;

  if (HiiInfo->HiiPkgListHandle != NULL) {
    HiiRemovePackages (HiiInfo->HiiPkgListHandle);
    HiiInfo->HiiPkgListHandle = NULL;
  }

  return EFI_SUCCESS;
}

/** Used instead of HiiInit() in partial init flow. Installs only string packages
   to allow Driver Health protocol reporting.

   @param[in,out]   UndiPrivateData   Points to the driver instance private data

   @retval   EFI_SUCCESS           Successful operation
   @retval   EFI_ALREADY_STARTED   HII string packages are already installed or weren't properly uninstalled
   @retval   EFI_OUT_OF_RESOURCES  Failed to register string packages
**/
EFI_STATUS
HiiAddStringPkgOnly (
  IN OUT  UNDI_PRIVATE_DATA  *UndiPrivateData
  )
{
  HII_INFO  *HiiInfo;
  EFI_GUID  HiiFormGuid = HII_FORM_GUID;

  ASSERT (UndiPrivateData != NULL);

  HiiInfo = &UndiPrivateData->HiiInfo;

  IF_RETURN (HiiInfo->HiiPkgListHandle != NULL, EFI_ALREADY_STARTED);

  HiiInfo->HiiInstallHandle = UndiPrivateData->ControllerHandle;
  HiiInfo->HiiPkgListHandle = HiiAddPackages (
                                &HiiFormGuid,
                                HiiInfo->HiiInstallHandle,
                                gUndiDxeHiiStringsPkgPtr,
                                NULL
                                );
  IF_NULL_RETURN (HiiInfo->HiiPkgListHandle, EFI_OUT_OF_RESOURCES);

  return EFI_SUCCESS;
}
