/* SPDX-License-Identifier: GPL-2.0-only */
/*
 * aQuantia Corporation Network Driver
 * Copyright (C) 2019-2020 aQuantia Corporation. All rights reserved
 */

/* File hw_atl2_internal.h: Definition of Atlantic2 chip specific
 * constants.
 */

#ifndef HW_ATL2_INTERNAL_H
#define HW_ATL2_INTERNAL_H

#include "../aq_common.h"

#define HW_ATL2_MTU_JUMBO  16352U
#define HW_ATL2_MTU        1514U

#define HW_ATL2_TX_RINGS 4U
#define HW_ATL2_RX_RINGS 4U

#define HW_ATL2_RINGS_MAX 32U
#define HW_ATL2_TXD_SIZE       (16U)
#define HW_ATL2_RXD_SIZE       (16U)

#define HW_ATL2_MAC_UC   0U
#define HW_ATL2_MAC_MIN  1U
#define HW_ATL2_MAC_MAX  33U

/* UCAST/MCAST filters */
#define HW_ATL2_UCAST_FILTERS_MAX 38

/* interrupts */
#define HW_ATL2_ERR_INT 8U
#define HW_ATL2_INT_MASK  (0xFFFFFFFFU)

#define HW_ATL2_TXD_CTL_DESC_TYPE_TXTS  (0x00000003)

#define HW_ATL2_TXD_CTL_TS_EN       (0x40000000)
#define HW_ATL2_TXD_CTL_TS_PTP      (0x80000000)

#define HW_ATL2_TXBUF_MAX              128U
#define HW_ATL2_PTP_TXBUF_SIZE           8U

#define HW_ATL2_RXBUF_MAX              192U
#define HW_ATL2_PTP_RXBUF_SIZE          16U
#define HW_ATL2_PTP_HWTS_RXBUF_SIZE      8U

#define HW_ATL2_RSS_REDIRECTION_MAX 64U

#define HW_ATL2_TCRSS_4_8  1
#define HW_ATL2_TC_MAX 1U
#define HW_ATL2_RSS_MAX 8U

#define HW_ATL2_LRO_RXD_MAX 16U
#define HW_ATL2_RS_SLIP_ENABLED  0U

/* (256k -1(max pay_len) - 54(header)) */
#define HAL_ATL2_LSO_MAX_SEGMENT_SIZE 262089U

/* (256k -1(max pay_len) - 74(header)) */
#define HAL_ATL2_LSO_IPV6_MAX_SEGMENT_SIZE 262069U

#define HW_ATL2_RXD_WB_PKTTYPE_VLAN		(1 << 5)
#define HW_ATL2_RXD_WB_PKTTYPE_VLAN_DOUBLE	(1 << 6)

#define HW_ATL2_INTR_MODER_MAX  0x1FF
#define HW_ATL2_INTR_MODER_MIN  0xFF

#define HW_ATL2_MIN_RXD \
	(ALIGN(AQ_CFG_SKB_FRAGS_MAX + 1U, AQ_HW_RXD_MULTIPLE))
#define HW_ATL2_MIN_TXD \
	(ALIGN(AQ_CFG_SKB_FRAGS_MAX + 1U, AQ_HW_TXD_MULTIPLE))

#define HW_ATL2_MAX_RXD 8184U
#define HW_ATL2_MAX_TXD 8184U

#define HW_ATL2_FW_SM_ACT_RSLVR  0x3U

#define HW_ATL2_RPF_TAG_UC_OFFSET      0x0
#define HW_ATL2_RPF_TAG_ALLMC_OFFSET   0x6
#define HW_ATL2_RPF_TAG_ET_OFFSET      0x7
#define HW_ATL2_RPF_TAG_VLAN_OFFSET    0xA
#define HW_ATL2_RPF_TAG_UNTAG_OFFSET   0xE
#define HW_ATL2_RPF_TAG_L3_V4_OFFSET   0xF
#define HW_ATL2_RPF_TAG_L3_V6_OFFSET   0x12
#define HW_ATL2_RPF_TAG_L4_OFFSET      0x15
#define HW_ATL2_RPF_TAG_L4_FLEX_OFFSET 0x18
#define HW_ATL2_RPF_TAG_FLEX_OFFSET    0x1B
#define HW_ATL2_RPF_TAG_PCP_OFFSET     0x1D

#define HW_ATL2_RPF_TAG_UC_MASK    (0x0000003F << HW_ATL2_RPF_TAG_UC_OFFSET)
#define HW_ATL2_RPF_TAG_ALLMC_MASK (0x00000001 << HW_ATL2_RPF_TAG_ALLMC_OFFSET)
#define HW_ATL2_RPF_TAG_UNTAG_MASK (0x00000001 << HW_ATL2_RPF_TAG_UNTAG_OFFSET)
#define HW_ATL2_RPF_TAG_VLAN_MASK  (0x0000000F << HW_ATL2_RPF_TAG_VLAN_OFFSET)
#define HW_ATL2_RPF_TAG_ET_MASK    (0x00000007 << HW_ATL2_RPF_TAG_ET_OFFSET)
#define HW_ATL2_RPF_TAG_L3_V4_MASK (0x00000007 << HW_ATL2_RPF_TAG_L3_V4_OFFSET)
#define HW_ATL2_RPF_TAG_L3_V6_MASK (0x00000007 << HW_ATL2_RPF_TAG_L3_V6_OFFSET)
#define HW_ATL2_RPF_TAG_L4_MASK    (0x00000007 << HW_ATL2_RPF_TAG_L4_OFFSET)
#define HW_ATL2_RPF_TAG_PCP_MASK   (0x00000007 << HW_ATL2_RPF_TAG_PCP_OFFSET)


#define HW_ATL2_RPF_TAG_BASE_UC    (1 << HW_ATL2_RPF_TAG_UC_OFFSET)
#define HW_ATL2_RPF_TAG_BASE_ALLMC (1 << HW_ATL2_RPF_TAG_ALLMC_OFFSET)
#define HW_ATL2_RPF_TAG_BASE_UNTAG (1 << HW_ATL2_RPF_TAG_UNTAG_OFFSET)
#define HW_ATL2_RPF_TAG_BASE_VLAN  (1 << HW_ATL2_RPF_TAG_VLAN_OFFSET)

#define HW_ATL2_RPF_L2_PROMISC_OFF_INDEX   0
#define HW_ATL2_RPF_VLAN_PROMISC_OFF_INDEX 1
#define HW_ATL2_RPF_L3L4_USER_INDEX        48
#define HW_ATL2_RPF_ET_PCP_USER_INDEX      64
#define HW_ATL2_RPF_VLAN_USER_INDEX        80
#define HW_ATL2_RPF_PCP_TO_TC_INDEX        120

#define HW_ATL2_RPF_L3_CMD_EN       BIT(0)
#define HW_ATL2_RPF_L3_CMD_SA_EN    BIT(1)
#define HW_ATL2_RPF_L3_CMD_DA_EN    BIT(2)
#define HW_ATL2_RPF_L3_CMD_PROTO_EN BIT(3)

#define HW_ATL2_RPF_L3_V6_CMD_EN       BIT(0x10)
#define HW_ATL2_RPF_L3_V6_CMD_SA_EN    BIT(0x11)
#define HW_ATL2_RPF_L3_V6_CMD_DA_EN    BIT(0x12)
#define HW_ATL2_RPF_L3_V6_CMD_PROTO_EN BIT(0x13)

#define HW_ATL2_RPF_L4_CMD_EN       BIT(0)
#define HW_ATL2_RPF_L4_CMD_DP_EN    BIT(1)
#define HW_ATL2_RPF_L4_CMD_SP_EN    BIT(2)


#define HW_ATL2_ACTION(ACTION, RSS, INDEX, VALID, TS_VALID) \
	(((ACTION & 0x3U) << 8) | \
	((RSS & 0x1U) << 7) | \
	((INDEX & 0x3FU) << 2) | \
	((TS_VALID & 0x1U) << 1)) | \
	((VALID & 0x1U) << 0)

#define HW_ATL2_ACTION_DROP HW_ATL2_ACTION(0, 0, 0, 1, 0)
#define HW_ATL2_ACTION_DISABLE HW_ATL2_ACTION(0, 0, 0, 0, 0)
#define HW_ATL2_ACTION_ASSIGN_QUEUE(QUEUE) HW_ATL2_ACTION(1, 0, (QUEUE), 1, 0)
#define HW_ATL2_ACTION_ASSIGN_TC(TC) HW_ATL2_ACTION(1, 1, (TC), 1, 0)

#define HW_ATL2_RPF_L3L4_FILTERS_SIZE 8

struct hw_atl2_l3_filter {
	u8 proto;
	u8 usage;
	u32 cmd;
	u32 srcip[4];
	u32 dstip[4];
};

struct hw_atl2_l4_filter {
	u8 usage;
	u32 cmd;
	u16 sport;
	u16 dport;
};

struct hw_atl2_l3l4_filter {
	s8 l3_index;
	s8 l4_index;
};

struct hw_atl2_active_filters_l3 {
	u8 active_ipv4;
	u8 active_ipv6;
};

struct hw_atl2_priv {
	struct hw_atl2_l3_filter l3_filters[HW_ATL2_RPF_L3L4_FILTERS_SIZE];
	struct hw_atl2_l4_filter l4_filters[HW_ATL2_RPF_L3L4_FILTERS_SIZE];
	struct hw_atl2_l3l4_filter l3l4_filters[HW_ATL2_RPF_L3L4_FILTERS_SIZE];
	struct hw_atl2_active_filters_l3 l3_active_filters;
	struct statistics_s last_stats;
	u32 cable_diag_tid1;
};

#endif /* HW_ATL2_INTERNAL_H */
