#include "aqdiag.h"
#include <linux/mm.h>

#define to_bar(obj) container_of((obj), struct aqdiag_bar, kobj)
struct bar_sysfs_entry {
	struct attribute attr;
	ssize_t (*show)(struct aqdiag_bar *, char *);
	ssize_t (*store)(struct aqdiag_bar *, char*, size_t);
};

#define aqdev_bar_attr(_field, _fmt)					\
	static ssize_t							\
	bar_##_field##_show(struct aqdiag_bar *bar, char *buf)		\
	{								\
		return sprintf(buf, _fmt, bar->_field);			\
	}								\
	static struct bar_sysfs_entry bar_##_field##_attr =		\
		__ATTR(_field, S_IRUGO, bar_##_field##_show, NULL);

#pragma GCC diagnostic ignored "-Wformat"
aqdev_bar_attr(addr, "0x%lx\n");
aqdev_bar_attr(len, "0x%lx\n");
aqdev_bar_attr(index, "%d\n");
#pragma GCC diagnostic warning "-Wformat"

static ssize_t bar_attr_show(struct kobject *kobj, struct attribute *attr,
			     char *buf)
{
	struct aqdiag_bar *bar = to_bar(kobj);
	struct bar_sysfs_entry *entry =
		container_of(attr, struct bar_sysfs_entry, attr);

	if (!entry->show)
		return -EIO;

	return entry->show(bar, buf);
}

static const struct sysfs_ops bar_ops = {
	.show = bar_attr_show,
};

static void bar_release(struct kobject *kobj)
{
	struct aqdiag_bar *bar = to_bar(kobj);
	kfree(bar);
}

static struct attribute *bar_attrs[] = {
	&bar_addr_attr.attr,
	&bar_len_attr.attr,
	&bar_index_attr.attr,
	NULL,
};

static struct kobj_type bar_type = {
	.sysfs_ops = &bar_ops,
	.release = bar_release,
	.default_attrs = bar_attrs,
};

#ifndef DEVICE_ATTR_RO // for older versions of linux (< 3.11), this is not defined
#define DEVICE_ATTR_RO(_name) \
	struct device_attribute dev_attr_##_name = __ATTR_RO(_name)
#endif

#define to_aqdev(kdev) container_of((kdev), struct aqdiag_dev, dev)
#define aqdiag_dev_attr_ro(_field, _fmt, _exp)				\
	static ssize_t _field##_show(struct device *dev,		\
				     struct device_attribute *attr,	\
				     char *buf)				\
	{								\
		struct aqdiag_dev *aqdev = to_aqdev(dev);		\
		return sprintf(buf, _fmt, (_exp));			\
	}								\
	static DEVICE_ATTR_RO(_field);

#pragma GCC diagnostic ignored "-Wformat"
aqdiag_dev_attr_ro(name, "aqdiag%d\n", aqdev->minor);
aqdiag_dev_attr_ro(inttype, "%d", aqdev->interrupts.type);
aqdiag_dev_attr_ro(vectors, "%d", aqdev->interrupts.num);
#pragma GCC diagnostic warning "-Wformat"

static struct attribute *aqdiag_attrs[] = {
	&dev_attr_name.attr,
	&dev_attr_inttype.attr,
	&dev_attr_vectors.attr,
	NULL,
};

static struct attribute_group aqdiag_attr_grp = {
	.attrs = aqdiag_attrs,
};

static void aqdiag_release_bars(struct aqdiag_dev *dev)
{
	int i;

	for (i = 0; i < PCI_ROM_RESOURCE; i++) {
		struct aqdiag_bar *bar = dev->sysfs_bar[i];
		if (!bar)
			continue;
		kobject_del(&bar->kobj);
		kobject_put(&bar->kobj);
	}

	kobject_del(dev->sysfs_bars);
	kobject_put(dev->sysfs_bars);
}

static int aqdiag_create_bars(struct aqdiag_dev *dev)
{
	int ret;
	int mask = pci_select_bars(dev->pdev, IORESOURCE_MEM | IORESOURCE_IO);
	int i;
	struct aqdiag_bar *bar;

	ret = -EINVAL;
	dev->sysfs_bars = kobject_create_and_add("bars", &dev->dev.kobj);
	if (!dev->sysfs_bars)
		goto err;

	for (i = 0; mask && i < PCI_ROM_RESOURCE; mask >>= 1, i++){
		const char *pref = pci_resource_flags(dev->pdev, i) & IORESOURCE_MEM ? "mem" : "io";

		if (!(mask & 1))
			continue;

		ret = -ENOMEM;
		bar = kzalloc(sizeof(*bar), GFP_KERNEL);
		if (!bar)
			goto err;
		kobject_init(&bar->kobj, &bar_type);
		bar->index = i;
		bar->addr = pci_resource_start(dev->pdev, i);
		bar->len = pci_resource_len(dev->pdev, i);
		dev->sysfs_bar[i] = bar;
		ret = kobject_add(&bar->kobj, dev->sysfs_bars, "%sbar%d", pref, i);
		if (ret)
			goto err;
	}

	return 0;

err:
	aqdiag_release_bars(dev);
	return ret;
}


int aqdiag_create_attrs(struct aqdiag_dev *dev)
{
	int ret;

	ret = sysfs_create_group(&dev->dev.kobj, &aqdiag_attr_grp);
	if (ret)
		goto err_group;

	dev->sysfs_mem = kobject_create_and_add("mem", &dev->dev.kobj);
	if (!dev->sysfs_mem)
		goto err_mem;

	ret = aqdiag_create_bars(dev);
	if (ret)
		goto err_bar;

	return 0;

err_bar:
	kobject_del(dev->sysfs_mem);
	kobject_put(dev->sysfs_mem);
err_mem:
	sysfs_remove_group(&dev->dev.kobj, &aqdiag_attr_grp);
err_group:
	dev_err(&dev->dev, "Couldn't create sysfs files: %d\n", ret);
	return ret;
}

void aqdiag_del_attrs(struct aqdiag_dev *dev)
{
	aqdiag_release_bars(dev);
	kobject_del(dev->sysfs_mem);
	kobject_put(dev->sysfs_mem);
	sysfs_remove_group(&dev->dev.kobj, &aqdiag_attr_grp);
}

#define to_memreg(obj) container_of(obj, struct aqdiag_memreg, kobj)
struct memreg_sysfs_entry {
	struct attribute attr;
	ssize_t (*show)(struct aqdiag_memreg *, char *);
	ssize_t (*store)(struct aqdiag_memreg *, char*, size_t);
};

#define aqdev_memreg_attr(_field, _fmt)					\
	static ssize_t							\
	memreg_##_field##_show(struct aqdiag_memreg *memreg, char *buf)	\
	{								\
		return sprintf(buf, _fmt, memreg->_field);		\
	}								\
	static struct memreg_sysfs_entry memreg_##_field##_attr =	\
		__ATTR(_field, S_IRUGO, memreg_##_field##_show, NULL);

#pragma GCC diagnostic ignored "-Wformat"
aqdev_memreg_attr(vaddr, "0x%lx\n");
aqdev_memreg_attr(paddr, "0x%lx\n");
aqdev_memreg_attr(size, "0x%x\n");
aqdev_memreg_attr(real_size, "0x%x\n");
aqdev_memreg_attr(index, "%d\n");
#pragma GCC diagnostic warning "-Wformat"

static ssize_t memreg_attr_show(struct kobject *kobj, struct attribute *attr,
			     char *buf)
{
	struct aqdiag_memreg *memreg = to_memreg(kobj);
	struct memreg_sysfs_entry *entry =
		container_of(attr, struct memreg_sysfs_entry, attr);

	if (!entry->show)
		return -EIO;

	return entry->show(memreg, buf);
}

static const struct sysfs_ops memreg_ops = {
	.show = memreg_attr_show,
};

static void memreg_release(struct kobject *kobj)
{
	struct aqdiag_memreg *memreg = to_memreg(kobj);
	kfree(memreg);
}

static struct attribute *memreg_attrs[] = {
	&memreg_vaddr_attr.attr,
	&memreg_paddr_attr.attr,
	&memreg_size_attr.attr,
	&memreg_real_size_attr.attr,
	&memreg_index_attr.attr,
	NULL,
};

struct kobj_type memreg_type = {
	.sysfs_ops = &memreg_ops,
	.release = memreg_release,
	.default_attrs = memreg_attrs,
};
/*
static int memreg_op_fault(struct vm_area_struct *vma, struct vm_fault *vmf)
{
	struct aqdiag_memreg *memreg = vma->vm_private_data;
	struct page *page;

	page = virt_to_page(memreg->vaddr + (vmf->pgoff << PAGE_SHIFT));
	get_page(page);
	vmf->page = page;
	return 0;
}

static struct vm_operations_struct memreg_vm_ops = {
	.fault = memreg_op_fault,
};
*/

static int memreg_mmap(struct file *file, struct kobject *kobj, struct bin_attribute *attr,
		       struct vm_area_struct *vma)
{
	struct aqdiag_memreg *memreg = attr->private;
	unsigned long requested = (vma->vm_end - vma->vm_start) >> PAGE_SHIFT;
	unsigned long pages = (unsigned long)memreg->real_size >> PAGE_SHIFT;

	if (vma->vm_pgoff + requested > pages)
		return -EINVAL;
	
#if defined(__arm__) || defined(__aarch64__)
	// had issues with writes to descriptors/packets not being seen by HW for arm systems. this function seemed to fix this
#ifdef pgprot_dmacoherent
	vma->vm_page_pgprot = prot_dmacoherent(vma->vm_page_prot);
#else //!defined(pgprot_dmacoherent)
	vma->vm_page_prot = pgprot_noncached(vma->vm_page_prot);
	//vma->vm_page_prot = pgprot_writecombine(vma->vm_page_prot);
#endif //defined(pgprot_dmacoherent)
#endif

	if (remap_pfn_range(vma, vma->vm_start, memreg->paddr >> PAGE_SHIFT,
			    vma->vm_end - vma->vm_start, vma->vm_page_prot))
		return -EAGAIN;
	return 0;
}

int aqdiag_publish_memreg(struct aqdiag_memreg *memreg)
{
	int ret;
	struct bin_attribute *mmap_attr = &memreg->mmap_attr;

	ret = kobject_add(&memreg->kobj, memreg->dev->sysfs_mem,
			  "%d", memreg->index);
	if (ret)
		goto err_add;

	mmap_attr->mmap = memreg_mmap;
	mmap_attr->attr.name = "mmap";
	mmap_attr->attr.mode = S_IRUSR | S_IWUSR;
	mmap_attr->size = memreg->real_size;
	mmap_attr->private = memreg;
	ret = sysfs_create_bin_file(&memreg->kobj, mmap_attr);
	if (ret)
		goto err_map_add;

	return 0;

err_map_add:
	kobject_del(&memreg->kobj);
err_add:
	kobject_put(&memreg->kobj);
	return ret;
}

void aqdiag_hide_memreg(struct aqdiag_memreg *memreg)
{
	sysfs_remove_bin_file(&memreg->kobj, &memreg->mmap_attr);
	kobject_del(&memreg->kobj);
	kobject_put(&memreg->kobj);
}

