<?php

interface Filter
{
	function filter($objArray);
}

abstract class BaseFilter implements Filter
{
	protected $reverse = false;
	
	protected abstract function internalPass($obj);
	
	public function pass($obj) 
	{
		$ret = false;
		
		if ($this->reverse) {
			$ret = ! $this->internalPass($obj);
		}
		else {
			$ret = $this->internalPass($obj);
		}
		
		return $ret;
	}
	
	public function isReverse()
	{
		return $this->reverse;
	}
	
	public function setReverse()
	{
		$this->reverse = true;
	}
	
	public function filter($objArray)
	{
		return array_filter($objArray, array($this, "pass"));
	}
}

class AndFilter extends BaseFilter
{
	private $filters = array();
	
	public function __construct($filters = array())
	{
		$this->filters = $filters;
		if (! isset($this->filters)) {
			$this->filters = array();
		}
	}

	public function addFilter($filter)
	{
		$this->filters[] = $filter;
	}
	
	public function getFilters()
	{
		return $this->filters;
	}
	
	protected function internalPass($obj)
	{
		if (! isset($this->filters)) {
			return true;
		}
		
		foreach ($this->filters as $filter)
		{
			if(!$filter->pass($obj))
			{				
				return false;
			}
		}
		
		return true;
	}
}

class OrFilter extends BaseFilter
{
	private $filters;
	
	public function __construct($filters = array())
	{
		$this->filters = $filters;
	}
	
	public function internalPass($obj)
	{
		foreach ($this->filters as $filter)
		{
			if($filter->pass($obj))
			{
				return true;
			}
		}
		
		return false;
	}
}

class HdValidMoreThanSizeFilter extends BaseFilter 
{
	private $size;
	private $convert;
	private $cannotEq;
	
	public function getSize()
	{
		return $this->size;
	}
	
	public function setSize($size)
	{
		$this->size = $size;
	}
	
	public function setCannotEqual()
	{
		$this->cannotEq = true;
	}
	
	public function setConvertor($convert)
	{
		$this->convert = $convert;
	}
	
	public function internalPass($hd)
	{
		if ($this->convert) {
			$convert = $this->convert;
			$HBAID = $hd->getAdapterID();
			if (! isset($HBAID) || $HBAID == GET_ALL) {
				return false;
			}
			$sysFeature=AdapterFeature::getInstance($HBAID);
			$UseThorSizePolicy = $sysFeature->isThor;
			if ($hd->isFree())
				$size = $convert($hd->getSize(),$UseThorSizePolicy);
			else 
				$size = $convert($hd->getValidSize(),$UseThorSizePolicy);
		}
		else 
		{
			if ($hd->isFree())
				$size = $hd->getSize();
			else 
			    $size = $hd->getValidSize();
		}
		
		if($this->cannotEq)
		{
			return $size > $this->size;
		}
		else {
			return  $size >= $this->size;
		}

	}
}

class HdIsFreeFilter extends BaseFilter 
{
	public function internalPass($hd)
	{
		return $hd->isFree();
	}
}

class HdIsAssignFilter extends BaseFilter
{
	public function internalPass($hd)
	{
		return $hd->isAssigned();
	}
}

class HdIsOfflineFilter extends BaseFilter 
{
	public function internalPass($hd)
	{
		return $hd->isOffline();
	}
}

class HdIsForeignFilter extends BaseFilter 
{
	public function internalPass($hd)
	{
		return $hd->isForeign();
	}
}

class HdCanUseFilter extends AndFilter
{
	protected function internalPass($hd)
	{
		$feature = AdapterFeature::getInstance($hd->getAdapterID());
		if ($feature->supportDG || $feature->isOdin || $feature->isVanir) {
			$this->addFilter(new HdIsFreeFilter());
		}
		
		return $hd->canLogicalUsed() && parent::internalPass($hd);
	}
}

class HdTypeFilter extends BaseFilter
{
	private $hd;
	public function setHd($HD)
	{
		$this->hd = $HD;
	}
	
	public function internalPass($hd)
	{
		$feature = AdapterFeature::getInstance($hd->getAdapterID());
		if ($feature->supportSameTypePDsForCreate) {
			if ($hd->isSAS() && $this->hd->isSATA()) {
				return false;
			}
			if ($hd->isSATA() && $this->hd->isSAS()) {
				return false;
			}
		}
		
		return true;
	}
}

class HdIsSpareFilter extends BaseFilter
{
	public function internalPass($hd)
	{
		return $hd->isSpare();
	}
}

class HdIsNotOtherDedicatedSpareFilter extends BaseFilter
{
	private $DGID = GET_ALL;
	public function SetDGID($DGID)
	{
		$this->DGID = $DGID;
	}
	public function internalPass($hd)
	{
		if ($hd->isDedicatedSpare())
			return ($hd->getDedicatedDGID() == $this->DGID);
		else 
			return true;
	}
}

class HdIsCryptoFilter extends BaseFilter
{
	public function internalPass($hd)
	{
                $crypto = $hd->getCrypto();
		if (($crypto & AESCryptoStatus::HD_AES_CRYPTO_DISK) == AESCryptoStatus::HD_AES_CRYPTO_DISK)
			return true;
		else 
			return false;
	}
}

class HdNotMemberFilter extends BaseFilter 
{
	private $members = array();
	
	public function setMembers($members)
	{
		$this->members = $members;
	}
	
	public function addMember($hd)
	{
		$members[] = $hd;
	}
	
	public function getMembers()
	{
		return $this->members;
	}
	
	public function internalPass($hd)
	{
		foreach ($this->members as $HDInfo)
		{
			if ($hd->getDeviceID() == $HDInfo->getDeviceID()) {
				return false;
			}
		}
		return true;
	}
}


class HdCreateFilter extends AndFilter 
{
	public function __construct()
	{
		$f1 = new HdValidMoreThanSizeFilter();
		$f1->setSize(0);
		$f1->setCannotEqual();
		$f1->setConvertor("ConvertSize");
		$f2 = new HdIsAssignFilter();
		$f2->setReverse();
		$f3 = new HdCanUseFilter();
		$f4 = new HdIsOfflineFilter();
		$f4->setReverse();
		$f5 = new HdIsSpareFilter();
		$f5->setReverse();
		$f6 = new HdIsForeignFilter();
		$f6->setReverse();
		$f7 = new HdIsCryptoFilter();
		$f7->setReverse();
		
		$this->addFilter($f1);
		$this->addFilter($f2);
		$this->addFilter($f3);
		$this->addFilter($f4);
		$this->addFilter($f5);
		$this->addFilter($f6);
		$this->addFilter($f7);
	}
}

class HdMPFilter extends AndFilter
{
	public function __construct()
	{
		$canUse = new HdCanUseFilter();
		$isOffline = new HdIsOfflineFilter();
		$isOffline->setReverse();
		
		$isSpare = new HdIsSpareFilter();
		$free = new HdIsFreeFilter();
		$free->setReverse();
				
		$filter = new OrFilter();
		$filter->addFilter($isSpare);
		$filter->addFilter($free);
		
		$this->addFilter($canUse);
		$this->addFilter($isOffline);
		$this->addFilter($filter);
	}
}

class HdRebuildFilter extends AndFilter
{	
	public function __construct($rebuildSize, $memberHDs)
	{
		//$f1 = new HdValidMoreThanSizeFilter(); //remove size check for tmp
		//$f1->setSize($rebuildSize);
		$f2 = new HdIsAssignFilter();
		$f2->setReverse();
		$f3 = new HdCanUseFilter();
		$f4 = new HdIsOfflineFilter();
		$f4->setReverse();
		
		$f5 = new HdNotMemberFilter();
		$f5->setMembers($memberHDs);
		
		$f6 = new HdTypeFilter();
		$f6->setHd(array_shift($memberHDs));
		
		//$this->addFilter($f1);
		$this->addFilter($f2);
		$this->addFilter($f3);
		$this->addFilter($f4);
		$this->addFilter($f5);
		$this->addFilter($f6);
	}
}

class HdRebuildForDgFilter extends HdRebuildFilter
{
	public function __construct($rebuildSize, $memberHDs, $DGID)
	{
		parent::__construct($rebuildSize, $memberHDs);
		
		$f1 = new HdIsNotOtherDedicatedSpareFilter();
		$f1->SetDGID($DGID);
		$this->addFilter($f1);
	}
}

class HdMigrateFilter extends AndFilter
{
	public function __construct($migrateSize, $memberHDs)
	{
		//$f1 = new HdValidMoreThanSizeFilter();  //remove size check for tmp
		//$f1->setSize($migrateSize);
		$f2 = new HdIsAssignFilter();
		$f2->setReverse();
		$f3 = new HdCanUseFilter();
		$f4 = new HdIsOfflineFilter();
		$f4->setReverse();
		
		$f5 = new HdNotMemberFilter();
		$f5->setMembers($memberHDs);
				
		$f6 = new HdIsSpareFilter();
		$f6->setReverse();
		
		$f7 = new HdTypeFilter();
		$f7->setHd(array_shift($memberHDs));
		
		//$this->addFilter($f1);
		$this->addFilter($f2);
		$this->addFilter($f3);
		$this->addFilter($f4);
		$this->addFilter($f5);
		$this->addFilter($f6);
		$this->addFilter($f7);
	}
}

class DgUsedToSpareFilter extends BaseFilter 
{
	private $hd;
	
	public function setHDToSpare($hd)
	{
		$this->hd = $hd;
	}
	
	protected function internalPass($dg)
	{
		//check raid mode
		if ($dg->getRaid_Level() == RAIDMode::LD_MODE_RAID0 ) {
			return false;
		}
		//check size
		if ($this->hd->getValidSize() < $dg->getSmallestAvailablePDSize()) {
			return false;
		}
		
		//check type
		$hdList = $dg->getHDList();
		if (sizeof($hdList) == 0) {
			return false;
		}
		$hd = array_shift($hdList);		
		
		$feature = AdapterFeature::getInstance($hd->getAdapterID());

		if ($feature->supportSameTypePDsForCreate) {
			if ($hd->isSAS() && $this->hd->isSATA()) {
				return false;
			}
			if ($hd->isSATA() && $this->hd->isSAS()) {
				return false;
			}
		}
		return true;
	}
}

?>