# <copyright>
# INTEL CONFIDENTIAL
#
# Copyright 2021 Intel Corporation
#
# This software and the related documents are Intel copyrighted materials, and your use of
# them is governed by the express license under which they were provided to you ("License").
# Unless the License provides otherwise, you may not use, modify, copy, publish, distribute,
# disclose or transmit this software or the related documents without Intel's prior written
# permission.
#
# This software and the related documents are provided as is, with no express or implied
# warranties, other than those that are expressly stated in the License.
#
# <copyright>

# Suppress irrelevant PS Script Analyzer warnings (trailing Param() is needed to help PSSA parse the file)
[Diagnostics.CodeAnalysis.SuppressMessageAttribute('PSAvoidGlobalVars', 'Global:FormatEnumerationLimit')]
[Diagnostics.CodeAnalysis.SuppressMessageAttribute("PSAvoidUsingPositionalParameters", "", Scope="function")]
[Diagnostics.CodeAnalysis.SuppressMessageAttribute("PSUseShouldProcessForStateChangingFunctions", "", Scope="function")] Param()

#.ExternalHelp IntelEthernetCmdlets.dll-Help.xml
function Get-IntelEthernetLogConfig
{
    [CmdletBinding()]
    Param(
    [parameter(Mandatory=$false)]
    [SupportsWildcards()]
    [ValidateNotNullOrEmpty()]
    [String[]]
    $Name = '',
    [parameter(Mandatory=$false,Position=0,ValueFromPipeline=$true)]
    [ValidateNotNullOrEmpty()]
    [object[]]
    $Adapter = $null,
    [parameter(Mandatory=$false,Position=1)]
    [SupportsWildcards()]
    [ValidateNotNullOrEmpty()]
    [String[]]
    $Module = '',
    [parameter(Mandatory=$false)]
    [SupportsWildcards()]
    [ValidateNotNullOrEmpty()]
    [String[]]
    $Configuration = '',
    [parameter(Mandatory=$false)]
    [ValidateNotNullOrEmpty()]
    [String[]]
    $SubModule = ''
    )
    Begin
    {
        $AdapterName = $Name
        $ModuleNames = $Module
        $ConfigurationNames = $Configuration
        $SubModuleSum = $SUBMODULE_DEFAULT_SUM
        $script:ErrorMessagesGet = @()
        $script:WarningMessagesGet = @()
        $LogConfigs = @()
        $DefaultFormatEnumLimit = $global:FormatEnumerationLimit
        $global:FormatEnumerationLimit = -1
        GetIntelEthernetDevices
        if ($script:SupportedAdapters)
        {
            $script:MSNetAdapters = @(Get-NetAdapter -InterfaceDescription $script:SupportedAdapters.Name -ErrorAction SilentlyContinue)
        }
    }
    Process
    {
        $Adapters = $Adapter

        $PreProcessedAdapterNames = ValidateGetAdapterNameParams $AdapterName $Adapters ([ref]$script:ErrorMessagesGet)
        $AdapterNames = @(GetSupportedAdapters $PreProcessedAdapterNames ([ref]$script:WarningMessagesGet))

        foreach ($a in $AdapterNames)
        {
            $TmpStatusMsg = CheckDeviceError $a
            if (-not [string]::IsNullOrEmpty($TmpStatusMsg))
            {
                $script:WarningMessagesGet += $TmpStatusMsg
                continue
            }

            $Local:IntelLogConfigurations = @{}
            $IntelLogConfigurations = GetConfigHashTable $a "\AdditionalConfiguration\INTEL_LOG_CONFIGURATIONS"
            $Local:IntelLogConfigModules = @{}
            $IntelLogConfigModules = GetConfigHashTable $a "\AdditionalConfiguration\INTEL_LOG_CONFIG_MODULES"
            $Local:IntelLogConfigLevels = @{}
            $IntelLogConfigLevels = GetConfigHashTable $a "\AdditionalConfiguration\INTEL_LOG_CONFIG_LEVELS"

            if ($IntelLogConfigLevels.Count -eq 0)
            {
                $IntelLogConfigLevels = $DRIVER_LOG_LEVEL_MAP
            }

            $DisplayConfigNames = @()
            if (!(ValidateConfigurationParam $ConfigurationNames $ModuleNames ([ref]$DisplayConfigNames) "Get" $a))
            {
                continue
            }

            $DisplayModules = @()
            $DisplayLevels = @()
            $DriverPassed = $null

            if ($DisplayConfigNames)
            {
                $MethodOutput = InvokeCimMethod "IntlLan_SetGenData" $a "WmiGetFwLogLevels"
                if (!(ValidateMethodOutput $MethodOutput "Get"))
                {
                    continue
                }

                GetConfigLogs $DisplayConfigNames ([ref]$DisplayModules) ([ref]$DisplayLevels) $MethodOutput
            }
            else
            {
                $DriverPassed = $ModuleNames.Where({$DRIVER_MODULE_NAME -like $_})

                if (-Not $DriverPassed -and $SubModule)
                {
                    $script:ErrorMessagesGet += $Messages.InvalidParams
                    continue
                }
                elseif ($SubModule)
                {
                    $SubModuleSum = GetSubModuleBitmask $SubModule ([ref]$script:ErrorMessagesGet)
                    if ($SubModuleSum -eq -1)
                    {
                        continue
                    }
                }

                if ($DriverPassed -or -Not $ModuleNames)
                {
                    $LogConfigs += GetDriverLogConfig $DriverPassed $SubModuleSum $a ([ref]$script:ErrorMessagesGet)
                }

                $MethodOutput = $null
                if ($IntelLogConfigModules.Count -gt 0)
                {
                    $MethodOutput = InvokeCimMethod "IntlLan_SetGenData" $a "WmiGetFwLogLevels"

                    if (!(ValidateMethodOutput $MethodOutput "Get"))
                    {
                        continue
                    }
                }

                # ValidateGetModuleParam will error if Driver is passed and not present here.
                $IntelLogConfigModules[$DRIVER_MODULE_NAME] = $DRIVER_MODULE_ID

                ValidateGetModuleParam ([ref]$DisplayModules) ([ref]$DisplayLevels) $ModuleNames $MethodOutput $a
            }

            if ($DisplayModules -and $DisplayLevels)
            {
                $LogConfigs += GatherOutput $a $DisplayModules $DisplayLevels $DriverPassed
            }
        }
    }
    End
    {
        $global:FormatEnumerationLimit = $DefaultFormatEnumLimit
        if ($LogConfigs)
        {
            Write-Output $LogConfigs
        }
        foreach ($WarningMessage in $script:WarningMessagesGet)
        {
            Write-Warning $WarningMessage
        }

        foreach ($ErrorMessage in $script:ErrorMessagesGet)
        {
            Write-Error $ErrorMessage
        }
    }
}

#.ExternalHelp IntelEthernetCmdlets.dll-Help.xml
function Set-IntelEthernetLogConfig
{
    [CmdletBinding()]
    Param(
    [parameter(Mandatory=$false)]
    [SupportsWildcards()]
    [ValidateNotNullOrEmpty()]
    [String[]]
    $Name = '',
    [parameter(Mandatory=$false,Position=0,ValueFromPipeline=$true)]
    [ValidateNotNullOrEmpty()]
    [object[]]
    $Adapter = $null,
    [parameter(Mandatory=$false,Position=1)]
    [SupportsWildcards()]
    [ValidateNotNullOrEmpty()]
    [String[]]
    $Module = '',
    [parameter(Mandatory=$false,Position=2)]
    [SupportsWildcards()]
    [ValidateNotNullOrEmpty()]
    [String]
    $Level = '',
    [parameter(Mandatory=$false)]
    [SupportsWildcards()]
    [ValidateNotNullOrEmpty()]
    [String[]]
    $Configuration = '',
    [parameter(Mandatory=$false)]
    [ValidateNotNullOrEmpty()]
    [String[]]
    $SubModule = ''
    )
    Begin
    {
        $ModuleNames = $Module
        $LevelName = $Level
        $ConfigurationNames = $Configuration
        $SubModuleSum = $SUBMODULE_DEFAULT_SUM
        $script:WarningMessagesSet = @()
        $script:ErrorMessagesSet = @()
        $LogConfigs = @()
        GetIntelEthernetDevices
        if ($script:SupportedAdapters)
        {
            $script:MSNetAdapters = @(Get-NetAdapter -InterfaceDescription $script:SupportedAdapters.Name -ErrorAction SilentlyContinue)
        }
    }
    Process
    {
        do
        {
            $PreProcessedAdapterNames = ValidateSetAdapterNameParams $Name $Adapter ([ref]$script:ErrorMessagesSet)
            $AdapterNames = @(GetSupportedAdapters $PreProcessedAdapterNames ([ref]$script:ErrorMessagesSet))

            if ([string]::IsNullOrEmpty($ConfigurationNames) -and ([string]::IsNullOrEmpty($ModuleNames)))
            {
                $script:ErrorMessagesSet += $Messages.InvalidParamsConfigOrModule
                continue
            }

            foreach($a in $AdapterNames)
            {
                $TmpStatusMsg = CheckDeviceError $a
                if (-not [string]::IsNullOrEmpty($TmpStatusMsg))
                {
                    $script:ErrorMessagesSet += $TmpStatusMsg
                    continue
                }

                $DisplayModules = @()
                $DisplayLevels = @()
                $InputModulesId = [uint32[]]::new(64)
                $InputLevelId = [uint32[]]::new(64)

                $Local:IntelLogConfigurations = @{}
                $IntelLogConfigurations = GetConfigHashTable $a "\AdditionalConfiguration\INTEL_LOG_CONFIGURATIONS"
                $Local:IntelLogConfigModules = @{}
                $IntelLogConfigModules = GetConfigHashTable $a "\AdditionalConfiguration\INTEL_LOG_CONFIG_MODULES"

                $IntelLogConfigModules[$DRIVER_MODULE_NAME] = $DRIVER_MODULE_ID

                $Local:IntelLogConfigLevels = @{}
                $IntelLogConfigLevels = GetConfigHashTable $a "\AdditionalConfiguration\INTEL_LOG_CONFIG_LEVELS"

                if ($IntelLogConfigLevels.Count -eq 0)
                {
                    $IntelLogConfigLevels = $DRIVER_LOG_LEVEL_MAP
                }

                $ModuleCount = 0
                $DisplayConfigNames = @()
                if (!(ValidateConfigurationParam $ConfigurationNames $ModuleNames ([ref]$DisplayConfigNames) "Set" $a))
                {
                    continue
                }

                $bDriverPassed = $false
                $InputConfigurationIDs = $null
                if ($DisplayConfigNames)
                {
                    $InputConfigurationIDs = $IntelLogConfigurations[$DisplayConfigNames]
                }
                else
                {
                    ValidateSetModuleParam ([ref]$InputModulesId) ([ref]$DisplayModules) $ModuleNames $a

                    if ($DisplayModules -contains $DRIVER_MODULE_NAME)
                    {
                        $bDriverPassed = $true
                    }
                    elseif ($SubModule)
                    {
                        $script:ErrorMessagesSet += $Messages.InvalidParams
                        continue
                    }

                    $ModuleCount = $DisplayModules.Count
                }

                if ($ModuleCount -ne 0)
                {
                    if (!(ValidateLevelParam ([ref]$InputLevelId) ([ref]$DisplayLevels) $LevelName $ModuleCount $a $bDriverPassed))
                    {
                        continue
                    }

                    if ($bDriverPassed)
                    {
                        # Driver module is not included in Wmi method params
                        $ModuleCount -= 1

                        if ($InputLevelId[0] -lt 3)
                        {
                            $script:WarningMessagesSet += $Messages.InvalidDriverLevel
                            $bDriverPassed = $false
                        }
                        else
                        {
                            if ($SubModule)
                            {
                                $SubModuleSum = GetSubModuleBitmask $SubModule ([ref]$script:ErrorMessagesSet)
                                if ($SubModuleSum -eq -1)
                                {
                                    continue
                                }
                            }

                            if (-not (SetSubModuleRegistryValue $a $SubModuleSum))
                            {
                                $script:ErrorMessagesSet += $Messages.InvalidSystemConfigFailure
                                continue
                            }

                            $LevelName = GetLevelName($InputLevelId[0])

                            if (-Not (SetDriverLogLevel $a $LevelName))
                            {
                                $script:ErrorMessagesSet += $Messages.InvalidSystemConfigFailure
                                continue
                            }
                        }
                    }

                    if ($ModuleCount -gt 0)
                    {
                        $params = @{Count    = [uint32]$ModuleCount;
                                ModuleID = $InputModulesId;
                                Loglevel = $InputLevelId}

                        $MethodOutput = InvokeCimMethod "IntlLan_SetGenData" $a "WmiSetFwLogLevels" $params
                    }

                    if ($bDriverPassed)
                    {
                        # Driver is set, display it. Avoid call to Validate if no other modules passed.
                        if ($ModuleCount -eq 0 -or (ValidateMethodOutput $MethodOutput "Set"))
                        {
                            $LogConfigs += Get-IntelEthernetLogConfig -Name $a -Module $DisplayModules -SubModule $SubModuleSum
                        }
                        else
                        {
                            $LogConfigs += Get-IntelEthernetLogConfig -Name $a -Module $DRIVER_MODULE_NAME -SubModule $SubModuleSum
                        }
                    }
                    elseif (ValidateMethodOutput $MethodOutput "Set")
                    {
                        $LogConfigs += Get-IntelEthernetLogConfig -Name $a -Module $DisplayModules
                    }
                }
                elseif ($null -ne $InputConfigurationIDs)
                {
                    $count = 1
                    $DisplayConfigList = @()
                    if (!(ValidateLevelParam ([ref]$InputLevelId) ([ref]$DisplayLevels) $LevelName $count $a))
                    {
                        continue
                    }

                    $bContinue = $true
                    foreach ($ConfigID in $InputConfigurationIDs)
                    {
                        $params = @{ConfigurationID = [uint32]$ConfigID;
                                    Level = $InputLevelId[0]}

                        $MethodOutput = InvokeCimMethod "IntlLan_SetGenData" $a "WmiSetFwLogConfiguration" $params

                        if (!(ValidateMethodOutput $MethodOutput "Set"))
                        {
                            $bContinue = $false
                            continue
                        }

                        $DisplayConfigList += ($IntelLogConfigurations.GetEnumerator() | Where-Object { $_.Value -eq $ConfigID} ).Name
                    }

                    if (!$bContinue)
                    {
                        continue
                    }
                    $LogConfigs += Get-IntelEthernetLogConfig -Name $a -Configuration $DisplayConfigList
                }
            }
        } while ($false)
    }
    End
    {
        if ($LogConfigs)
        {
            Write-Output $LogConfigs
        }

        foreach ($WarningMessage in $script:WarningMessagesSet)
        {
            Write-Warning $WarningMessage
        }

        foreach ($ErrorMessage in $script:ErrorMessagesSet)
        {
            Write-Error $ErrorMessage
        }
    }
}

#.ExternalHelp IntelEthernetCmdlets.dll-Help.xml
function Start-IntelEthernetLog
{
    [CmdletBinding()]
    Param(
    [parameter(Mandatory=$false)]
    [ValidateNotNullOrEmpty()]
    [String]
    $Name = '',
    [parameter(Mandatory=$false,Position=0,ValueFromPipeline=$true)]
    [ValidateNotNullOrEmpty()]
    [object]
    $Adapter = $null,
    [parameter(Mandatory=$false,Position=1)]
    [ValidateNotNullOrEmpty()]
    [string]
    $Path = '',
    [parameter(Mandatory=$false)]
    [switch]
    $Force,
    [parameter(Mandatory=$false)]
    [switch]
    $Append
    )
    Begin
    {
        $SubModuleSum = 0xFFFF
        # Set LogmanLevel to Normal by default
        $LogmanLevel = 4
        $script:ErrorMessagesStart = @()
        $script:WarningMessagesStart = @()
        GetIntelEthernetDevices
        if ($script:SupportedAdapters)
        {
            $script:MSNetAdapters = @(Get-NetAdapter -InterfaceDescription $script:SupportedAdapters.Name -ErrorAction SilentlyContinue)
        }
    }
    End
    {
        do
        {
            if ($Name -match '\*')
            {
                $script:ErrorMessagesStart += $Messages.AdapterNotFound -f $Name
                break
            }

            $PreProcessedAdapterNames = ValidateSetAdapterNameParams $Name $Adapter ([ref]$script:ErrorMessagesStart)
            $AdapterName = GetSupportedAdapters $PreProcessedAdapterNames ([ref]$script:ErrorMessagesStart)

            if (([string]::IsNullOrEmpty($AdapterName)))
            {
                break
            }

            if (-not (ValidateSingleAdapter $input $AdapterName ([ref]$script:ErrorMessagesStart)))
            {
                break
            }

            $TmpStatusMsg = CheckDeviceError $AdapterName
            if (-not [string]::IsNullOrEmpty($TmpStatusMsg))
            {
                $script:ErrorMessagesStart += $TmpStatusMsg
                break
            }

            $UseDefaultPath = $false
            if (([string]::IsNullOrEmpty($Path)))
            {
                $UseDefaultPath = $true
            }

            $LogmanGUID = GetLogmanGUID $AdapterName ([ref]$script:ErrorMessagesStart)
            $LogName = GetLogName $AdapterName ([ref]$script:ErrorMessagesStart)

            if ($null -eq $LogmanGUID -or $null -eq $LogName)
            {
                break
            }

            if (LogmanProcessRunning $AdapterName $LogName $LogmanGUID)
            {
                break
            }

            $LogmanPath = ''
            if (-not (ValidatePathParams ([ref]$LogmanPath) $UseDefaultPath $LogName ([ref]$script:ErrorMessagesStart)))
            {
                break
            }

            $DriverLogLevel = GetDriverLogLevel $AdapterName
            if ($DriverLogLevel)
            {
                $Local:IntelLogConfigLevels = @{}
                $IntelLogConfigLevels = GetConfigHashTable $AdapterName "\AdditionalConfiguration\INTEL_LOG_CONFIG_LEVELS"

                if ($IntelLogConfigLevels.Count -eq 0)
                {
                    $IntelLogConfigLevels = $DRIVER_LOG_LEVEL_MAP
                }

                # Driver log levels are normal levels value + 1
                $LogmanLevel = $IntelLogConfigLevels[$DriverLogLevel] + 1

                $LogMetadataName = "$LogName.txt"

                $LogMetadataPath = ''
                if (-Not $UseDefaultPath)
                {
                    if ([IO.Path]::GetExtension($Path) -eq ".txt")
                    {
                        $Path += ".txt"
                    }
                    else
                    {
                        $Path = [IO.Path]::ChangeExtension($Path, ".txt")
                    }
                }

                $iBeforeErrorCount = $script:ErrorMessagesStart.Count
                if (-not (ValidatePathParams ([ref]$LogMetadataPath) $UseDefaultPath $LogMetadataName ([ref]$script:ErrorMessagesStart)))
                {
                    if ($iBeforeErrorCount -lt $script:ErrorMessagesStart.Count -and $script:ErrorMessagesStart[-1] -eq $Messages.LogmanFileExists -f $AdapterName)
                    {
                        # Driver Metadata File is always replaced if it exists.
                        $script:ErrorMessagesStart = $script:ErrorMessagesStart | Select-Object -First ($script:ErrorMessagesStart.Count - 1)
                    }
                    else
                    {
                        break
                    }
                }

                $AdapterRegistryPath = GetAdapterRegistryConfigPath $AdapterName ""
                if ($null -eq $AdapterRegistryPath)
                {
                    $script:ErrorMessagesStart += $Messages.NoCmdletSupport -f $AdapterName
                    break
                }

                $SubModuleRegistryValue = Get-ItemProperty -Path $AdapterRegistryPath -Name $SUBMODULE_REGISTRYVAR_NAME -ErrorAction SilentlyContinue
                if ($SubModuleRegistryValue)
                {
                    $SubModuleSum = $SubModuleRegistryValue.SubModule

                    if (-Not (CreateDriverLogMetadataFile $AdapterName))
                    {
                        break
                    }
                }
            }

            LogmanStart $LogmanPath $LogName $LogmanGUID $SubModuleSum $LogmanLevel
            if (ValidateLogmanStart $AdapterName)
            {
                $LogmanPath = Resolve-Path -Path $LogmanPath -ErrorAction SilentlyContinue
                Write-Output $LogmanPath
            }

            break
        } while ($false)

        foreach ($WarningMessage in $script:WarningMessagesStart)
        {
            Write-Warning $WarningMessage
        }

        foreach ($ErrorMessage in $script:ErrorMessagesStart)
        {
            Write-Error $ErrorMessage
        }
    }
}

#.ExternalHelp IntelEthernetCmdlets.dll-Help.xml
function Stop-IntelEthernetLog
{
    [CmdletBinding()]
    Param(
    [parameter(Mandatory=$false)]
    [ValidateNotNullOrEmpty()]
    [String]
    $Name = '',
    [parameter(Mandatory=$false,Position=0,ValueFromPipeline=$true)]
    [ValidateNotNullOrEmpty()]
    [object]
    $Adapter = $null
    )
    Begin
    {
        $script:ErrorMessagesStop = @()
        $script:WarningMessagesStop = @()
        $bErrorObtained = $false
        GetIntelEthernetDevices
        if ($script:SupportedAdapters)
        {
            $script:MSNetAdapters = @(Get-NetAdapter -InterfaceDescription $script:SupportedAdapters.Name -ErrorAction SilentlyContinue)
        }
    }
    Process
    {
        $PreProcessedAdapterNames = ValidateGetAdapterNameParams $Name $Adapter ([ref]$script:ErrorMessagesStop)
        $AdapterNames = @(GetSupportedAdapters $PreProcessedAdapterNames ([ref]$script:WarningMessagesStop))

        foreach ($a in $AdapterNames)
        {
            $TmpStatusMsg = CheckDeviceError $a
            if (-not [string]::IsNullOrEmpty($TmpStatusMsg))
            {
                $script:WarningMessagesStop += $TmpStatusMsg
                continue
            }

            if (-Not $bErrorObtained)
            {
                $LogName = GetLogName $a ([ref]$script:ErrorMessagesStop)
            }
            else
            {
                $LogName = GetLogName $a ([ref]@())
            }

            if ($null -eq $LogName)
            {
                $bErrorObtained = $true
                continue
            }

            LogmanStop $LogName
            if (-Not $bErrorObtained)
            {
                if (-Not (ValidateLogmanStop $a ([ref]$script:ErrorMessagesStop)))
                {
                    $bErrorObtained = $true
                }
            }
        }
    }
    End
    {
        foreach ($WarningMessage in $script:WarningMessagesStop)
        {
            Write-Warning $WarningMessage
        }

        foreach ($ErrorMessage in $script:ErrorMessagesStop)
        {
            Write-Error $ErrorMessage
        }
    }
}

#.ExternalHelp IntelEthernetCmdlets.dll-Help.xml
function Disable-IntelEthernetLogConfig
{
    [CmdletBinding()]
    Param(
    [parameter(Mandatory=$false)]
    [SupportsWildcards()]
    [ValidateNotNullOrEmpty()]
    [String[]]
    $Name = '',
    [parameter(Mandatory=$false,Position=0,ValueFromPipeline=$true)]
    [ValidateNotNullOrEmpty()]
    [object[]]
    $Adapter = $null
    )
    Begin
    {
        $AdapterName = $Name
        $script:ErrorMessagesDisable = @()
        $script:WarningMessagesDisable = @()
        $LogConfigs = @()
        $DefaultFormatEnumLimit = $global:FormatEnumerationLimit
        $global:FormatEnumerationLimit = -1
        GetIntelEthernetDevices
        if ($script:SupportedAdapters)
        {
            $script:MSNetAdapters = @(Get-NetAdapter -InterfaceDescription $script:SupportedAdapters.Name -ErrorAction SilentlyContinue)
        }
    }
    Process
    {
        $Adapters = $Adapter

        $PreProcessedAdapterNames = ValidateGetAdapterNameParams $AdapterName $Adapters ([ref]$script:ErrorMessagesDisable)
        $AdapterNames = @(GetSupportedAdapters $PreProcessedAdapterNames ([ref]$script:WarningMessagesDisable))

        foreach ($a in $AdapterNames)
        {
            $TmpStatusMsg = CheckDeviceError $a
            if (-not [string]::IsNullOrEmpty($TmpStatusMsg))
            {
                $script:ErrorMessagesDisable += $TmpStatusMsg
                continue
            }

            $DriverConfig = Set-IntelEthernetLogConfig -Name $a -Module $DRIVER_MODULE_NAME -SubModule @($SUB_MODULE_MAP.Keys)[0] -Level "Normal"
            if (-Not $DriverConfig)
            {
                continue
            }

            $MethodOutput = InvokeCimMethod "IntlLan_SetGenData" $a "WmiDisableFwLogLevels"

            if ($null -ne $MethodOutput)
            {
                if (!(ValidateMethodOutput $MethodOutput "Disable"))
                {
                    continue
                }
            }

            # Will pass Driver as module, so returns IntelDriverLogConfiguration
            $LogConfigs += Get-IntelEthernetLogConfig -Name $a -Module "*" -SubModule @($SUB_MODULE_MAP.Keys)[0]
        }
    }
    End
    {
        $global:FormatEnumerationLimit = $DefaultFormatEnumLimit
        if ($LogConfigs)
        {
            Write-Output $LogConfigs
        }

        foreach ($WarningMessage in $script:WarningMessagesDisable)
        {
            Write-Warning $WarningMessage
        }

        foreach ($ErrorMessage in $script:ErrorMessagesDisable)
        {
            Write-Error $ErrorMessage
        }
    }
}

#.ExternalHelp IntelEthernetCmdlets.dll-Help.xml
function Reset-IntelEthernetLogConfig
{
    [CmdletBinding()]
    Param(
    [parameter(Mandatory=$false)]
    [SupportsWildcards()]
    [ValidateNotNullOrEmpty()]
    [String[]]
    $Name = '',
    [parameter(Mandatory=$false,Position=0,ValueFromPipeline=$true)]
    [ValidateNotNullOrEmpty()]
    [object[]]
    $Adapter = $null
    )
    Begin
    {
        $AdapterName = $Name
        $script:ErrorMessagesReset = @()
        $script:WarningMessagesReset = @()
        $LogConfigs = @()
        $DefaultFormatEnumLimit = $global:FormatEnumerationLimit
        $global:FormatEnumerationLimit = -1
        GetIntelEthernetDevices
        if ($script:SupportedAdapters)
        {
            $script:MSNetAdapters = @(Get-NetAdapter -InterfaceDescription $script:SupportedAdapters.Name -ErrorAction SilentlyContinue)
        }
    }
    Process
    {
        $Adapters = $Adapter

        $PreProcessedAdapterNames = ValidateGetAdapterNameParams $AdapterName $Adapters ([ref]$script:ErrorMessagesReset)
        $AdapterNames = @(GetSupportedAdapters $PreProcessedAdapterNames ([ref]$script:WarningMessagesReset))

        foreach ($a in $AdapterNames)
        {
            $TmpStatusMsg = CheckDeviceError $a
            if (-not [string]::IsNullOrEmpty($TmpStatusMsg))
            {
                $script:ErrorMessagesReset += $TmpStatusMsg
                continue
            }

            # Set driver level back to Normal
            $null = Set-IntelEthernetLogConfig -Name $a -Module $DRIVER_MODULE_NAME

            $MethodOutput = InvokeCimMethod "IntlLan_SetGenData" $a "WmiResetFwLogLevels"

            if ($null -ne $MethodOutput)
            {
                if (!(ValidateMethodOutput $MethodOutput "Reset"))
                {
                    continue
                }
            }

            $LogConfigs += Get-IntelEthernetLogConfig -Name $a -Module "*"
        }
    }
    End
    {
        $global:FormatEnumerationLimit = $DefaultFormatEnumLimit
        if ($LogConfigs)
        {
            Write-Output $LogConfigs
        }

        foreach ($WarningMessage in $script:WarningMessagesReset)
        {
            Write-Warning $WarningMessage
        }

        foreach ($ErrorMessage in $script:ErrorMessagesReset)
        {
            Write-Error $ErrorMessage
        }
    }
}

Function GatherOutput($AdapterName, $DisplayModules, $DisplayLevels, $DriverPassed)
{
    $LogConfigs = @()
    $ValidLevels = $IntelLogConfigLevels.Keys
    $LogConfigCount = $DisplayModules.Count
    for ($i = 0; $i -lt $LogConfigCount; $i++)
    {
        if ($DriverPassed)
        {
            $LogConfigs += [PsCustomObject] @{ PSTypeName  = 'IntelDriverLogConfiguration';
                Name        = $AdapterName;
                Module      = $DisplayModules[$i];
                SubModule  = "";
                Level       = $DisplayLevels[$i];
                ValidSubModules = @();
                ValidLevels = $ValidLevels}
        }
        else
        {
            $LogConfigs += [PsCustomObject] @{ PSTypeName  = 'IntelLogConfiguration';
                Name        = $AdapterName;
                Module      = $DisplayModules[$i];
                Level       = $DisplayLevels[$i];
                ValidSubModules = @();
                ValidLevels = $ValidLevels}
        }
    }

    return $LogConfigs
}

Function GetModuleName($ModuleId)
{
    return $IntelLogConfigModules.GetEnumerator().Where({$_.Value -eq $ModuleId}).Name
}

Function GetLevelName($LevelId)
{
    return $IntelLogConfigLevels.GetEnumerator().Where({$_.Value -eq $LevelId}).Name
}

Function GetDriverLogLevel($AdapterName)
{
    $DriverRegistryValue = $null

    $AdapterRegistryPath = GetAdapterRegistryConfigPath $AdapterName ""
    if ($null -ne $AdapterRegistryPath)
    {
        $DriverRegistryValue = Get-ItemProperty -Path $AdapterRegistryPath -Name $DRIVERLOG_REGISTRYVAR_NAME -ErrorAction SilentlyContinue
    }

    if ($null -ne $DriverRegistryValue)
    {
        $DriverRegistryValue = $DriverRegistryValue.DriverLogLevel
    }

    return $DriverRegistryValue
}

Function GetDriverLogConfig($DriverPassed, $SubModuleSum, $AdapterName, [ref]$ErrorMessages)
{
    $LogConfigs = @()
    $InactiveSubModuleArray = @()
    $ActiveSubModuleArray = @()

    $DriverLevel = GetDriverLogLevel $AdapterName
    if (-Not $DriverLevel)
    {
        # Set to Normal by default
        $DriverLevel = @($DRIVER_LOG_LEVEL_MAP.Keys)[1]
        if (-Not (SetDriverLogLevel $AdapterName $DriverLevel))
        {
            $ErrorMessages.Value += $Messages.InvalidSystemConfigFailure
            break
        }
    }

    $AdapterRegistryPath = GetAdapterRegistryConfigPath $AdapterName ""
    if ($null -eq $AdapterRegistryPath)
    {
        $ErrorMessages.Value += $Messages.NoCmdletSupport -f $AdapterName
        break
    }

    $SubModuleRegistryValue = Get-ItemProperty -Path $AdapterRegistryPath -Name $SUBMODULE_REGISTRYVAR_NAME -ErrorAction SilentlyContinue
    if (-Not $SubModuleRegistryValue -or [string]::IsNullOrEmpty($SubModuleRegistryValue.SubModule))
    {
        # If driver passed but set has not been used, enable default driver logging
        if (-not (SetSubModuleRegistryValue $a $SUBMODULE_DEFAULT_SUM))
        {
            $ErrorMessages.Value += $Messages.InvalidSystemConfigFailure
            break
        }

        $RegistrySubModuleSum = $SUBMODULE_DEFAULT_SUM
    }
    else
    {
        $RegistrySubModuleSum = $SubModuleRegistryValue.SubModule
    }

    foreach ($DisplayName in $SUB_MODULE_MAP.Keys)
    {
        if ($SUB_MODULE_MAP[$DisplayName] -band $SubModuleSum)
        {
            if ($SUB_MODULE_MAP[$DisplayName] -band $RegistrySubModuleSum)
            {
                $ActiveSubModuleArray += $DisplayName
            }
            else
            {
                $InactiveSubModuleArray += $DisplayName
            }
        }
    }

    if ($DriverPassed)
    {
        foreach ($ActiveSubModule in $ActiveSubModuleArray)
        {
            $LogConfigs += [PsCustomObject] @{ PSTypeName  = 'IntelDriverLogConfiguration';
                                            Name        = $AdapterName;
                                            Module      = $DRIVER_MODULE_NAME;
                                            SubModule   = $ActiveSubModule;
                                            Level       = $DriverLevel;
                                            ValidSubModules = $SUB_MODULE_MAP.Keys;
                                            ValidLevels = @($DRIVER_LOG_LEVEL_MAP.Keys)[0,1] }
        }
        foreach ($InactiveSubModule in $InactiveSubModuleArray)
        {
            $LogConfigs += [PsCustomObject] @{ PSTypeName  = 'IntelDriverLogConfiguration';
                                            Name        = $AdapterName;
                                            Module      = $DRIVER_MODULE_NAME;
                                            SubModule   = $InactiveSubModule;
                                            Level       = @($DRIVER_LOG_LEVEL_MAP.Keys)[-1]; # Disabled
                                            ValidSubModules = $SUB_MODULE_MAP.Keys;
                                            ValidLevels = @($DRIVER_LOG_LEVEL_MAP.Keys)[0,1] }
        }
    }
    else
    {
        $LogConfigs += [PsCustomObject] @{ PSTypeName  = 'IntelLogConfiguration';
                                        Name            = $AdapterName;
                                        Module          = $DRIVER_MODULE_NAME;
                                        SubModule       = @($ActiveSubModuleArray);
                                        Level           = $DriverLevel;
                                        ValidSubModules = $SUB_MODULE_MAP.Keys;
                                        ValidLevels = @($DRIVER_LOG_LEVEL_MAP.Keys)[0,1] }
    }

    return $LogConfigs
}

Function GetSubModuleBitmask($SubModuleArray, [ref]$ErrorMessages)
{
    $iSubModuleSum = 0

    foreach ($SubModule in $SubModuleArray)
    {
        if ($SubModule -match "^0[xX][A-Fa-f0-9]+$" -or $SubModule -match "^[1-9]\d*$")
        {
            if ([int]$SubModule -le $SUM_SUBMODULES)
            {
                $iSubModuleSum = $iSubModuleSum -bor $SubModule
            }
            else
            {
                $iSubModuleSum = $SUM_SUBMODULES
                continue
            }
        }
        elseif ($SUB_MODULE_MAP.Keys -Contains $SubModule)
        {
            $iSubModuleSum = $iSubModuleSum -bor $SUB_MODULE_MAP[$SubModule]
        }
        else
        {
            $ErrorMessages.Value += $Messages.InvalidSubModule -f $a, $SubModule
            $iSubModuleSum = -1
            break
        }
    }

    return $iSubModuleSum
}

Function GetLogmanGUID($AdapterName, [ref]$ErrorMessages)
{
    $CurrentAdapter = $script:PnpDevice.Where({ $_.FriendlyName -eq $AdapterName })
    if ($CurrentAdapter)
    {
        switch ($CurrentAdapter.Service)
        {
            'i40ea' { return $I40EA_WPP_GUID; break }
            'i40eb' { return $I40EB_WPP_GUID; break }
            'ixw'   { return $IXW_WPP_GUID; break }
            'icea' {
                $WppGuid = GetItemPropertyValue $AdapterName "\AdditionalConfiguration" "WPP_GUID"

                if (-Not $WppGuid)
                {
                    $WppGuid = $ICEA_WPP_GUID
                }

                return $WppGuid
                break
            }
            'iceb' {
                $WppGuid = GetItemPropertyValue $AdapterName "\AdditionalConfiguration" "WPP_GUID"

                if (-Not $WppGuid)
                {
                    $WppGuid = $ICEB_WPP_GUID
                }

                return $WppGuid
                break
            }
            'scea' {
                $WppGuid = GetItemPropertyValue $AdapterName "\AdditionalConfiguration" "WPP_GUID"

                if (-Not $WppGuid)
                {
                    $WppGuid = $SCEA_WPP_GUID
                }

                return $WppGuid
                break
            }
            'sceb' {
                $WppGuid = GetItemPropertyValue $AdapterName "\AdditionalConfiguration" "WPP_GUID"

                if (-Not $WppGuid)
                {
                    $WppGuid = $SCEB_WPP_GUID
                }

                return $WppGuid
                break
            }
            'ixw' {
                $WppGuid = GetItemPropertyValue $AdapterName "\AdditionalConfiguration" "WPP_GUID"

                if (-Not $WppGuid)
                {
                    $WppGuid = $IXW_WPP_GUID
                }

                return $WppGuid
                break
            }
            default {
                $TmpErrorMsg = $Messages.NoCmdletSupport -f $AdapterName
                if (-not ($ErrorMessages.Value -contains $TmpErrorMsg))
                {
                    $ErrorMessages.Value += $TmpErrorMsg
                }

                break
            }
        }
    }

    return $null
}

Function GetLogName($AdapterName, [ref]$ErrorMessages)
{
    $CurrentAdapter = $script:PnpDevice.Where({ $_.FriendlyName -eq $AdapterName })
    if ($CurrentAdapter)
    {
        switch ($CurrentAdapter.Service)
        {
            'i40ea' { return $I40EA_LOGNAME; break }
            'i40eb' { return $I40EB_LOGNAME; break }
            'ixw'   { return $IXW_LOGNAME; break }
            'icea' {
                $LogName = GetItemPropertyValue $AdapterName "\AdditionalConfiguration" "FW_LOG_NAME"
                if (-Not $LogName)
                {
                    $LogName = $ICEA_LOGNAME
                }

                return $LogName
                break
            }
            'iceb' {
                $LogName = GetItemPropertyValue $AdapterName "\AdditionalConfiguration" "FW_LOG_NAME"
                if (-Not $LogName)
                {
                    $LogName = $ICEB_LOGNAME
                }

                return $LogName
                break
            }
            'scea' {
                $LogName = GetItemPropertyValue $AdapterName "\AdditionalConfiguration" "FW_LOG_NAME"
                if (-Not $LogName)
                {
                    $LogName = $SCEA_LOGNAME
                }

                return $LogName
                break
            }
            'sceb' {
                $LogName = GetItemPropertyValue $AdapterName "\AdditionalConfiguration" "FW_LOG_NAME"
                if (-Not $LogName)
                {
                    $LogName = $SCEB_LOGNAME
                }

                return $LogName
                break
            }
            'ixw' {
                $LogName = GetItemPropertyValue $AdapterName "\AdditionalConfiguration" "FW_LOG_NAME"
                if (-Not $LogName)
                {
                    $LogName = $IXW_LOGNAME
                }

                return $LogName
                break
            }
            default {
                $TmpErrorMsg = $Messages.NoCmdletSupport -f $AdapterName
                if (-not ($ErrorMessages.Value -contains $TmpErrorMsg))
                {
                    $ErrorMessages.Value += $TmpErrorMsg
                }

                break
            }
        }
    }

    return $null
}

Function SetDriverLogLevel($AdapterName, $DriverLevel)
{
    $bSuccess = $false
    $SetDriverLogLevel = $null

    $AdapterRegistryPath = GetAdapterRegistryConfigPath $AdapterName ""
    if ($null -ne $AdapterRegistryPath)
    {
        $SetDriverLogLevel = Set-ItemProperty -Path $AdapterRegistryPath -Name $DRIVERLOG_REGISTRYVAR_NAME -Type "String" -Value $DriverLevel -PassThru -ErrorAction SilentlyContinue
    }

    if ($null -ne $SetDriverLogLevel -and -Not [string]::IsNullOrEmpty($SetDriverLogLevel.DriverLogLevel))
    {
        $bSuccess = $true
    }

    return $bSuccess
}

Function SetSubModuleRegistryValue($AdapterName, $SubModuleValue)
{
    $bSuccess = $false
    $SetSubModule = $null

    $AdapterRegistryPath = GetAdapterRegistryConfigPath $AdapterName ""
    if ($null -ne $AdapterRegistryPath)
    {
        $SetSubModule = Set-ItemProperty -Path $AdapterRegistryPath -Name $SUBMODULE_REGISTRYVAR_NAME -Value $SubModuleValue -Type "DWORD" -PassThru -ErrorAction SilentlyContinue
    }

    if ($null -ne $SetSubModule -and -Not [string]::IsNullOrEmpty($SetSubModule.SubModule))
    {
        $bSuccess = $true
    }

    return $bSuccess
}

Function CreateDriverLogMetadataFile($AdapterName)
{
    $strDriverName = ($script:SupportedAdapters.Where({ $_.Name -eq $AdapterName })).Service

    $strOSSpecificDriverName = (GetOSSpecificDriverName $strDriverName)
    if (-Not $strOSSpecificDriverName)
    {
        $script:ErrorMessagesStart += $Messages.NoCmdletSupport -f $AdapterName
        return $false
    }

    $DriverFolder = GetDriverInfParentFolder $strOSSpecificDriverName "Start"
    $DriverMetaData = Get-Item "$DriverFolder\$strOSSpecificDriverName.sys" -ErrorAction SilentlyContinue

    if ($DriverMetaData)
    {
        $DriverVersionInfo = $DriverMetaData.VersionInfo
        $DriverLength = "$strOSSpecificDriverName.sys Length: " + $DriverMetaData.Length
        $DriverChecksum = Get-FileHash $DriverMetaData

        try
        {
            Set-Content $LogMetadataPath -Value $DriverVersionInfo -ErrorAction Stop
            Add-Content $LogMetadataPath -Value $DriverLength -ErrorAction Stop
            Add-Content $LogMetadataPath -Value $DriverChecksum -ErrorAction Stop
        }
        catch
        {
            $script:ErrorMessagesStart += $Messages.PathIncorrect
        }
    }
    else
    {
        $script:ErrorMessagesStart += $Messages.NoCmdletSupport -f $AdapterName
        return $false
    }

    return $true
}

Function ValidateMethodOutput($MethodOutput, $Verb)
{
    Write-Verbose -Message "MethodOutput: $MethodOutput verb: $Verb"

    do
    {
        $Result = $true
        if ($null -eq $MethodOutput)
        {
            switch ($Verb)
            {
                Get {$script:WarningMessagesGet += $Messages.NoCmdletSupport -f $a; break}
                Set {$script:WarningMessagesSet += $Messages.NoCmdletSupport -f $a; break}
                Start {$script:WarningMessagesStart += $Messages.NoCmdletSupport -f $AdapterName; break}
                Stop {$script:WarningMessagesStop += $Messages.NoCmdletSupport -f $a; break}
                Disable {$script:WarningMessagesDisable += $Messages.NoCmdletSupport -f $a; break}
                Reset {$script:WarningMessagesReset += $Messages.NoCmdletSupport -f $a; break}
            }

            $Result = $false
            break
        }

        if ($MethodOutput.OutStatus -ne 0)
        {
            switch ($Verb)
            {
                Get {$script:ErrorMessagesGet += $Messages.OperationFailed -f $a, $Verb; break}
                Set {$script:ErrorMessagesSet += $Messages.OperationFailed -f $a, $Verb; break}
                Start {$script:ErrorMessagesStart += $Messages.OperationFailed -f $AdapterName, $Verb; break}
                Stop {$script:ErrorMessagesStop += $Messages.OperationFailed -f $a, $Verb; break}
                Disable {$script:ErrorMessagesDisable += $Messages.OperationFailed -f $a, $Verb; break}
                Reset {$script:ErrorMessagesReset += $Messages.OperationFailed -f $a, $Verb; break}
            }

            $Result = $false
        }
    } while ($false)

    Return $Result
}

Function ValidateGetModuleParam([ref]$DisplayModules, [ref]$DisplayLevels, $ModuleNames, $MethodOutput, $AdapterName)
{
    if ($ModuleNames)
    {
        foreach ($ModuleName in $ModuleNames)
        {
            $ModuleIds = $null
            $ModuleIDNames = $IntelLogConfigModules.Keys -Like $ModuleName
            $ModuleIDCount = $ModuleIDNames.Count

            if ($ModuleIDCount -gt 0)
            {
                $ModuleIDNames = $ModuleIDNames.Where({ $_ -NotLike $DRIVER_MODULE_NAME })

                if ($ModuleIDNames -and $ModuleIDNames.Count -gt 0)
                {
                    $ModuleIds = $IntelLogConfigModules[$ModuleIDNames]

                    foreach ($ModuleId in $ModuleIds)
                    {
                        for ($i = 0; $i -lt $MethodOutput.ModuleID.count; $i++)
                        {
                            if ($ModuleId -eq $MethodOutput.ModuleID[$i])
                            {
                                $DisplayModules.Value += GetModuleName($ModuleId)
                                $DisplayLevels.Value += GetLevelName($MethodOutput.LogLevel[$i])
                                break
                            }
                        }
                    }
                }
            }
            else
            {
                #if user passes in an array of modules and one (or more) is spelled incorrectly,
                #show error msg for that 1 and continue on getting the modules for the rest
                $script:ErrorMessagesGet += $Messages.InvalidModule -f $AdapterName, $ModuleName
            }
        }
    }
    else
    {
        GetAllModulesLevel $MethodOutput $DisplayModules $DisplayLevels
    }
}

Function GetAllModulesLevel($MethodOutput, [ref]$DisplayModules, [ref]$DisplayLevels)
{
    if ($MethodOutput)
    {
        for ($i = 0; $i -lt $MethodOutput.ModuleID.count; $i++)
        {
            # driver returns an array of 64 items but only a few of them are populated:
            # 31 modules on CVL/CNV, 22 on LKV
            # Since we don't know how many modules are populated, we break when we hit an empty one
            # but account for the case where the first module is 'General == 0'
            if ($i -gt 0 -and $MethodOutput.ModuleID[$i] -eq 0)
            {
                break
            }
            $DisplayModules.Value += GetModuleName($MethodOutput.ModuleID[$i])
            $DisplayLevels.Value += GetLevelName($MethodOutput.LogLevel[$i])
        }
    }
}

Function ValidateLevelParam([ref]$InputLevelId, [ref]$DisplayLevels, $LevelName, $Count, $AdapterName, $DriverPassed)
{
    do
    {
        $NewLevelId = $null
        $Result = $true

        $LevelMatchArray = $IntelLogConfigLevels.GetEnumerator().Where({$_.Name -like $LevelName})
        if ($LevelMatchArray)
        {
            $NewLevelId = @($LevelMatchArray.Value)
        }
        # Handle default case of "Normal" if Level is not passed
        elseif ($DriverPassed -and -not $LevelName)
        {
            $NewLevelId = @($IntelLogConfigLevels["Normal"])
        }

        if ($null -eq $NewLevelId -or 0 -eq $NewLevelId.Count)
        {
            $script:ErrorMessagesSet += $Messages.InvalidLevel -f $AdapterName, $LevelName, $ModuleNames[0]
            $Result = $false
            break
        }

        if ($NewLevelId.count -gt 1)
        {
            $script:ErrorMessagesSet += $Messages.InvalidLevelWildcard
            $Result = $false
            break
        }

        for ($i = 0; $i -lt $Count; $i++)
        {
            $InputLevelId.Value[$i] = $NewLevelId[0]
            $DisplayLevels.Value += GetLevelName($NewLevelId[0])
        }

    } while ($false)

    Return $Result
}

Function ValidateSetModuleParam([ref]$InputModulesId, [ref]$DisplayModules, $ModuleNames, $AdapterName)
{
    $count = 0
    foreach ($ModuleName in $ModuleNames)
    {
        $ModuleIds = $null
        $ModuleIDNames = $IntelLogConfigModules.Keys -Like $ModuleName
        $ModuleIDCount = $ModuleIDNames.Count

        if ($ModuleIDCount -gt 0)
        {
            $ModuleIDNames = $ModuleIDNames.Where({ $_ -NotLike $DRIVER_MODULE_NAME })
            if (-Not $ModuleIDNames -or $ModuleIDCount -gt $ModuleIDNames.Count)
            {
                $DisplayModules.Value += $DRIVER_MODULE_NAME
            }

            if ($ModuleIDNames -and $ModuleIDNames.Count -gt 0)
            {
                $ModuleIds = $IntelLogConfigModules[$ModuleIDNames]

                foreach ($ModuleId in $ModuleIds)
                {
                    $InputModulesId.Value[$count] = $ModuleId
                    $DisplayModules.Value += GetModuleName($ModuleId)
                    $count++
                }
            }
        }
        else
        {
            #if user passes in an array of modules and one (or more) is spelled incorrectly,
            #show error msg for that 1 and continue on getting the modules for the rest
            $script:ErrorMessagesSet += $Messages.InvalidModule -f $AdapterName, $ModuleName
        }
    }
}

Function LogmanProcessRunning($AdapterName, $LogName, $GUID)
{
    $Result = $false

    $LogmanQueryString = logman query $LogName -ets
    if ($LogmanQueryString -match ($GUID))
    {
        $script:ErrorMessagesStart += $Messages.LogmanRunning -f $AdapterName
        $Result = $true
    }

    return $Result
}

Function LogmanStart($LogmanPath, $LogName, $GUID, $SubModuleSum, $Level)
{
    if ($Append)
    {
        $null = logman start $LogName -ets -o $LogmanPath -p $GUID $SubModuleSum $Level -a
    }
    else
    {
        $null = logman start $LogName -ets -o $LogmanPath -p $GUID $SubModuleSum $Level
    }
}

Function LogmanStop($LogName)
{
    $null = logman stop $LogName -ets
}


Function ValidateLogmanStart($AdapterName)
{
    $Result = $true
    # -2147024629, -2147024773, -2147024893, -2147024891 error codes mean directory name is invalid
    # or filename, directory name, or volume label syntax is incorrect or permission is denied.
    if ($LASTEXITCODE -eq 0)
    {
        $Result = $true
    }
    elseif ($LASTEXITCODE -eq -2147024629 -or $LASTEXITCODE -eq -2147024773 -or $LASTEXITCODE -eq -2147024893 -or $LASTEXITCODE -eq -2147024891)
    {
        $script:ErrorMessagesStart += $Messages.PathIncorrect
        $Result = $false
    }
    else
    {
        $script:ErrorMessagesStart += $Messages.OperationFailed -f $AdapterName, "Start"
        $Result = $false
    }
    return $Result
}

Function ValidateLogmanStop($AdapterName, [ref]$ErrorMessages)
{
    $Result = $true

    if ($LASTEXITCODE -eq 0)
    {
        $Result = $true
    }
    elseif ($LASTEXITCODE -eq -2144337918)
    {
        $ErrorMessages.Value += $Messages.NoLogsStarted -f $AdapterName
        $Result = $false
    }
    else
    {
        $ErrorMessages.Value += $Messages.OperationFailed -f $AdapterName, "Stop"
        $Result = $false
    }
    Return $Result
}

Function ValidateConfigurationParam($ConfigurationNames, $ModuleNames, [ref]$DisplayConfigNames, $Verb, $a)
{
    $Result = $true

    do
    {
        if ($ConfigurationNames)
        {
            if ($ModuleNames)
            {
                switch ($Verb)
                {
                    Set { $script:ErrorMessagesSet += $Messages.InvalidParamsConfigAndModule; break }
                    Get { $script:ErrorMessagesGet += $Messages.InvalidParamsConfigAndModule; break }
                }
                $Result = $false
                break
            }

            if ($IntelLogConfigurations.Count -eq 0)
            {
                if ($Verb -eq "Set")
                {
                    $script:ErrorMessagesSet += $Messages.NoConfigurationSupport -f $a
                }
                elseif ($Verb -eq "Get")
                {
                    $script:WarningMessagesGet += $Messages.NoConfigurationSupport -f $a
                }

                $Result = $false
                break
            }

            foreach ($ConfigName in $ConfigurationNames)
            {
                $TmpDisplayConfigNames = $IntelLogConfigurations.Keys -Like $ConfigName

                if ($TmpDisplayConfigNames.Count -gt 0)
                {
                    $DisplayConfigNames.Value += $TmpDisplayConfigNames
                }
                else
                {
                    #if user passes in an array of configs and one (or more) is spelled incorrectly,
                    #show error msg for that 1 and continue on getting the config for the rest
                    switch ($Verb)
                    {
                        Set { $script:ErrorMessagesSet += $Messages.InvalidConfigurationName -f $a, [string]$ConfigName; break }
                        Get { $script:ErrorMessagesGet += $Messages.InvalidConfigurationName -f $a, [string]$ConfigName; break }
                    }

                    $Result = $false
                }
            }
        }
    } while ($false)

    return $Result
}

Function GetConfigLogs($DisplayConfigNames, [ref]$DisplayModules, [ref]$DisplayLevels, $MethodOutput)
{
    $ConfigurationID = $IntelLogConfigurations[$DisplayConfigNames]
    foreach ($ConfigID in $ConfigurationID)
    {
        $params = @{ConfigurationID = [uint32]$ConfigID}
        $GetConfigMethodOutput = InvokeCimMethod "IntlLan_SetGenData" $a "WmiGetFwLogLevelsForConfiguration" $params

        for ($i = 0; $i -lt $GetConfigMethodOutput.count; $i++)
        {
            $DisplayModules.Value += GetModuleName($MethodOutput.ModuleID[$GetConfigMethodOutput.ModuleID[$i]])
            $DisplayLevels.Value += GetLevelName($MethodOutput.LogLevel[$GetConfigMethodOutput.ModuleID[$i]])
        }
    }
}

function GetConfigHashTable($AdapterName, $ConfigPath)
{
    $ConfigHashTable = @{}

    $FWLogConfigPath = GetAdapterRegistryConfigPath $AdapterName $ConfigPath
    if ($null -ne $FWLogConfigPath)
    {
        $LogConfigs = Get-Item $FWLogConfigPath -ErrorAction SilentlyContinue
        if ($null -ne $LogConfigs)
        {
            $LogConfigs = $LogConfigs.Property
            foreach ($Config in $LogConfigs)
            {
                $ConfigHashTable.add($Config, (Get-ItemProperty -Path $FWLogConfigPath -Name $Config -ErrorAction SilentlyContinue).$Config)
            }
        }
    }

    return $ConfigHashTable
}

function GetAdapterRegistryConfigPath($AdapterName, $Path)
{
    $ConfigPath = $null
    $AdapterObject = $script:MSNetAdapters.Where({$_.ifDesc -eq $AdapterName})
    $972Key = @(Get-ItemProperty -Path "HKLM:\SYSTEM\CurrentControlSet\Control\Class\{4d36e972-e325-11ce-bfc1-08002be10318}\*" -Name "NetCfgInstanceId" -ErrorAction SilentlyContinue)
    $Adapter972Key = $972Key.Where({$_.NetCfgInstanceId -eq $AdapterObject.InterfaceGuid})
    if ($Adapter972Key)
    {
        $ConfigPath = $Adapter972Key.PSPath + $Path
    }

    return $ConfigPath
}

function GetItemPropertyValue($AdapterName, $Path, $PropertyName)
{
    $PropertyItem = $null

    $FWLogConfigPath = GetAdapterRegistryConfigPath $AdapterName $Path
    if ($null -ne $FWLogConfigPath)
    {
        $PropertyItem = Get-ItemProperty -Path $FWLogConfigPath -Name $PropertyName -ErrorAction SilentlyContinue
    }

    $PropertyValue = $null
    if ($null -ne $PropertyItem)
    {
        $PropertyValue = $PropertyItem.$PropertyName
    }

    return $PropertyValue
}

# SIG # Begin signature block
# MIIt8QYJKoZIhvcNAQcCoIIt4jCCLd4CAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCCwCZRJ+YDG5d/6
# mSFlRiZ4orlFrJH7Y1RpoOXCpzFCNaCCEfMwggVvMIIEV6ADAgECAhBI/JO0YFWU
# jTanyYqJ1pQWMA0GCSqGSIb3DQEBDAUAMHsxCzAJBgNVBAYTAkdCMRswGQYDVQQI
# DBJHcmVhdGVyIE1hbmNoZXN0ZXIxEDAOBgNVBAcMB1NhbGZvcmQxGjAYBgNVBAoM
# EUNvbW9kbyBDQSBMaW1pdGVkMSEwHwYDVQQDDBhBQUEgQ2VydGlmaWNhdGUgU2Vy
# dmljZXMwHhcNMjEwNTI1MDAwMDAwWhcNMjgxMjMxMjM1OTU5WjBWMQswCQYDVQQG
# EwJHQjEYMBYGA1UEChMPU2VjdGlnbyBMaW1pdGVkMS0wKwYDVQQDEyRTZWN0aWdv
# IFB1YmxpYyBDb2RlIFNpZ25pbmcgUm9vdCBSNDYwggIiMA0GCSqGSIb3DQEBAQUA
# A4ICDwAwggIKAoICAQCN55QSIgQkdC7/FiMCkoq2rjaFrEfUI5ErPtx94jGgUW+s
# hJHjUoq14pbe0IdjJImK/+8Skzt9u7aKvb0Ffyeba2XTpQxpsbxJOZrxbW6q5KCD
# J9qaDStQ6Utbs7hkNqR+Sj2pcaths3OzPAsM79szV+W+NDfjlxtd/R8SPYIDdub7
# P2bSlDFp+m2zNKzBenjcklDyZMeqLQSrw2rq4C+np9xu1+j/2iGrQL+57g2extme
# me/G3h+pDHazJyCh1rr9gOcB0u/rgimVcI3/uxXP/tEPNqIuTzKQdEZrRzUTdwUz
# T2MuuC3hv2WnBGsY2HH6zAjybYmZELGt2z4s5KoYsMYHAXVn3m3pY2MeNn9pib6q
# RT5uWl+PoVvLnTCGMOgDs0DGDQ84zWeoU4j6uDBl+m/H5x2xg3RpPqzEaDux5mcz
# mrYI4IAFSEDu9oJkRqj1c7AGlfJsZZ+/VVscnFcax3hGfHCqlBuCF6yH6bbJDoEc
# QNYWFyn8XJwYK+pF9e+91WdPKF4F7pBMeufG9ND8+s0+MkYTIDaKBOq3qgdGnA2T
# OglmmVhcKaO5DKYwODzQRjY1fJy67sPV+Qp2+n4FG0DKkjXp1XrRtX8ArqmQqsV/
# AZwQsRb8zG4Y3G9i/qZQp7h7uJ0VP/4gDHXIIloTlRmQAOka1cKG8eOO7F/05QID
# AQABo4IBEjCCAQ4wHwYDVR0jBBgwFoAUoBEKIz6W8Qfs4q8p74Klf9AwpLQwHQYD
# VR0OBBYEFDLrkpr/NZZILyhAQnAgNpFcF4XmMA4GA1UdDwEB/wQEAwIBhjAPBgNV
# HRMBAf8EBTADAQH/MBMGA1UdJQQMMAoGCCsGAQUFBwMDMBsGA1UdIAQUMBIwBgYE
# VR0gADAIBgZngQwBBAEwQwYDVR0fBDwwOjA4oDagNIYyaHR0cDovL2NybC5jb21v
# ZG9jYS5jb20vQUFBQ2VydGlmaWNhdGVTZXJ2aWNlcy5jcmwwNAYIKwYBBQUHAQEE
# KDAmMCQGCCsGAQUFBzABhhhodHRwOi8vb2NzcC5jb21vZG9jYS5jb20wDQYJKoZI
# hvcNAQEMBQADggEBABK/oe+LdJqYRLhpRrWrJAoMpIpnuDqBv0WKfVIHqI0fTiGF
# OaNrXi0ghr8QuK55O1PNtPvYRL4G2VxjZ9RAFodEhnIq1jIV9RKDwvnhXRFAZ/ZC
# J3LFI+ICOBpMIOLbAffNRk8monxmwFE2tokCVMf8WPtsAO7+mKYulaEMUykfb9gZ
# pk+e96wJ6l2CxouvgKe9gUhShDHaMuwV5KZMPWw5c9QLhTkg4IUaaOGnSDip0TYl
# d8GNGRbFiExmfS9jzpjoad+sPKhdnckcW67Y8y90z7h+9teDnRGWYpquRRPaf9xH
# +9/DUp/mBlXpnYzyOmJRvOwkDynUWICE5EV7WtgwggYcMIIEBKADAgECAhAz1wio
# kUBTGeKlu9M5ua1uMA0GCSqGSIb3DQEBDAUAMFYxCzAJBgNVBAYTAkdCMRgwFgYD
# VQQKEw9TZWN0aWdvIExpbWl0ZWQxLTArBgNVBAMTJFNlY3RpZ28gUHVibGljIENv
# ZGUgU2lnbmluZyBSb290IFI0NjAeFw0yMTAzMjIwMDAwMDBaFw0zNjAzMjEyMzU5
# NTlaMFcxCzAJBgNVBAYTAkdCMRgwFgYDVQQKEw9TZWN0aWdvIExpbWl0ZWQxLjAs
# BgNVBAMTJVNlY3RpZ28gUHVibGljIENvZGUgU2lnbmluZyBDQSBFViBSMzYwggGi
# MA0GCSqGSIb3DQEBAQUAA4IBjwAwggGKAoIBgQC70f4et0JbePWQp64sg/GNIdMw
# hoV739PN2RZLrIXFuwHP4owoEXIEdiyBxasSekBKxRDogRQ5G19PB/YwMDB/NSXl
# wHM9QAmU6Kj46zkLVdW2DIseJ/jePiLBv+9l7nPuZd0o3bsffZsyf7eZVReqskmo
# PBBqOsMhspmoQ9c7gqgZYbU+alpduLyeE9AKnvVbj2k4aOqlH1vKI+4L7bzQHkND
# brBTjMJzKkQxbr6PuMYC9ruCBBV5DFIg6JgncWHvL+T4AvszWbX0w1Xn3/YIIq62
# 0QlZ7AGfc4m3Q0/V8tm9VlkJ3bcX9sR0gLqHRqwG29sEDdVOuu6MCTQZlRvmcBME
# Jd+PuNeEM4xspgzraLqVT3xE6NRpjSV5wyHxNXf4T7YSVZXQVugYAtXueciGoWnx
# G06UE2oHYvDQa5mll1CeHDOhHu5hiwVoHI717iaQg9b+cYWnmvINFD42tRKtd3V6
# zOdGNmqQU8vGlHHeBzoh+dYyZ+CcblSGoGSgg8sCAwEAAaOCAWMwggFfMB8GA1Ud
# IwQYMBaAFDLrkpr/NZZILyhAQnAgNpFcF4XmMB0GA1UdDgQWBBSBMpJBKyjNRsjE
# osYqORLsSKk/FDAOBgNVHQ8BAf8EBAMCAYYwEgYDVR0TAQH/BAgwBgEB/wIBADAT
# BgNVHSUEDDAKBggrBgEFBQcDAzAaBgNVHSAEEzARMAYGBFUdIAAwBwYFZ4EMAQMw
# SwYDVR0fBEQwQjBAoD6gPIY6aHR0cDovL2NybC5zZWN0aWdvLmNvbS9TZWN0aWdv
# UHVibGljQ29kZVNpZ25pbmdSb290UjQ2LmNybDB7BggrBgEFBQcBAQRvMG0wRgYI
# KwYBBQUHMAKGOmh0dHA6Ly9jcnQuc2VjdGlnby5jb20vU2VjdGlnb1B1YmxpY0Nv
# ZGVTaWduaW5nUm9vdFI0Ni5wN2MwIwYIKwYBBQUHMAGGF2h0dHA6Ly9vY3NwLnNl
# Y3RpZ28uY29tMA0GCSqGSIb3DQEBDAUAA4ICAQBfNqz7+fZyWhS38Asd3tj9lwHS
# /QHumS2G6Pa38Dn/1oFKWqdCSgotFZ3mlP3FaUqy10vxFhJM9r6QZmWLLXTUqwj3
# ahEDCHd8vmnhsNufJIkD1t5cpOCy1rTP4zjVuW3MJ9bOZBHoEHJ20/ng6SyJ6UnT
# s5eWBgrh9grIQZqRXYHYNneYyoBBl6j4kT9jn6rNVFRLgOr1F2bTlHH9nv1HMePp
# GoYd074g0j+xUl+yk72MlQmYco+VAfSYQ6VK+xQmqp02v3Kw/Ny9hA3s7TSoXpUr
# OBZjBXXZ9jEuFWvilLIq0nQ1tZiao/74Ky+2F0snbFrmuXZe2obdq2TWauqDGIgb
# MYL1iLOUJcAhLwhpAuNMu0wqETDrgXkG4UGVKtQg9guT5Hx2DJ0dJmtfhAH2KpnN
# r97H8OQYok6bLyoMZqaSdSa+2UA1E2+upjcaeuitHFFjBypWBmztfhj24+xkc6Zt
# CDaLrw+ZrnVrFyvCTWrDUUZBVumPwo3/E3Gb2u2e05+r5UWmEsUUWlJBl6MGAAjF
# 5hzqJ4I8O9vmRsTvLQA1E802fZ3lqicIBczOwDYOSxlP0GOabb/FKVMxItt1UHeG
# 0PL4au5rBhs+hSMrl8h+eplBDN1Yfw6owxI9OjWb4J0sjBeBVESoeh2YnZZ/WVim
# VGX/UUIL+Efrz/jlvzCCBlwwggTEoAMCAQICEQC0WMhOLa9BaZ9kSX5iJ3F/MA0G
# CSqGSIb3DQEBCwUAMFcxCzAJBgNVBAYTAkdCMRgwFgYDVQQKEw9TZWN0aWdvIExp
# bWl0ZWQxLjAsBgNVBAMTJVNlY3RpZ28gUHVibGljIENvZGUgU2lnbmluZyBDQSBF
# ViBSMzYwHhcNMjQwMjI2MDAwMDAwWhcNMjUwMjI1MjM1OTU5WjCBuzEQMA4GA1UE
# BRMHMjE4OTA3NDETMBEGCysGAQQBgjc8AgEDEwJVUzEZMBcGCysGAQQBgjc8AgEC
# EwhEZWxhd2FyZTEdMBsGA1UEDxMUUHJpdmF0ZSBPcmdhbml6YXRpb24xCzAJBgNV
# BAYTAlVTMRMwEQYDVQQIDApDYWxpZm9ybmlhMRowGAYDVQQKDBFJbnRlbCBDb3Jw
# b3JhdGlvbjEaMBgGA1UEAwwRSW50ZWwgQ29ycG9yYXRpb24wggGiMA0GCSqGSIb3
# DQEBAQUAA4IBjwAwggGKAoIBgQDBCfpjptqBxrQLJGyUHE47EvbngKTbZ0xMZoUj
# CJVmRhCCzWtZeKwlwhuI3bJyq4sSeejZxY7IMjroOoditsPm5xYohctw0UO+j1Th
# L71qce9bigWpDFDBBqksK5+011j/XPA+kRu/gJBolI50N8tIHHsH31NzD09/sN7U
# V242zTBy0TnMwanTXLMux/kVJbIloWSHRn0wIZmGuWESmWDrsLQEtSIo4zyUlzvQ
# UmJrtHMmJc3Rw/5TE7rC9Zq4Yt6s+BNu8i5howcK7yEOtiw/sKIlbACFJqpp6EUT
# Kwi7RRLKkuoL7G/+50XrJlCQqDbYxQAm7Tc2oFBVZW9xf4gUz3f48iflabLvDmc0
# pVWgDF0OmX+SzsHf94GYG3slCw8JJKfU66TfJEModuiDPwfgA6ripNWdBHqaDoY7
# JQPt6T6wierKjp64ABBHwyYSD55RIMUm/w33oe0i44tAlvUTkujJzwUQKpjXQ9av
# FyA2VqPea77rc3yiCRNeGQTpyO0CAwEAAaOCAbwwggG4MB8GA1UdIwQYMBaAFIEy
# kkErKM1GyMSixio5EuxIqT8UMB0GA1UdDgQWBBSC0NSIL647v94GegQBXPynnV+p
# cDAOBgNVHQ8BAf8EBAMCB4AwDAYDVR0TAQH/BAIwADATBgNVHSUEDDAKBggrBgEF
# BQcDAzBJBgNVHSAEQjBAMDUGDCsGAQQBsjEBAgEGATAlMCMGCCsGAQUFBwIBFhdo
# dHRwczovL3NlY3RpZ28uY29tL0NQUzAHBgVngQwBAzBLBgNVHR8ERDBCMECgPqA8
# hjpodHRwOi8vY3JsLnNlY3RpZ28uY29tL1NlY3RpZ29QdWJsaWNDb2RlU2lnbmlu
# Z0NBRVZSMzYuY3JsMHsGCCsGAQUFBwEBBG8wbTBGBggrBgEFBQcwAoY6aHR0cDov
# L2NydC5zZWN0aWdvLmNvbS9TZWN0aWdvUHVibGljQ29kZVNpZ25pbmdDQUVWUjM2
# LmNydDAjBggrBgEFBQcwAYYXaHR0cDovL29jc3Auc2VjdGlnby5jb20wLgYDVR0R
# BCcwJaAjBggrBgEFBQcIA6AXMBUME1VTLURFTEFXQVJFLTIxODkwNzQwDQYJKoZI
# hvcNAQELBQADggGBADRT5U3ne/vFqaxPbuhkYXvhfhBMHNi+fHEkOCjPhEyqOkeU
# 1a7bjucUQh+Jb6yOlEKzb8KppbyTDNZzlo5NpkpBtFimyPY/h7pboom7mDOFXS9/
# NUuESrc/niHOTRkTzipQa125g5y/hgZ8a6+XoMjKi/rWOvllhdyUiJi6KY9x5+IN
# nXSgYZu/7xnGge/UybwaVrCVqmirark7p8I3vPOmIQeeGupn7qyzFdiMK5EEpPUI
# uO4po7YGOTQDgpdPjUQGmmGqbkrGgvH2fT2W/Ti8IZSgBM+3i3Rtqo50gOTOe9py
# fG30f9aFUtFHFc9BAA3kvG+Xqr4MLOdFYgQRGFXNjN5IA6zc0admMuG8m/hVasJN
# p+ACnv8HeWID2O6oTGPhwHZkvfgqL05qEO6ZiThnzwWDukiduuceeYxIVqyYW253
# hzgZCKnjWVdDT3gUWoO2TJNR7sZuP/gP7I2hyotU8uRTl3SvlFfbaVGHj+xVqR1k
# taptv3zLnJYUhbTyNjGCG1QwghtQAgEBMGwwVzELMAkGA1UEBhMCR0IxGDAWBgNV
# BAoTD1NlY3RpZ28gTGltaXRlZDEuMCwGA1UEAxMlU2VjdGlnbyBQdWJsaWMgQ29k
# ZSBTaWduaW5nIENBIEVWIFIzNgIRALRYyE4tr0Fpn2RJfmIncX8wDQYJYIZIAWUD
# BAIBBQCgajAZBgkqhkiG9w0BCQMxDAYKKwYBBAGCNwIBBDAcBgorBgEEAYI3AgEL
# MQ4wDAYKKwYBBAGCNwIBFjAvBgkqhkiG9w0BCQQxIgQgoCe+FVJjVPVUEmbs3A6l
# rzN4VNWSuyexDjX1MGoj9ZIwDQYJKoZIhvcNAQEBBQAEggGAsomLBbf7Vi3LL3e9
# AP2LabMXXApEc05np8hsJ8MC9vgnS+f80SEzBCoZw/UWLdJJQ7GM981ubWvq3Bpz
# 27AUG3q0P0NUvt8krtsUPjthP80N4R0Xg1r+xN8+MIDRq+ZueIFpp3G1FA4Vy1Wt
# 5g6hh0N6lOjOq+Jlfa6Nt/ViDgi6K0xtJ/j66p2Q+kTaq5E2lDYwpqXAXGuPvZIL
# Z7fKewUXJGZ7BdaLP1yoWcQ3v3sZEo0ioNZhRoXxjI5RZK5wJ4vD6JKNcUHb8LSc
# zOhWA126QMPwNifF3qmDoyiseIarw9lVxZ+Cad7/MTNUJ8iIW6ciIy9NpoEDoXAw
# phy95Kh8LdEVujBb6R03lTzzD3XL6+y7gs8PSiQXCSPt0sPgnTbxwnJrh3GEPsSe
# WSCnWSwLasqEOLJEfND4eOCX1a7bk1UkVFtCN0JGPskpzuLRiwxRdfjWkZUsnG5b
# GIa5v7gQtTc7yfSg9sE1gXJUxrXAtoTVo8S7dLyMDfFL8379oYIYzTCCGMkGCisG
# AQQBgjcDAwExghi5MIIYtQYJKoZIhvcNAQcCoIIYpjCCGKICAQMxDzANBglghkgB
# ZQMEAgIFADCB8wYLKoZIhvcNAQkQAQSggeMEgeAwgd0CAQEGCisGAQQBsjECAQEw
# MTANBglghkgBZQMEAgEFAAQgD2E6kGzNU9TTUQ5sOFknvM1yMgKirWE7DSa7hYs9
# aRACFEuKayVO2WUPcPtnzgm1P/6w0BURGA8yMDI0MTIwNDE4MzA0N1qgcqRwMG4x
# CzAJBgNVBAYTAkdCMRMwEQYDVQQIEwpNYW5jaGVzdGVyMRgwFgYDVQQKEw9TZWN0
# aWdvIExpbWl0ZWQxMDAuBgNVBAMTJ1NlY3RpZ28gUHVibGljIFRpbWUgU3RhbXBp
# bmcgU2lnbmVyIFIzNaCCEv8wggZdMIIExaADAgECAhA6UmoshM5V5h1l/MwS2OmJ
# MA0GCSqGSIb3DQEBDAUAMFUxCzAJBgNVBAYTAkdCMRgwFgYDVQQKEw9TZWN0aWdv
# IExpbWl0ZWQxLDAqBgNVBAMTI1NlY3RpZ28gUHVibGljIFRpbWUgU3RhbXBpbmcg
# Q0EgUjM2MB4XDTI0MDExNTAwMDAwMFoXDTM1MDQxNDIzNTk1OVowbjELMAkGA1UE
# BhMCR0IxEzARBgNVBAgTCk1hbmNoZXN0ZXIxGDAWBgNVBAoTD1NlY3RpZ28gTGlt
# aXRlZDEwMC4GA1UEAxMnU2VjdGlnbyBQdWJsaWMgVGltZSBTdGFtcGluZyBTaWdu
# ZXIgUjM1MIICIjANBgkqhkiG9w0BAQEFAAOCAg8AMIICCgKCAgEAjdFn9MFIm739
# OEk6TWGBm8PY3EWlYQQ2jQae45iWgPXUGVuYoIa1xjTGIyuw3suUSBzKiyG0/c/Y
# n++d5mG6IyayljuGT9DeXQU9k8GWWj2/BPoamg2fFctnPsdTYhMGxM06z1+Ft0Ba
# v8ybww21ii/faiy+NhiUM195+cFqOtCpJXxZ/lm9tpjmVmEqpAlRpfGmLhNdkqiE
# uDFTuD1GsV3jvuPuPGKUJTam3P53U4LM0UCxeDI8Qz40Qw9TPar6S02XExlc8X1Y
# siE6ETcTz+g1ImQ1OqFwEaxsMj/WoJT18GG5KiNnS7n/X4iMwboAg3IjpcvEzw4A
# ZCZowHyCzYhnFRM4PuNMVHYcTXGgvuq9I7j4ke281x4e7/90Z5Wbk92RrLcS35hO
# 30TABcGx3Q8+YLRy6o0k1w4jRefCMT7b5mTxtq5XPmKvtgfPuaWPkGZ/tbxInyND
# A7YgOgccULjp4+D56g2iuzRCsLQ9ac6AN4yRbqCYsG2rcIQ5INTyI2JzA2w1vsAH
# PRbUTeqVLDuNOY2gYIoKBWQsPYVoyzaoBVU6O5TG+a1YyfWkgVVS9nXKs8hVti3V
# pOV3aeuaHnjgC6He2CCDL9aW6gteUe0AmC8XCtWwpePx6QW3ROZo8vSUe9AR7mMd
# u5+FzTmW8K13Bt8GX/YBFJO7LWzwKAUCAwEAAaOCAY4wggGKMB8GA1UdIwQYMBaA
# FF9Y7UwxeqJhQo1SgLqzYZcZojKbMB0GA1UdDgQWBBRo76QySWm2Ujgd6kM5LPQU
# ap4MhTAOBgNVHQ8BAf8EBAMCBsAwDAYDVR0TAQH/BAIwADAWBgNVHSUBAf8EDDAK
# BggrBgEFBQcDCDBKBgNVHSAEQzBBMDUGDCsGAQQBsjEBAgEDCDAlMCMGCCsGAQUF
# BwIBFhdodHRwczovL3NlY3RpZ28uY29tL0NQUzAIBgZngQwBBAIwSgYDVR0fBEMw
# QTA/oD2gO4Y5aHR0cDovL2NybC5zZWN0aWdvLmNvbS9TZWN0aWdvUHVibGljVGlt
# ZVN0YW1waW5nQ0FSMzYuY3JsMHoGCCsGAQUFBwEBBG4wbDBFBggrBgEFBQcwAoY5
# aHR0cDovL2NydC5zZWN0aWdvLmNvbS9TZWN0aWdvUHVibGljVGltZVN0YW1waW5n
# Q0FSMzYuY3J0MCMGCCsGAQUFBzABhhdodHRwOi8vb2NzcC5zZWN0aWdvLmNvbTAN
# BgkqhkiG9w0BAQwFAAOCAYEAsNwuyfpPNkyKL/bJT9XvGE8fnw7Gv/4SetmOkjK9
# hPPa7/Nsv5/MHuVus+aXwRFqM5Vu51qfrHTwnVExcP2EHKr7IR+m/Ub7PamaeWfl
# e5x8D0x/MsysICs00xtSNVxFywCvXx55l6Wg3lXiPCui8N4s51mXS0Ht85fkXo3a
# uZdo1O4lHzJLYX4RZovlVWD5EfwV6Ve1G9UMslnm6pI0hyR0Zr95QWG0MpNPP0u0
# 5SHjq/YkPlDee3yYOECNMqnZ+j8onoUtZ0oC8CkbOOk/AOoV4kp/6Ql2gEp3bNC7
# DOTlaCmH24DjpVgryn8FMklqEoK4Z3IoUgV8R9qQLg1dr6/BjghGnj2XNA8ujta2
# JyoxpqpvyETZCYIUjIs69YiDjzftt37rQVwIZsfCYv+DU5sh/StFL1x4rgNj2t8G
# ccUfa/V3iFFW9lfIJWWsvtlC5XOOOQswr1UmVdNWQem4LwrlLgcdO/YAnHqY52Qw
# nBLiAuUnuBeshWmfEb5oieIYMIIGFDCCA/ygAwIBAgIQeiOu2lNplg+RyD5c9Mfj
# PzANBgkqhkiG9w0BAQwFADBXMQswCQYDVQQGEwJHQjEYMBYGA1UEChMPU2VjdGln
# byBMaW1pdGVkMS4wLAYDVQQDEyVTZWN0aWdvIFB1YmxpYyBUaW1lIFN0YW1waW5n
# IFJvb3QgUjQ2MB4XDTIxMDMyMjAwMDAwMFoXDTM2MDMyMTIzNTk1OVowVTELMAkG
# A1UEBhMCR0IxGDAWBgNVBAoTD1NlY3RpZ28gTGltaXRlZDEsMCoGA1UEAxMjU2Vj
# dGlnbyBQdWJsaWMgVGltZSBTdGFtcGluZyBDQSBSMzYwggGiMA0GCSqGSIb3DQEB
# AQUAA4IBjwAwggGKAoIBgQDNmNhDQatugivs9jN+JjTkiYzT7yISgFQ+7yavjA6B
# g+OiIjPm/N/t3nC7wYUrUlY3mFyI32t2o6Ft3EtxJXCc5MmZQZ8AxCbh5c6WzeJD
# B9qkQVa46xiYEpc81KnBkAWgsaXnLURoYZzksHIzzCNxtIXnb9njZholGw9djnjk
# TdAA83abEOHQ4ujOGIaBhPXG2NdV8TNgFWZ9BojlAvflxNMCOwkCnzlH4oCw5+4v
# 1nssWeN1y4+RlaOywwRMUi54fr2vFsU5QPrgb6tSjvEUh1EC4M29YGy/SIYM8ZpH
# admVjbi3Pl8hJiTWw9jiCKv31pcAaeijS9fc6R7DgyyLIGflmdQMwrNRxCulVq8Z
# pysiSYNi79tw5RHWZUEhnRfs/hsp/fwkXsynu1jcsUX+HuG8FLa2BNheUPtOcgw+
# vHJcJ8HnJCrcUWhdFczf8O+pDiyGhVYX+bDDP3GhGS7TmKmGnbZ9N+MpEhWmbiAV
# PbgkqykSkzyYVr15OApZYK8CAwEAAaOCAVwwggFYMB8GA1UdIwQYMBaAFPZ3at0/
# /QET/xahbIICL9AKPRQlMB0GA1UdDgQWBBRfWO1MMXqiYUKNUoC6s2GXGaIymzAO
# BgNVHQ8BAf8EBAMCAYYwEgYDVR0TAQH/BAgwBgEB/wIBADATBgNVHSUEDDAKBggr
# BgEFBQcDCDARBgNVHSAECjAIMAYGBFUdIAAwTAYDVR0fBEUwQzBBoD+gPYY7aHR0
# cDovL2NybC5zZWN0aWdvLmNvbS9TZWN0aWdvUHVibGljVGltZVN0YW1waW5nUm9v
# dFI0Ni5jcmwwfAYIKwYBBQUHAQEEcDBuMEcGCCsGAQUFBzAChjtodHRwOi8vY3J0
# LnNlY3RpZ28uY29tL1NlY3RpZ29QdWJsaWNUaW1lU3RhbXBpbmdSb290UjQ2LnA3
# YzAjBggrBgEFBQcwAYYXaHR0cDovL29jc3Auc2VjdGlnby5jb20wDQYJKoZIhvcN
# AQEMBQADggIBABLXeyCtDjVYDJ6BHSVY/UwtZ3Svx2ImIfZVVGnGoUaGdltoX4hD
# skBMZx5NY5L6SCcwDMZhHOmbyMhyOVJDwm1yrKYqGDHWzpwVkFJ+996jKKAXyIIa
# Uf5JVKjccev3w16mNIUlNTkpJEor7edVJZiRJVCAmWAaHcw9zP0hY3gj+fWp8MbO
# ocI9Zn78xvm9XKGBp6rEs9sEiq/pwzvg2/KjXE2yWUQIkms6+yslCRqNXPjEnBnx
# uUB1fm6bPAV+Tsr/Qrd+mOCJemo06ldon4pJFbQd0TQVIMLv5koklInHvyaf6vAT
# JP4DfPtKzSBPkKlOtyaFTAjD2Nu+di5hErEVVaMqSVbfPzd6kNXOhYm23EWm6N2s
# 2ZHCHVhlUgHaC4ACMRCgXjYfQEDtYEK54dUwPJXV7icz0rgCzs9VI29DwsjVZFpO
# 4ZIVR33LwXyPDbYFkLqYmgHjR3tKVkhh9qKV2WCmBuC27pIOx6TYvyqiYbntinmp
# Oqh/QPAnhDgexKG9GX/n1PggkGi9HCapZp8fRwg8RftwS21Ln61euBG0yONM6noD
# 2XQPrFwpm3GcuqJMf0o8LLrFkSLRQNwxPDDkWXhW+gZswbaiie5fd/W2ygcto78X
# CSPfFWveUOSZ5SqK95tBO8aTHmEa4lpJVD7HrTEn9jb1EGvxOb1cnn0CMIIGgjCC
# BGqgAwIBAgIQNsKwvXwbOuejs902y8l1aDANBgkqhkiG9w0BAQwFADCBiDELMAkG
# A1UEBhMCVVMxEzARBgNVBAgTCk5ldyBKZXJzZXkxFDASBgNVBAcTC0plcnNleSBD
# aXR5MR4wHAYDVQQKExVUaGUgVVNFUlRSVVNUIE5ldHdvcmsxLjAsBgNVBAMTJVVT
# RVJUcnVzdCBSU0EgQ2VydGlmaWNhdGlvbiBBdXRob3JpdHkwHhcNMjEwMzIyMDAw
# MDAwWhcNMzgwMTE4MjM1OTU5WjBXMQswCQYDVQQGEwJHQjEYMBYGA1UEChMPU2Vj
# dGlnbyBMaW1pdGVkMS4wLAYDVQQDEyVTZWN0aWdvIFB1YmxpYyBUaW1lIFN0YW1w
# aW5nIFJvb3QgUjQ2MIICIjANBgkqhkiG9w0BAQEFAAOCAg8AMIICCgKCAgEAiJ3Y
# uUVnnR3d6LkmgZpUVMB8SQWbzFoVD9mUEES0QUCBdxSZqdTkdizICFNeINCSJS+l
# V1ipnW5ihkQyC0cRLWXUJzodqpnMRs46npiJPHrfLBOifjfhpdXJ2aHHsPHggGsC
# i7uE0awqKggE/LkYw3sqaBia67h/3awoqNvGqiFRJ+OTWYmUCO2GAXsePHi+/JUN
# Aax3kpqstbl3vcTdOGhtKShvZIvjwulRH87rbukNyHGWX5tNK/WABKf+Gnoi4cmi
# sS7oSimgHUI0Wn/4elNd40BFdSZ1EwpuddZ+Wr7+Dfo0lcHflm/FDDrOJ3rWqauU
# P8hsokDoI7D/yUVI9DAE/WK3Jl3C4LKwIpn1mNzMyptRwsXKrop06m7NUNHdlTDE
# MovXAIDGAvYynPt5lutv8lZeI5w3MOlCybAZDpK3Dy1MKo+6aEtE9vtiTMzz/o2d
# YfdP0KWZwZIXbYsTIlg1YIetCpi5s14qiXOpRsKqFKqav9R1R5vj3NgevsAsvxsA
# nI8Oa5s2oy25qhsoBIGo/zi6GpxFj+mOdh35Xn91y72J4RGOJEoqzEIbW3q0b2iP
# uWLA911cRxgY5SJYubvjay3nSMbBPPFsyl6mY4/WYucmyS9lo3l7jk27MAe145GW
# xK4O3m3gEFEIkv7kRmefDR7Oe2T1HxAnICQvr9sCAwEAAaOCARYwggESMB8GA1Ud
# IwQYMBaAFFN5v1qqK0rPVIDh2JvAnfKyA2bLMB0GA1UdDgQWBBT2d2rdP/0BE/8W
# oWyCAi/QCj0UJTAOBgNVHQ8BAf8EBAMCAYYwDwYDVR0TAQH/BAUwAwEB/zATBgNV
# HSUEDDAKBggrBgEFBQcDCDARBgNVHSAECjAIMAYGBFUdIAAwUAYDVR0fBEkwRzBF
# oEOgQYY/aHR0cDovL2NybC51c2VydHJ1c3QuY29tL1VTRVJUcnVzdFJTQUNlcnRp
# ZmljYXRpb25BdXRob3JpdHkuY3JsMDUGCCsGAQUFBwEBBCkwJzAlBggrBgEFBQcw
# AYYZaHR0cDovL29jc3AudXNlcnRydXN0LmNvbTANBgkqhkiG9w0BAQwFAAOCAgEA
# Dr5lQe1oRLjlocXUEYfktzsljOt+2sgXke3Y8UPEooU5y39rAARaAdAxUeiX1ktL
# J3+lgxtoLQhn5cFb3GF2SSZRX8ptQ6IvuD3wz/LNHKpQ5nX8hjsDLRhsyeIiJsms
# 9yAWnvdYOdEMq1W61KE9JlBkB20XBee6JaXx4UBErc+YuoSb1SxVf7nkNtUjPfcx
# uFtrQdRMRi/fInV/AobE8Gw/8yBMQKKaHt5eia8ybT8Y/Ffa6HAJyz9gvEOcF1VW
# XG8OMeM7Vy7Bs6mSIkYeYtddU1ux1dQLbEGur18ut97wgGwDiGinCwKPyFO7Apcm
# VJOtlw9FVJxw/mL1TbyBns4zOgkaXFnnfzg4qbSvnrwyj1NiurMp4pmAWjR+Pb/S
# IduPnmFzbSN/G8reZCL4fvGlvPFk4Uab/JVCSmj59+/mB2Gn6G/UYOy8k60mKcma
# AZsEVkhOFuoj4we8CYyaR9vd9PGZKSinaZIkvVjbH/3nlLb0a7SBIkiRzfPfS9T+
# JesylbHa1LtRV9U/7m0q7Ma2CQ/t392ioOssXW7oKLdOmMBl14suVFBmbzrt5V5c
# QPnwtd3UOTpS9oCG+ZZheiIvPgkDmA8FzPsnfXW5qHELB43ET7HHFHeRPRYrMBKj
# kb8/IN7Po0d0hQoF4TeMM+zYAJzoKQnVKOLg8pZVPT8xggSRMIIEjQIBATBpMFUx
# CzAJBgNVBAYTAkdCMRgwFgYDVQQKEw9TZWN0aWdvIExpbWl0ZWQxLDAqBgNVBAMT
# I1NlY3RpZ28gUHVibGljIFRpbWUgU3RhbXBpbmcgQ0EgUjM2AhA6UmoshM5V5h1l
# /MwS2OmJMA0GCWCGSAFlAwQCAgUAoIIB+TAaBgkqhkiG9w0BCQMxDQYLKoZIhvcN
# AQkQAQQwHAYJKoZIhvcNAQkFMQ8XDTI0MTIwNDE4MzA0N1owPwYJKoZIhvcNAQkE
# MTIEMHFv+prEFGBW2lc2kRUeh4bwESDY3LUVNP302COZnXABvpS3ybmMTk/AsAuX
# Is0RyjCCAXoGCyqGSIb3DQEJEAIMMYIBaTCCAWUwggFhMBYEFPhgmBmm+4gs9+hS
# l/KhGVIaFndfMIGHBBTGrlTkeIbxfD1VEkiMacNKevnC3TBvMFukWTBXMQswCQYD
# VQQGEwJHQjEYMBYGA1UEChMPU2VjdGlnbyBMaW1pdGVkMS4wLAYDVQQDEyVTZWN0
# aWdvIFB1YmxpYyBUaW1lIFN0YW1waW5nIFJvb3QgUjQ2AhB6I67aU2mWD5HIPlz0
# x+M/MIG8BBSFPWMtk4KCYXzQkDXEkd6SwULaxzCBozCBjqSBizCBiDELMAkGA1UE
# BhMCVVMxEzARBgNVBAgTCk5ldyBKZXJzZXkxFDASBgNVBAcTC0plcnNleSBDaXR5
# MR4wHAYDVQQKExVUaGUgVVNFUlRSVVNUIE5ldHdvcmsxLjAsBgNVBAMTJVVTRVJU
# cnVzdCBSU0EgQ2VydGlmaWNhdGlvbiBBdXRob3JpdHkCEDbCsL18Gzrno7PdNsvJ
# dWgwDQYJKoZIhvcNAQEBBQAEggIAKj5APxvDdk9XEUo80K956IhiAfQVBPHh+tWN
# aVx9sluGETB1G3crkA15wGpQSa1MrwN/d3oECJyR2m80PzTGgtQeCI+EA/Ax+MJA
# gYoZ17x8KcZXi6uJnXlVv1FYtOQecAqN682U62eAISe0ODt5sYHt6At+FoAXgwMM
# R2i+H2pNcMTbZvIvUTM5nLJ6TUldfNqPxamHoROqz4fIN9+rrnbcb2Zz1c7q+iOJ
# frRPbyhHuLDJTZ2JE451uMyy6l83fn5Ex7C2+rUHgXA14IFGvcvA878tcws5rnHh
# p3m9pyg5mdrStjuuIPaQCtmqtICACiQQxxCgfdZUW60nJYgUFe9bYw/8qSFD29va
# fKPKuU6o4IMEJwGzEMvYUSdIWQIB5RF4yLO1XARLwjiLAXyOvAPqZU7fsqV+lS8r
# /IrJ4A+dB2V0byPm1neZczaBpfzvHsSZ3MAAINGBjJaJIbVnvI93gX4d9RFNX9cH
# YPsJyG0z4T+IDfQ8EB4TSt9D2iIUo9x4AkYk9480PqvLdttwvx5GjQyGgEhciWhl
# vATb3OUKoTCE1FXMlVCDkkc3dHeJrNPghLVCnS6hz8NJ10UIDUxnldWZ5adE5X9u
# yZ6rnyM5EFPyNj4N8ukVwGO3VUF9z1VdkGXeLeJWvC56Hken1qQAO9Bq1fJV4rKV
# CzNffto=
# SIG # End signature block
