/* SPDX-License-Identifier: GPL-2.0-only */
/* Copyright (C) 2013-2024 Intel Corporation */

#ifndef CONFIG_TRACEPOINTS
#if !defined(_I40E_TRACE_H_)
#define _I40E_TRACE_H_
/* If the Linux kernel tracepoints are not available then the i40e_trace*
 * macros become nops.
 */

#define i40e_trace(trace_name, args...)
#define i40e_trace_enabled(trace_name) (0)
#endif /* !defined(_I40E_TRACE_H_) */
#else /* CONFIG_TRACEPOINTS */
/*
 * Modeled on trace-events-sample.h
 */

/*
 * This file is named i40e_trace.h.
 *
 * Since this include file's name is different from the trace
 * subsystem name, we'll have to define TRACE_INCLUDE_FILE at the end
 * of this file.
 */
#undef TRACE_SYSTEM
#define TRACE_SYSTEM i40e

/*
 * See trace-events-sample.h for a detailed description of why this
 * guard clause is different from most normal include files.
 */
#if !defined(_I40E_TRACE_H_) || defined(TRACE_HEADER_MULTI_READ)
#define _I40E_TRACE_H_

#include <linux/tracepoint.h>

/*
 * i40e_trace() macro enables shared code to refer to trace points
 * like:
 *
 * trace_i40e{,vf}_example(args...)
 *
 * ... as:
 *
 * i40e_trace(example, args...)
 *
 * ... to resolve to the PF or VF version of the tracepoint without
 * ifdefs, and to allow tracepoints to be disabled entirely at build
 * time.
 *
 * Trace point should always be referred to in the driver via this
 * macro.
 *
 * Similarly, i40e_trace_enabled(trace_name) wraps references to
 * trace_i40e{,vf}_<trace_name>_enabled() functions.
 */
#define _I40E_TRACE_NAME(trace_name) (trace_ ## i40e ## _ ## trace_name)
#define I40E_TRACE_NAME(trace_name) _I40E_TRACE_NAME(trace_name)

#define i40e_trace(trace_name, args...) I40E_TRACE_NAME(trace_name)(args)

#define i40e_trace_enabled(trace_name) I40E_TRACE_NAME(trace_name##_enabled)()

/*
 * Events common to PF and VF. Corresponding versions will be defined
 * for both, named trace_i40e_* and trace_i40evf_*. The i40e_trace()
 * macro above will select the right trace point name for the driver
 * being built from shared code.
 */

/* Events related to a vsi & ring */
DECLARE_EVENT_CLASS(
	i40e_tx_template,

	TP_PROTO(struct i40e_ring *ring,
		 struct i40e_tx_desc *desc,
		 struct i40e_tx_buffer *buf),

	TP_ARGS(ring, desc, buf),

	/*
	 * The convention here is to make the first fields in the
	 * TP_STRUCT match the TP_PROTO exactly. This enables the use
	 * of the args struct generated by the tplist tool (from the
	 * bcc-tools package) to be used for those fields. To access
	 * fields other than the tracepoint args will require the
	 * tplist output to be adjusted.
	 */
	TP_STRUCT__entry(
		__field(void*, ring)
		__field(void*, desc)
		__field(void*, buf)
		__string(devname, ring->netdev->name)
	),

	TP_fast_assign(
		__entry->ring = ring;
		__entry->desc = desc;
		__entry->buf = buf;
		_kc__assign_str(devname, ring->netdev->name);
	),

	TP_printk(
		"netdev: %s ring: %p desc: %p buf %p",
		__get_str(devname), __entry->ring,
		__entry->desc, __entry->buf)
);

DEFINE_EVENT(
	i40e_tx_template, i40e_clean_tx_irq,
	TP_PROTO(struct i40e_ring *ring,
		 struct i40e_tx_desc *desc,
		 struct i40e_tx_buffer *buf),

	TP_ARGS(ring, desc, buf));

DEFINE_EVENT(
	i40e_tx_template, i40e_clean_tx_irq_unmap,
	TP_PROTO(struct i40e_ring *ring,
		 struct i40e_tx_desc *desc,
		 struct i40e_tx_buffer *buf),

	TP_ARGS(ring, desc, buf));

DECLARE_EVENT_CLASS(
	i40e_rx_template,

	TP_PROTO(struct i40e_ring *ring,
		 union i40e_rx_desc *desc,
		 struct sk_buff *skb),

	TP_ARGS(ring, desc, skb),

	TP_STRUCT__entry(
		__field(void*, ring)
		__field(void*, desc)
		__field(void*, skb)
		__string(devname, ring->netdev->name)
	),

	TP_fast_assign(
		__entry->ring = ring;
		__entry->desc = desc;
		__entry->skb = skb;
		_kc__assign_str(devname, ring->netdev->name);
	),

	TP_printk(
		"netdev: %s ring: %p desc: %p skb %p",
		__get_str(devname), __entry->ring,
		__entry->desc, __entry->skb)
);

DEFINE_EVENT(
	i40e_rx_template, i40e_clean_rx_irq,
	TP_PROTO(struct i40e_ring *ring,
		 union i40e_rx_desc *desc,
		 struct sk_buff *skb),

	TP_ARGS(ring, desc, skb));

DEFINE_EVENT(
	i40e_rx_template, i40e_clean_rx_irq_rx,
	TP_PROTO(struct i40e_ring *ring,
		 union i40e_rx_desc *desc,
		 struct sk_buff *skb),

	TP_ARGS(ring, desc, skb));

DECLARE_EVENT_CLASS(
	i40e_xmit_template,

	TP_PROTO(struct sk_buff *skb,
		 struct i40e_ring *ring),

	TP_ARGS(skb, ring),

	TP_STRUCT__entry(
		__field(void*, skb)
		__field(void*, ring)
		__string(devname, ring->netdev->name)
	),

	TP_fast_assign(
		__entry->skb = skb;
		__entry->ring = ring;
		_kc__assign_str(devname, ring->netdev->name);
	),

	TP_printk(
		"netdev: %s skb: %p ring: %p",
		__get_str(devname), __entry->skb,
		__entry->ring)
);

DEFINE_EVENT(
	i40e_xmit_template, i40e_xmit_frame_ring,
	TP_PROTO(struct sk_buff *skb,
		 struct i40e_ring *ring),

	TP_ARGS(skb, ring));

DEFINE_EVENT(
	i40e_xmit_template, i40e_xmit_frame_ring_drop,
	TP_PROTO(struct sk_buff *skb,
		 struct i40e_ring *ring),

	TP_ARGS(skb, ring));

/*
 * Events unique to the PF.
 */
DECLARE_EVENT_CLASS(
	i40e_state_template,

	TP_PROTO(struct i40e_pf *pf, u64 val),

	TP_ARGS(pf, val),

	TP_STRUCT__entry(
		__field(u64, val)
		__field(u64, state)
		__field(u64, bus)
	),

	TP_fast_assign(
		__entry->val = val;
		__entry->state = *(u64 *)pf->state;
		__entry->bus = (((u64)pf->hw.bus.bus_id) << 32) |
			(((u64)pf->hw.bus.device) << 16) | pf->hw.bus.func;
	),

	TP_printk(
		"state: bus %02x:%02x.%1x state=%016llx val=%llx",
		(unsigned int)(__entry->bus >> 32),
		0xffff & (unsigned int)(__entry->bus >> 16),
		0xffff & (unsigned int)__entry->bus,
		 __entry->state, __entry->val)
);

DEFINE_EVENT(
	i40e_state_template, i40e_state_reset,
	TP_PROTO(struct i40e_pf *pf, u64 val),

	TP_ARGS(pf, val));

DEFINE_EVENT(
	i40e_state_template, i40e_state_reset_pci_prepare,
	TP_PROTO(struct i40e_pf *pf, u64 val),

	TP_ARGS(pf, val));

DEFINE_EVENT(
	i40e_state_template, i40e_state_reset_pci_done,
	TP_PROTO(struct i40e_pf *pf, u64 val),

	TP_ARGS(pf, val));

DEFINE_EVENT(
	i40e_state_template, i40e_state_reset_corer,
	TP_PROTO(struct i40e_pf *pf, u64 val),

	TP_ARGS(pf, val));

DEFINE_EVENT(
	i40e_state_template, i40e_state_reset_globr,
	TP_PROTO(struct i40e_pf *pf, u64 val),

	TP_ARGS(pf, val));

DEFINE_EVENT(
	i40e_state_template, i40e_state_reset_empr,
	TP_PROTO(struct i40e_pf *pf, u64 val),

	TP_ARGS(pf, val));

DEFINE_EVENT(
	i40e_state_template, i40e_state_hmc_error,
	TP_PROTO(struct i40e_pf *pf, u64 val),

	TP_ARGS(pf, val));

DEFINE_EVENT(
	i40e_state_template, i40e_state_rebuild,
	TP_PROTO(struct i40e_pf *pf, u64 val),

	TP_ARGS(pf, val));

DEFINE_EVENT(
	i40e_state_template, i40e_state_arq,
	TP_PROTO(struct i40e_pf *pf, u64 val),

	TP_ARGS(pf, val));

DEFINE_EVENT(
	i40e_state_template, i40e_state_asq,
	TP_PROTO(struct i40e_pf *pf, u64 val),

	TP_ARGS(pf, val));

DEFINE_EVENT(
	i40e_state_template, i40e_state_udp_sync,
	TP_PROTO(struct i40e_pf *pf, u64 val),

	TP_ARGS(pf, val));

DEFINE_EVENT(
	i40e_state_template, i40e_state_watchdog,
	TP_PROTO(struct i40e_pf *pf, u64 val),

	TP_ARGS(pf, val));

DEFINE_EVENT(
	i40e_state_template, i40e_state_link,
	TP_PROTO(struct i40e_pf *pf, u64 val),

	TP_ARGS(pf, val));

DEFINE_EVENT(
	i40e_state_template, i40e_state_recovery,
	TP_PROTO(struct i40e_pf *pf, u64 val),

	TP_ARGS(pf, val));

DECLARE_EVENT_CLASS(
	i40e_ioctl_template,

	TP_PROTO(struct i40e_pf *pf, u64 val),

	TP_ARGS(pf, val),

	TP_STRUCT__entry(
		__field(u64, val)
		__field(u64, state)
		__field(u64, bus)
		__field(pid_t, pid)
		__array(char, comm, TASK_COMM_LEN)
	),

	TP_fast_assign(
		__entry->val = val;
		__entry->state = *(u64 *)pf->state;
		__entry->bus = (((u64)pf->hw.bus.bus_id) << 32) |
			(((u64)pf->hw.bus.device) << 16) | pf->hw.bus.func;
		__entry->pid = current->pid;
		memcpy(__entry->comm, current->comm, sizeof(__entry->comm) - 1);
		__entry->comm[TASK_COMM_LEN - 1] = 0;
	),

	TP_printk(
		"state: bus %02x:%02x.%1x state=%016llx val=%llx %5d:%s",
		(unsigned int)(__entry->bus >> 32),
		0xffff & (unsigned int)(__entry->bus >> 16),
		0xffff & (unsigned int)__entry->bus,
		__entry->state, __entry->val,
		__entry->pid, __entry->comm)
);

DEFINE_EVENT(
	i40e_ioctl_template, i40e_ioctl_get_drvinfo,
	TP_PROTO(struct i40e_pf *pf, u64 val),

	TP_ARGS(pf, val));

DEFINE_EVENT(
	i40e_ioctl_template, i40e_ioctl_get_eeprom_len,
	TP_PROTO(struct i40e_pf *pf, u64 val),

	TP_ARGS(pf, val));

DEFINE_EVENT(
	i40e_ioctl_template, i40e_ioctl_get_eeprom,
	TP_PROTO(struct i40e_pf *pf, u64 val),

	TP_ARGS(pf, val));

DEFINE_EVENT(
	i40e_ioctl_template, i40e_ioctl_set_eeprom,
	TP_PROTO(struct i40e_pf *pf, u64 val),

	TP_ARGS(pf, val));

DEFINE_EVENT(
	i40e_ioctl_template, i40e_ioctl_get_module_info,
	TP_PROTO(struct i40e_pf *pf, u64 val),

	TP_ARGS(pf, val));

DEFINE_EVENT(
	i40e_ioctl_template, i40e_ioctl_get_module_eeprom,
	TP_PROTO(struct i40e_pf *pf, u64 val),

	TP_ARGS(pf, val));

DEFINE_EVENT(
	i40e_ioctl_template, i40e_ioctl_get_link_ksettings,
	TP_PROTO(struct i40e_pf *pf, u64 val),

	TP_ARGS(pf, val));

DECLARE_EVENT_CLASS(
	i40e_nvmupd_template,

	TP_PROTO(struct i40e_hw *hw,
		 struct i40e_nvm_access *cmd, int ret_val, int err),

	TP_ARGS(hw, cmd, ret_val, err),

	TP_STRUCT__entry(
		__field(int, ret_val)
		__field(int, err)
		__field(int, status)
		__field(u64, bus)
		__field(u32, command)
		__field(u32, config)
		__field(u32, offset)
		__field(u32, data_size)
	),

	TP_fast_assign(
		__entry->ret_val = ret_val;
		__entry->err = err;
		__entry->status = hw->aq.asq_last_status;
		__entry->bus = (((u64)hw->bus.bus_id) << 32) |
			(((u64)hw->bus.device) << 16) | hw->bus.func;
		__entry->command = cmd->command;
		__entry->config = cmd->config;
		__entry->offset = cmd->offset;
		__entry->data_size = cmd->data_size;
	),

	TP_printk(
		"nvmupd: bus %02x:%02x.%1x err=%d status=0x%x errno=%d module=%d offset=0x%x size=%d",
		(unsigned int)(__entry->bus >> 32),
		0xffff & (unsigned int)(__entry->bus >> 16),
		0xffff & (unsigned int)__entry->bus,
		__entry->ret_val, __entry->status, __entry->err,
		(__entry->config & I40E_NVM_MOD_PNT_MASK),
		__entry->offset, __entry->data_size)
);

DEFINE_EVENT(
	i40e_nvmupd_template, i40e_nvmupd_write,
	TP_PROTO(struct i40e_hw *hw,
		 struct i40e_nvm_access *cmd, int ret_val, int err),

	TP_ARGS(hw, cmd, ret_val, err));

DEFINE_EVENT(
	i40e_nvmupd_template, i40e_nvmupd_read,
	TP_PROTO(struct i40e_hw *hw,
		 struct i40e_nvm_access *cmd, int ret_val, int err),

	TP_ARGS(hw, cmd, ret_val, err));

#endif /* _I40E_TRACE_H_ */
/* This must be outside ifdef _I40E_TRACE_H */

/* This trace include file is not located in the .../include/trace
 * with the kernel tracepoint definitions, because we're a loadable
 * module.
 */
#undef TRACE_INCLUDE_PATH
#define TRACE_INCLUDE_PATH .
#undef TRACE_INCLUDE_FILE
#define TRACE_INCLUDE_FILE i40e_trace
#include <trace/define_trace.h>
#endif /* CONFIG_TRACEPOINTS */
